<?php

namespace Widget\StockBundle\Controller\BackendAPI;

use Backend\BaseBundle\Controller\BackendAPI\BaseBackendAPIController;
use Sensio\Bundle\FrameworkExtraBundle\Configuration\Method;
use Sensio\Bundle\FrameworkExtraBundle\Configuration\Route;
use Sensio\Bundle\FrameworkExtraBundle\Configuration\Security;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\Validator\Constraints\All;
use Symfony\Component\Validator\Constraints\Collection;
use Symfony\Component\Validator\Constraints\GreaterThan;
use Symfony\Component\Validator\Constraints\NotBlank;
use Symfony\Component\Validator\Constraints\Regex;
use Symfony\Component\Validator\ConstraintViolationListInterface;
use Symfony\Component\Validator\Validator\ValidatorInterface;
use Widget\StockBundle\Model\StockStylePeer;
use Widget\StockBundle\Model\StockStyleQuery;
use Widget\StockBundle\Service\InsertStockLogService;

/**
 * @Route("/stockstyle")
 * @Security("has_role_or_superadmin('ROLE_STOCK')")
 */
class StockStyleBatchQuantityController extends BaseBackendAPIController
{
    protected function getFormConfig()
    {
        
    }
    
    /**
     * 取得搜尋條件下所有庫存(無分頁)
     * @Route("s/batchvalue")
     * @Method({"GET"})
     * @Security("has_role_or_superadmin('ROLE_STOCK_READ')")
     */
    public function getBatchQuantityAction(Request $request)
    {
        $parameters = array_merge(array(
            'search' => array(),
            'sort' => array(),
        ), $request->query->all());

        $query = StockStyleQuery::create()
            ->distinct();

        $query = $this->applySearchQuery($query, StockStylePeer::class, $parameters);
        return $this->createJsonSerializeResponse($query->find(), array('list'));
    }

    /**
     * 更新送過來的庫存
     * @Route("s/batchvalue")
     * @Method({"PUT"})
     * @Security("has_role_or_superadmin('ROLE_STOCK_WRITE')")
     */
    public function updateBatchQuantityAction(Request $request)
    {
        $parameters = json_decode($request->getContent(), true);

        /** @var ValidatorInterface $validator */
        $errors = $this->validateParameters($parameters);
        if (count($errors) != 0) {
            return $this->createJsonResponse($this->createBatchQuantityErrorArray($errors), Response::HTTP_BAD_REQUEST);
        }
        
        /** @var InsertStockLogService $insertStockLogService */
        $insertStockLogService = $this->get('widget.stock_bundle.service.insert_stock_service');

        foreach ($parameters as $parameter) {
            $stock = StockStyleQuery::create()->findPk($parameter['id']);
            if ($stock != null && $stock->getQuantity() != $parameter['quantity']) {
                $oldQuantity = $stock->getQuantity();
                $stock->setQuantity($parameter['quantity'])->save();
                $insertStockLogService->insertStyleLog($stock->getId(), $oldQuantity, $this->getUser());
            }
        }
        
        return $this->createJsonResponse();
    }
    

    protected function validateParameters($parameters)
    {
        $validator = $this->get('validator');
        $constraints = array(
            new All(array(
                    new NotBlank(),
                    new Collection(array(
                        'fields' => array(
                            'id' => array(
                                new NotBlank(),
                            ),
                            'quantity' => array(
                                new NotBlank(),
                                new GreaterThan(array(
                                    'value'   => -1,
                                    'message' => 'error.stock.greater_than_negative',
                                )),
                                new Regex(array(
                                    'pattern' => '/^\d+$/',
                                    'message' => 'error.stock.integer',
                                )),
                            ),
                        ),
                    )),
                )
            ),
        );
        $validator->validate($parameters, $constraints);
        
        return $validator->validate($parameters, $constraints);        
    }

    protected function createBatchQuantityErrorArray(ConstraintViolationListInterface $errors)
    {
        $errorMessage = array();
        foreach ($errors as $error){
            $index = $this->numericPropertyPath($error->getPropertyPath());
            $errorMessage[$index] = array(
                'index' => $index,
                'error' => $error->getMessage()
            );
        }
        return array_values($errorMessage);
    }

    /**
     * 只需要數字 其他拿掉
     * @param $propertyPathString
     * @return mixed
     */
    protected function numericPropertyPath($propertyPathString)
    {
        return preg_replace("/[^0-9,.]/", "", $propertyPathString);
    }
}