<?php

namespace Widget\StockBundle\Controller\BackendAPI;

use Backend\BaseBundle\Controller\BackendAPI\BaseBackendAPIController;
use Backend\BaseBundle\Form\Type\APIFormTypeItem;
use JMS\DiExtraBundle\Annotation as DI;
use Sensio\Bundle\FrameworkExtraBundle\Configuration\Method;
use Sensio\Bundle\FrameworkExtraBundle\Configuration\Route;
use Sensio\Bundle\FrameworkExtraBundle\Configuration\Security;
use Symfony\Component\Form\Extension\Core\Type\IntegerType;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\Validator\Constraints\GreaterThan;
use Symfony\Component\Validator\Constraints\NotBlank;
use Widget\ProductBundle\Model\ProductPeer;
use Widget\ProductStyleBundle\Model\ProductStyleMixPeer;
use Widget\StockBundle\Model\Stock;
use Widget\StockBundle\Model\StockPeer;
use Widget\StockBundle\Model\StockQuery;
use Widget\StockBundle\Model\StockStyle;
use Widget\StockBundle\Model\StockStylePeer;
use Widget\StockBundle\Model\StockStyleQuery;
use Widget\StockBundle\Service\InsertStockLogService;

/**
 * @Route("/stockstyle")
 * @Security("has_role_or_superadmin('ROLE_STOCK')")
 */
class StockStyleController extends BaseBackendAPIController
{

    /**
     * @return APIFormTypeItem[]
     */
    protected function getFormConfig()
    {
        return array(
            (new APIFormTypeItem('quantity'))->setOptions(array(
                'label' => 'index.stock.quantity',
                'constraints' => array(
                    new NotBlank(array(
                        'message' => 'error.missing_field'
                    )),
                    new GreaterThan(array(
                        'value' => -1,
                        'message' => 'error.stock.greater_than_negative'
                    )),
                ),
            ))->setFieldType(IntegerType::class),
        );
    }

    /**
     * 搜尋
     * @Route("s")
     * @Method({"GET"})
     * @Security("has_role_or_superadmin('ROLE_STOCK_READ')")
     */
    public function searchAction(Request $request)
    {
        return $this->doSearch($request->query->all(), StockStyleQuery::create()->distinct(), StockStylePeer::class);
    }

    /**
     * @Route("/{id}")
     * @Security("has_role_or_superadmin('ROLE_STOCK_READ')")
     * @Method({"GET"})
     */
    public function readAction(Request $request, StockStyle $stockStyle)
    {
        return $this->createJsonSerializeResponse($stockStyle, array('detail'));
    }

    /**
     * @Route("/{id}")
     * @Security("has_role_or_superadmin('ROLE_STOCK_WRITE')")
     * @Method({"PUT"})
     */
    public function updateStockAction(Request $request, StockStyle $stockStyle)
    {
        $oldQuantity = $stockStyle->getQuantity();
        $response = $this->doProcessForm($stockStyle, $request->getContent());
        if ($response->getStatusCode() == 200){
            $insertStockLogService = $this->get('widget.stock_bundle.service.insert_stock_service');
            $insertStockLogService->insertStyleLog($stockStyle->getId(), $oldQuantity, $this->getUser());
        }
        return $response;
    }

    /**
     * 刪除
     * @Route("/{id}")
     * @Method({"DELETE"})
     * @Security("has_role_or_superadmin('ROLE_STOCK_WRITE')")
     */
    public function deleteAction(Request $request, StockStyle $stockStyle)
    {
        $stockStyle->delete();
        return $this->createJsonResponse();
    }
}