<?php

namespace Widget\StockBundle\Controller\BackendAPI;

use Backend\BaseBundle\Controller\BackendAPI\BaseBackendAPIController;
use Backend\BaseBundle\Form\Type\APIFormTypeItem;
use JMS\DiExtraBundle\Annotation as DI;
use Sensio\Bundle\FrameworkExtraBundle\Configuration\Method;
use Sensio\Bundle\FrameworkExtraBundle\Configuration\Route;
use Sensio\Bundle\FrameworkExtraBundle\Configuration\Security;
use Symfony\Component\Form\Extension\Core\Type\IntegerType;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\Validator\Constraints\GreaterThan;
use Symfony\Component\Validator\Constraints\NotBlank;
use Widget\StockBundle\Model\Stock;
use Widget\StockBundle\Model\StockPeer;
use Widget\StockBundle\Model\StockQuery;
use Widget\StockBundle\Service\InsertStockLogService;

/**
 * @Route("/stock")
 * @Security("has_role_or_superadmin('ROLE_STOCK')")
 */
class StockController extends BaseBackendAPIController
{
    /**
     * @return APIFormTypeItem[]
     */
    protected function getFormConfig()
    {
        return array(
            new APIFormTypeItem('id'),
            (new APIFormTypeItem('quantity'))->setOptions(array(
                'label' => 'index.stock.quantity',
                'constraints' => array(
                    new NotBlank(array(
                        'message' => 'error.missing_field'
                    )),
                    new GreaterThan(array(
                        'value' => -1,
                        'message' => 'error.stock.greater_than_negative'
                    )),
                ),                
            ))->setFieldType(IntegerType::class),
        );
    }


    /**
     * @Route("s")
     * @Method({"POST"})
     */
    public function createStockAction(Request $request)
    {
        return $this->doProcessForm(new Stock(), $request->getContent());
    }

    /**
     * 搜尋
     * @Route("s")
     * @Method({"GET"})
     * @Security("has_role_or_superadmin('ROLE_STOCK_READ')")
     */
    public function searchAction(Request $request)
    {
        return $this->doSearch($request->query->all(), StockQuery::create()->distinct(), StockPeer::class);
    }

    /**
     * 讀取
     * @Route("/{id}")
     * @Method({"GET"})
     * @Security("has_role_or_superadmin('ROLE_STOCK_READ')")
     */
    public function readAction(Stock $stock)
    {
        return $this->createJsonSerializeResponse($stock, array('detail'));
    }

    /**
     * @Route("/{id}")
     * @Security("has_role_or_superadmin('ROLE_STOCK_WRITE')")
     * @Method({"PUT"})
     */    
    public function updateStockAction(Request $request, $id)
    {
        $stock = StockQuery::create()->filterById($id)->findOneOrCreate();
        $oldQuantity = $stock->getQuantity();
        $response = $this->doProcessForm($stock, $request->getContent());
        if ($response->getStatusCode() == 200){
            $insertStockLogService = $this->get('widget.stock_bundle.service.insert_stock_service');
            $insertStockLogService->insertLog($stock->getId(), $oldQuantity, $this->getUser());
        }
        return $response;
    }

    /**
     * 搜尋
     * @Route("/{id}")
     * @Method({"DELETE"})
     * @Security("has_role_or_superadmin('ROLE_STOCK_WRITE')")
     */
    public function deleteAction(Request $request, Stock $stock)
    {
        $stock->delete();
        return $this->createJsonResponse();
    }    
}