<?php
namespace Widget\ProductBundle\Tests\API;


use Backend\BaseBundle\Propel\I18n;
use Backend\BaseBundle\Tests\Fixture\BaseWebTestCase;
use JMS\Serializer\SerializationContext;
use Symfony\Component\HttpFoundation\Response;
use Widget\CategoryBundle\Model\Category;
use Widget\CategoryBundle\Model\CategoryQuery;
use Widget\ProductBundle\Controller\API\ProductController;
use Widget\ProductBundle\Model\CategoryProductQuery;
use Widget\ProductBundle\Model\Product;
use Widget\ProductBundle\Model\ProductI18nQuery;
use Widget\ProductBundle\Model\ProductQuery;

class ProductControllerTest extends BaseWebTestCase
{
    public function test_getProductAction_invaild_id()
    {
        //arrange
        $id = '5564614313';

        //act
        $this->client->request('GET', $this->generateUrl('widget_product_api_product_getproduct', array('id' => $id)));
        $response = $this->client->getResponse();

        //assert
        $this->assertTrue($response->isNotFound());
    }

    public function test_getProductAction_invaild_status()
    {
        //arrange
        /** @var Product $product */
        $product = ProductQuery::create()
            ->findOne();
        $product->setStatusShelf(false);
        $product->save();
        $id = $product->getId();

        //act
        $this->client->request('GET', $this->generateUrl('widget_product_api_product_getproduct', array('id' => $id)));
        $response = $this->client->getResponse();

        //assert
        $this->assertTrue($response->isNotFound());
        $product->reload(true);
        $product->setStatusShelf(true);
        $product->save();
    }

    public function test_getProductAction_vaild()
    {
        //arrange
        /** @var Product $product */
        $product = ProductQuery::create()
            ->filterByStatusShelf(true)
            ->findOne();
        $id = $product->getId();

        //act
        $this->client->request('GET', $this->generateUrl('widget_product_api_product_getproduct', array('id' => $id)));
        $response = $this->client->getResponse();

        //assert
        $this->assertTrue($response->isOk());
    }

    /**
     * 檢查格式是否正確
     */
    public function test_getProductAction_vaild_data()
    {
        //arrange
        /** @var Product $product */
        $product = ProductQuery::create()
            ->filterByStatusShelf(true)
            ->findOne();
        $id = $product->getId();

        $context = SerializationContext::create()->setSerializeNull(true)->setGroups(array('detail'));
        $expectedData = json_decode(
            $this->client->getContainer()->get('jms_serializer')->serialize($product,
            'json',
            $context), true);

        //act
        $this->client->request('GET', $this->generateUrl('widget_product_api_product_getproduct', array('id' => $id)));
        $response = $this->client->getResponse();
        $result = json_decode($response->getContent(), true);

        //assert
        $this->assertTrue($response->isOk());
        $this->assertEquals($expectedData, $result);
    }

    /**
     * 檢查格式是否正確
     */
    public function test_getProductAction_vaild_price()
    {
        //arrange
        /** @var Product $product */
        $product = ProductQuery::create()
            ->filterByStatusShelf(true)
            ->findOne();
        $id = $product->getId();

        //act
        $this->client->request('GET', $this->generateUrl('widget_product_api_product_getproduct', array('id' => $id)));
        $response = $this->client->getResponse();

        //assert
        $this->assertTrue($response->isOk());
    }

    public function test_getCategoryAction_vaild()
    {
        //arrange
        /** @var Product $product */
        $product = ProductQuery::create()
            ->filterByStatusShelf(true)
            ->findOne();
        $id = $product->getId();
        $categories = $product->getCategories(
            CategoryQuery::create()
                ->useCategoryThreadQuery()
                    ->filterByThread('product')
                ->endUse()
        );

        //act
        $this->client->request('GET', $this->generateUrl('widget_product_api_product_getcategory', array('id' => $id)));
        $response = $this->client->getResponse();

        //assert
        $this->assertTrue($response->isOk());
    }
    
    public function test_searchAction_invaild_keyword()
    {
        //arrange
        //act
        $this->client->request('POST', $this->generateUrl('widget_product_api_product_search'));
        $response = $this->client->getResponse();

        //assert
        $this->assertEquals(Response::HTTP_OK, $response->getStatusCode());
    }

    public function test_searchAction_vaild_keyword()
    {
        //arrange
        $keyword = 'Est rem accusantium.';
        $sort = 'created_at';
        $orderby = 'DESC';
        $page = '1';
        $rowsperpage = '100';

        $query = ProductQuery::create();
        if ($query instanceof I18n){
            $product = ProductI18nQuery::create()
                ->filterByName("%{$keyword}%", \Criteria::LIKE)
                ->paginate($page, $rowsperpage);
        }else {
            /** @var Product $product */
            $product = $query
                ->filterByName("%{$keyword}%", \Criteria::LIKE)
                ->paginate($page, $rowsperpage);
        }

        $context = SerializationContext::create()->setGroups(array('list'));
        $expectedData = json_decode(
            $this->client->getContainer()->get('jms_serializer')->serialize($product,
                'json',
                $context), true);
        $keywordData = json_encode(array(
            'keyword' => $keyword,
            'sort' => $sort,
            'orderby' => $orderby,
            'page' => $page,
            'rowsperpage' => $rowsperpage,
        ));
        //act

        $this->client->request(
            'POST',
            $this->generateUrl('widget_product_api_product_search'),
            array(),
            array(),
            array(),
            $keywordData
        );
        $response = $this->client->getResponse();
        $result = json_decode($response->getContent(), true);

        //assert
        $this->assertTrue($response->isOk());
        $this->assertEquals($expectedData, $result);
    }

    public function test_useAdvancedQuery_with_no_category_id()
    {
        //arrange
        $parameters = array();
        $productQuery = $this->getMockBuilder(ProductQuery::class)
            ->setMethods(array('filterByCategory'))
            ->getMock();
        $productQuery
            ->expects($this->never())
            ->method('filterByCategory')
            ->willReturn(null);
        $productController = new ProductController();

        //act
        $this->callObjectMethod($productController, 'useAdvancedQuery', $productQuery, $parameters);

        //assert
    }

    public function test_useAdvancedQuery_with_category_id()
    {
        //arrange
        $categoryId = 12345;
        $parameters = array(
            'category_id' => $categoryId,
        );
        $productQuery = $this->getMockBuilder(ProductQuery::class)
            ->setMethods(array('useCategoryProductQuery', 'endUse'))
            ->getMock();
        $categoryProductQuery = $this->getMockBuilder(CategoryProductQuery::class)
            ->setMethods(array('filterByCategoryId'))
            ->getMock();
        $productQuery
            ->expects($this->atLeastOnce())
            ->method('useCategoryProductQuery')
            ->willReturn($categoryProductQuery);
        $productQuery
            ->expects($this->once())
            ->method('endUse');
        $categoryProductQuery
            ->expects($this->atLeastOnce())
            ->method('filterByCategoryId')
            ->willReturnCallback(function($categoryIdForTest) use($categoryId, $productQuery){
                $this->assertEquals($categoryId, $categoryIdForTest);
                return $productQuery;
            });
        $productController = new ProductController();

        //act
        $this->callObjectMethod($productController, 'useAdvancedQuery', $productQuery, $parameters);

        //assert
    }

    public function test_useAdvancedQuery_with_no_pricerange_start()
    {
        //arrange
        $parameters = array();
        $productQuery = $this->getMockBuilder(ProductQuery::class)
            ->setMethods(array('filterByPrice'))
            ->getMock();
        $productQuery
            ->expects($this->never())
            ->method('filterByPrice')
            ->willReturn(null);
        $productController = new ProductController();

        //act
        $this->callObjectMethod($productController, 'useAdvancedQuery', $productQuery, $parameters);

        //assert
    }

    public function test_useAdvancedQuery_with_pricerange_start()
    {
        //arrange
        $price = 9;
        $parameters = array(
            'pricerange_start' => $price,
        );
        $productQuery = $this->getMockBuilder(ProductQuery::class)
            ->setMethods(array('filterByPrice'))
            ->getMock();
        $productQuery
            ->expects($this->atLeastOnce())
            ->method('filterByPrice')
            ->willReturnCallback(function($priceForTest) use($price, $productQuery){
                $this->assertEquals($price, $priceForTest['min']);
                return $productQuery;
            });
        $productController = new ProductController();

        //act
        $this->callObjectMethod($productController, 'useAdvancedQuery', $productQuery, $parameters);

        //assert
    }

    public function test_useAdvancedQuery_with_no_pricerange_end()
    {
        //arrange
        $parameters = array();
        $productQuery = $this->getMockBuilder(ProductQuery::class)
            ->setMethods(array('filterByPrice'))
            ->getMock();
        $productQuery
            ->expects($this->never())
            ->method('filterByPrice')
            ->willReturn(null);
        $productController = new ProductController();

        //act
        $this->callObjectMethod($productController, 'useAdvancedQuery', $productQuery, $parameters);

        //assert
    }

    public function test_useAdvancedQuery_with_pricerange_end()
    {
        //arrange
        $price = 10;
        $parameters = array(
            'pricerange_end' => $price,
        );
        $productQuery = $this->getMockBuilder(ProductQuery::class)
            ->setMethods(array('filterByPrice'))
            ->getMock();
        $productQuery
            ->expects($this->atLeastOnce())
            ->method('filterByPrice')
            ->willReturnCallback(function($priceForTest) use($price, $productQuery){
                $this->assertEquals($price, $priceForTest['max']);
                return $productQuery;
            });
        $productController = new ProductController();

        //act
        $this->callObjectMethod($productController, 'useAdvancedQuery', $productQuery, $parameters);

        //assert
    }

    public function test_useAdvancedQuery_with_no_search_brief()
    {
        //arrange
        $parameters = array();
        $productQuery = $this->getMockBuilder(ProductQuery::class)
            ->setMethods(array('filterByBrief'))
            ->getMock();
        $productQuery
            ->expects($this->never())
            ->method('filterByBrief')
            ->willReturn(null);
        $productController = new ProductController();

        //act
        $this->callObjectMethod($productController, 'useAdvancedQuery', $productQuery, $parameters);

        //assert
    }

    public function test_useAdvancedQuery_with_search_brief_false()
    {
        //arrange
        $parameters = array(
            'search_brief' => false,
        );
        $productQuery = $this->getMockBuilder(ProductQuery::class)
            ->setMethods(array('filterByBrief'))
            ->getMock();
        $productQuery
            ->expects($this->never())
            ->method('filterByBrief')
            ->willReturn(null);
        $productController = new ProductController();

        //act
        $this->callObjectMethod($productController, 'useAdvancedQuery', $productQuery, $parameters);

        //assert
    }

    public function test_useAdvancedQuery_with_search_brief_true()
    {
        //arrange
        $keywords = 'test';
        $parameters = array(
            'search_brief' => true,
        );
        $productQuery = $this->getMockBuilder(ProductQuery::class)
            ->setMethods(array('filterByBrief', '_or'))
            ->getMock();
        $productQuery
            ->expects($this->atLeastOnce())
            ->method('_or')
            ->willReturn($productQuery);
        $productQuery
            ->expects($this->atLeastOnce())
            ->method('filterByBrief')
            ->willReturnCallback(function($value, $contition) use($keywords, $productQuery){
                $this->assertEquals("%{$keywords}%", $value);
                $this->assertEquals(\Criteria::LIKE, $contition);
                return $productQuery;
            });
        $productController = new ProductController();

        //act
        $this->callObjectMethod($productController, 'useAdvancedQuery', $productQuery, $parameters, $keywords);

        //assert
    }


    public function test_useAdvancedQuery_with_no_sub_category()
    {

        $this->markTestIncomplete("分支測試還沒寫");
        return;
        //arrange
        $parameters = array(
            'price' => 0,
        );
        $productQuery = $this->getMockBuilder(ProductQuery::class)
            ->setMethods(array('filterByPrice'))
            ->getMock();
        $productQuery
            ->expects($this->never())
            ->method('filterByPrice')
            ->willReturn(null);
        $productController = new ProductController();

        //act
        $this->callObjectMethod($productController, 'useAdvancedQuery', $productQuery, $parameters);

        //assert
    }

}
