<?php
namespace Widget\ProductBundle\Tests\API;

use Backend\BaseBundle\Tests\Fixture\BaseWebTestCase;
use JMS\Serializer\SerializationContext;
use Widget\CategoryBundle\Model\Category;
use Widget\CategoryBundle\Model\CategoryQuery;
use Widget\ProductBundle\Model\ProductQuery;

class ProductCategoryControllerTest extends BaseWebTestCase
{

    public function test_getProductCategoryAction_invalid()
    {
        //arrange
        $thread = 'product';
        $rootCategory = CategoryQuery::create()
            ->filterByTreeLevel(0)
            ->useCategoryThreadQuery()
                ->filterByThread($thread)
            ->endUse()
            ->findOne();

        /** @var Category $rootCategory */
        $categories = $rootCategory->getChildren(CategoryQuery::create()->filterByStatus(true));

        //act
        $this->client->request('GET', $this->generateUrl('widget_product_api_productcategory_getproductcategory', array('id' => 1)));
        $response = $this->client->getResponse();

        //assert
        $this->assertTrue($response->isNotFound());
    }

    /**
     * 找下層分支
     */
    public function test_getProductCategoryAction_valid()
    {
        //arrange
        $thread = 'product';
        $rootCategory = CategoryQuery::create()
            ->filterByTreeLevel(0)
            ->useCategoryThreadQuery()
                ->filterByThread($thread)
            ->endUse()
            ->findOne();

        /** @var Category $rootCategory */
        $categories = $rootCategory->getChildren(CategoryQuery::create()->filterByStatus(true));

        //act
        $this->client->request('GET', $this->generateUrl('widget_product_api_productcategory_getproductcategory'));
        $response = $this->client->getResponse();
        $result = json_decode($response->getContent(), true);

        //assert
        $this->assertTrue($response->isOk());
        $this->assertEquals(10, count($result));
    }

    /**
     * 找上層分支
     */
    public function test_getProductCategoryAncestorsAction_valid()
    {
        //arrange
        $thread = 'product';
        $rootCategory = CategoryQuery::create()
            ->filterByTreeLevel(1)
            ->useCategoryThreadQuery()
                ->filterByThread($thread)
            ->endUse()
            ->findOne();
        $id = $rootCategory->getId();

        /** @var Category $rootCategory */
        $categories = $ancestors = CategoryQuery::create()
            ->filterByStatus(true)
            ->inTree($rootCategory->getScopeValue())
            ->filterByTreeLeft($rootCategory->getLeftValue(), \Criteria::LESS_EQUAL)
            ->filterByTreeRight($rootCategory->getRightValue(), \Criteria::GREATER_EQUAL)
            ->orderByBranch()
            ->find();

        //act
        $this->client->request('GET', $this->generateUrl('widget_product_api_productcategory_getproductcategoryancestors', array('id' => $id)));
        $response = $this->client->getResponse();
        $result = json_decode($response->getContent(), true);

        //assert
        $this->assertTrue($response->isOk());
        $this->assertEquals(2, count($result));
    }

    public function test_getProductCategoryAction_invalid_root_status()
    {
        //arrange
        $thread = 'product';
        $rootCategory = CategoryQuery::create()
            ->filterByTreeLevel(0)
            ->useCategoryThreadQuery()
                ->filterByThread($thread)
            ->endUse()
            ->findOne();

        /** @var Category $rootCategory */
        $rootCategory->setStatus(false);
        $rootCategory->save();
        $categories = $rootCategory->getChildren(CategoryQuery::create()->filterByStatus(true));

        //act
        $this->client->request('GET', $this->generateUrl('widget_product_api_productcategory_getproductcategory'));
        $response = $this->client->getResponse();

        //assert
        $this->assertTrue($response->isNotFound());
        $rootCategory->reload(true);
        $rootCategory->setStatus(true);
        $rootCategory->save();
    }

    /**
     * 檢查格式是否正確
     */
    public function test_getProductCategoryAction_valid_data()
    {
        //arrange
        $thread = 'product';
        $rootCategory = CategoryQuery::create()
            ->filterByTreeLevel(0)
            ->useCategoryThreadQuery()
                ->filterByThread($thread)
            ->endUse()
            ->findOne();

        /** @var Category $rootCategory */
        $categories = (array) $rootCategory->getChildren(CategoryQuery::create()->filterByStatus(true));
        $context = SerializationContext::create()->setGroups(array('list'));
        $expectedData = json_decode($this->client->getContainer()->get('jms_serializer')->serialize($categories, 'json', $context), true);

        //act
        $this->client->request('GET', $this->generateUrl('widget_product_api_productcategory_getproductcategory'));
        $response = $this->client->getResponse();
        $result = json_decode($response->getContent(), true);

        //assert
        $this->assertTrue($response->isOk());
        $this->assertEquals($expectedData, $result);
    }

    public function test_getProductAction_invaild_URLid()
    {
        //arrange
        $thread = 'product';

        /** @var Category $category */
        $category = CategoryQuery::create()
            ->filterByStatus(true)
            ->useCategoryThreadQuery()
                ->filterByThread($thread)
            ->endUse()
            ->findOne();

        //act
        $this->client->request('GET', $this->generateUrl('widget_product_api_productcategory_getproduct', array('id' => 0)));
        $response = $this->client->getResponse();

        //assert
        $this->assertTrue($response->isNotFound());
    }

    public function test_getProductAction_invaild_status()
    {
        //arrange
        $thread = 'product';

        /** @var Category $category */
        $category = CategoryQuery::create()
            ->filterByStatus(false)
            ->useCategoryThreadQuery()
                ->filterByThread($thread)
            ->endUse()
            ->findOne();
        $id = $category->getId();

        //act
        $this->client->request('GET', $this->generateUrl('widget_product_api_productcategory_getproduct', array('id' => $id)));
        $response = $this->client->getResponse();

        //assert
        $this->assertTrue($response->isNotFound());
    }

    public function test_getProductAction_invaild_thread()
    {
        //arrange
        $id = '123134564';

        //act
        $this->client->request('GET', $this->generateUrl('widget_product_api_productcategory_getproduct', array('id' => $id)));
        $response = $this->client->getResponse();

        //assert
        $this->assertTrue($response->isNotFound());
    }

    public function test_getProductAction_vaild()
    {
        //arrange
        $thread = 'product';
        $requestData = array(
            'sort' => 'created_at',
            'orderby' => 'DESC',
            'page' => '1',
            'rowsperpage' => '10',
        );

        /** @var Category $category */
        $category = CategoryQuery::create()
            ->filterByStatus(true)
            ->useCategoryThreadQuery()
                ->filterByThread($thread)
            ->endUse()
            ->findOne();
        $id = $category->getId();

        //act
        $this->client->request(
            'GET',
            $this->generateUrl('widget_product_api_productcategory_getproduct', array_merge(array('id' => $id), $requestData))
        );
        $response = $this->client->getResponse();

        //assert
        $this->assertTrue($response->isOk());
    }

    /**
     * 檢查格式是否正確
     */
    public function test_getProductAction_vaild_data()
    {
        //arrange
        $thread = 'product';
        $requestData = array(
            'sort' => 'created_at',
            'orderby' => 'DESC',
            'page' => '1',
            'rowsperpage' => '10',
        );

        /** @var Category $category */
        $category = CategoryQuery::create()
            ->filterByStatus(true)
            ->filterByTreeLevel(1)
            ->useCategoryThreadQuery()
                ->filterByThread($thread)
            ->endUse()
            ->findOne();
        $id = $category->getId();

        $pager = ProductQuery::create()
            ->distinct()
            ->filterByStatusShelf(true)
            ->orderBy($requestData['sort'], $requestData['orderby'])
            ->useCategoryProductQuery()
                ->useCategoryQuery()
                    ->filterByTreeLeft($category->getTreeLeft(), \Criteria::GREATER_EQUAL)
                    ->filterByTreeRight($category->getTreeRight(), \Criteria::LESS_EQUAL)
                ->endUse()
            ->endUse();

        $data = array(
            'category' => $category,
            'products' => $pager->paginate($requestData['page'], $requestData['rowsperpage']),
        );
        $context = SerializationContext::create()->setSerializeNull(true)->setGroups(array('list'));
        $expectedData = json_decode($this->client->getContainer()->get('jms_serializer')->serialize($data, 'json', $context), true);

        //act
        $this->client->request(
            'GET',
            $this->generateUrl('widget_product_api_productcategory_getproduct', array_merge(array('id' => $id), $requestData))
        );
        $response = $this->client->getResponse();
        $result = json_decode($response->getContent(), true);

        //assert
        $this->assertTrue($response->isOk());
        $this->assertEquals($expectedData, $result);
    }

}