<?php
namespace Widget\ProductBundle\Controller\BackendAPI;

use Backend\BaseBundle\Controller\BackendAPI\BaseBackendAPIController;
use Backend\BaseBundle\Form\Type\APIFormTypeItem;
use Backend\BaseBundle\Form\Type\RelationType;
use Sensio\Bundle\FrameworkExtraBundle\Configuration\Method;
use Sensio\Bundle\FrameworkExtraBundle\Configuration\Route;
use Sensio\Bundle\FrameworkExtraBundle\Configuration\Security;
use Symfony\Component\HttpFoundation\Request;

use Symfony\Component\Validator\Constraints\GreaterThan;
use Symfony\Component\Validator\Constraints\NotBlank;
use Symfony\Component\Validator\Constraints\Regex;
use Symfony\Component\Validator\Constraints\Type;
use Widget\CategoryBundle\Model\CategoryQuery;
use Widget\PhotoBundle\Form\Type\APIPhotoListType;
use Widget\ProductBundle\Model\Product;
use Widget\ProductBundle\Model\ProductPeer;
use Widget\ProductBundle\Model\ProductQuery;

/**
 * @Route("/product")
 * @Security("has_role_or_superadmin('ROLE_PRODUCT')")
 */
class ProductController extends BaseBackendAPIController
{

    /**
     * 欄位表單設定
     * 主要是USER送過來的欄位
     * @return APIFormTypeItem[]
     */
    protected function getFormConfig()
    {
        return array(
            (new APIFormTypeItem('name'))->setOptions(array(
                'required' => true,
                'constraints' => array(
                    new NotBlank(array(
                        'message' => 'error.product.name.missing_field',
                    )),
                )                
            )),
            (new APIFormTypeItem('feature')),
            (new APIFormTypeItem('origin_price'))->setOptions(array(
                'required' => true,
                'constraints' => array(
                    new NotBlank(array(
                        'message' => 'error.product.origin_price.missing_field',
                    )),
                    new GreaterThan(array(
                        'value' => -1,
                        'message' => "error.product.greater_than_0",
                    ))                    
                )                
            )),
            (new APIFormTypeItem('price'))->setOptions(array(
                'required' => true,
                'constraints' => array(
                    new NotBlank(array(
                        'message' => 'error.product.price.missing_field',
                    )),
                    new GreaterThan(array(
                        'value' => -1,
                        'message' => "error.product.greater_than_0",
                    ))                    
                )                
            )),
            (new APIFormTypeItem('brief'))->setOptions(array(
                'constraints' => array(
                    new NotBlank(array(
                        'message' => 'error.product.brief.missing_field',
                    )),
                )
            )),
            new APIFormTypeItem('code'),
            new APIFormTypeItem('content'),
            new APIFormTypeItem('status_shelf'),
            (new APIFormTypeItem('photos'))->setFieldType(new APIPhotoListType()),
        );
    }

    /**
     * 新增
     * @Route("s")
     * @Method({"POST"})
     * @Security("has_role_or_superadmin('ROLE_PRODUCT_WRITE')")
     */
    public function createAction(Request $request)
    {
        return $this->doProcessForm(new Product(), $request->getContent());
    }

    /**
     * 搜尋
     * @Route("s")
     * @Method({"GET"})
     * @Security("has_role_or_superadmin('ROLE_PRODUCT_READ')")
     */
    public function searchAction(Request $request)
    {
        return $this->doSearch($request->query->all(), ProductQuery::create()->distinct(), ProductPeer::class);
    }

    /**
     * 讀取
     * @Route("/{id}")
     * @Method({"GET"})
     * @Security("has_role_or_superadmin('ROLE_PRODUCT_READ')")
     */
    public function readAction(Product $product)
    {
        return $this->createJsonSerializeResponse($product, array('detail'));
    }

    /**
     * 更新
     * @Route("/{id}")
     * @Method({"PUT"})
     * @Security("has_role_or_superadmin('ROLE_PRODUCT_WRITE')")
     */
    public function updateAction(Request $request, Product $product)
    {
        return $this->doProcessForm($product, $request->getContent());
    }

    /**
     * 刪除
     * 產品不能刪除
     * @Route("/{id}")
     * @Method({"DELETE"})
     * @Security("has_role_or_superadmin('ROLE_PRODUCT_WRITE')")
     */
    public function deleteAction(Product $product)
    {
//        $product->delete();
//        return $this->createJsonResponse();
    }

    /**
     * @Route("/check/stockbundle")
     * @Method({"GET"})
     * @Security("has_role_or_superadmin('ROLE_PRODUCT_WRITE')")
     */
    public function checkStockAction()
    {
        $bundles = $this->container->getParameter('kernel.bundles');
        $enableStockBundle = false;
        foreach ($bundles as $key => $bundle) {
            if ($key == 'WidgetStockBundle') {
                $enableStockBundle = true;
            }
        }
        return $this->createJsonResponse($enableStockBundle);
    }

    /**
     * 批次管理
     * @Route("s")
     * @Method({"PUT"})
     * @Security("has_role_or_superadmin('ROLE_PRODUCT_WRITE')")
     */
    public function batchAction(Request $request)
    {
        return parent::batchAction($request);
    }

    /**
     * 批次切換狀態處理
     * @param $ids
     * @param $con
     * @param $column
     */
    protected function batchSwitch($ids, $column, \PropelPDO $con)
    {
        $products = ProductQuery::create()->findPks($ids, $con);
        foreach ($products as $product) {
            $value = $product->getByName($column, \BasePeer::TYPE_FIELDNAME);
            $product->setByName($column, !$value, \BasePeer::TYPE_FIELDNAME);
            $product->save($con);
        }
    }

    /**
     * 批次設定值
     * @param $ids
     * @param $con
     * @param $column
     */
    protected function batchValue($ids, $column, $value, \PropelPDO $con)
    {
        $products = ProductQuery::create()->findPks($ids, $con);
        foreach ($products as $product) {
            $product->setByName($column, $value, \BasePeer::TYPE_FIELDNAME);
            $product->save($con);
        }
    }

    /**
     * 批次切換狀態處理
     * @param $ids
     * @param $con
     * @param $column
     */
    protected function batchDelete($ids, \PropelPDO $con)
    {
        ProductQuery::create()->findPks($ids, $con)->delete($con);
    }

    /**
     * 設定產品 產品分類 關聯
     * @Route("/{id}/categoryproduct")
     * @Method({"POST"})
     */
    public function relateToCategoryAction(Request $request, Product $product)
    {
        $requestParams = json_decode($request->getContent(), true);
        if (!isset($requestParams['categories']) || count($requestParams['categories']) == 0) {
            $emptyCategory = CategoryQuery::create()->findPks(array());
            $product->setCategories($emptyCategory)->save();
            return $this->createJsonSerializeResponse($product, array('list'));
        }

        $categoryIds = array();
        foreach ($requestParams['categories'] as $category){
            $categoryIds[] = $category['id'];
        }
        $categories = CategoryQuery::create()->findPks($categoryIds);
        $product->setCategories($categories)->save();
        return $this->createJsonSerializeResponse($product, array('list'));
    }
}