<?php
namespace Widget\ProductBundle\Controller\API;

use Backend\BaseBundle\Controller\API\BaseController;
use Backend\BaseBundle\Model\Site;
use Backend\BaseBundle\Propel\I18n;
use JMS\DiExtraBundle\Annotation as DI;
use Sensio\Bundle\FrameworkExtraBundle\Configuration\Cache;
use Sensio\Bundle\FrameworkExtraBundle\Configuration\Method;
use Sensio\Bundle\FrameworkExtraBundle\Configuration\ParamConverter;
use Sensio\Bundle\FrameworkExtraBundle\Configuration\Route;
use Symfony\Component\EventDispatcher\EventDispatcherInterface;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;
use Widget\CategoryBundle\Model\Category;
use Widget\CategoryBundle\Model\CategoryQuery;
use Widget\ProductBundle\Event\ProductModelJoinStockEvent;
use Widget\ProductBundle\Model\Product;
use Widget\ProductBundle\Model\ProductQuery;

/**
 * @Route("product")
 */
class ProductController extends BaseController
{
    /**
     * @DI\Inject()
     * @var EventDispatcherInterface
     */
    protected $eventDispatcher;

    /**
     * 取得Product的API
     * @Route("/{id}")
     * @Method({"GET"})
     */
    public function getProductAction(Request $request, Product $product)
    {
        if (!$product->getStatusShelf()) {
            throw $this->createNotFoundException();
        }
        if ($product instanceof I18n){
            $locale = $request->query->get('_locale');
            $product->setLocale($locale);
        }
        return $this->createJsonSerializeResponse($product, array('detail'));
    }

    /**
     * 取得Category的API
     * @Route("/{id}/categories")
     * @Method({"GET"})
     */
    public function getCategoryAction(Request $request, Product $product)
    {
        $threadId = $request->get('thread', 'product');
        $categories = $product->getCategories(
            CategoryQuery::create()
            ->filterByStatus(true)
            ->useCategoryThreadQuery()
                ->filterByThread($threadId)
            ->endUse()
        );

        return $this->createJsonSerializeResponse($categories, array('list'));
    }

    /**
     * Search的API
     * @Route("/search")
     * @Method({"POST"})
     */
    public function searchAction(Request $request)
    {
        $parameters = json_decode($request->getContent(), true);

        $page = $request->query->getInt('page', 1);
        $rowPerPage = $request->query->getInt('rowsperpage', 10);
        $sort = $request->query->get('sort');
        $orderby = $request->query->get('orderby');

        $productQuery = ProductQuery::create();

        if (isset($parameters['keyword']) || '' !== trim($parameters['keyword'])) {
            if ($productQuery instanceof I18n){
                $locale = $request->query->get('_locale');
                $productQuery->joinProductI18n($locale)
                    ->useProductI18nQuery()
                        ->filterByName("%{$parameters['keyword']}%", \Criteria::LIKE)
                    ->endUse();
            }else {
                $productQuery->filterByName("%{$parameters['keyword']}%", \Criteria::LIKE);
            }
        }
        if ($sort !== null && $orderby !== null) {
            $productQuery->orderBy($sort, $orderby);
        }

        $productQuery
            ->filterByStatusShelf(true) // 只搜尋上架產品
            ->distinct();  //重複的就要排除

        if (isset($parameters['advanced']) && is_array($parameters['advanced'])) {
            $this->useAdvancedQuery($productQuery, $parameters['advanced'], $parameters['keyword']);
        }

        $event = new ProductModelJoinStockEvent($productQuery);
        $this->eventDispatcher->dispatch(ProductModelJoinStockEvent::EVENT_NAME, $event);
        return $this->createJsonSerializeResponse($productQuery->paginate($page, $rowPerPage), array('list'));
    }

    protected function useAdvancedQuery(ProductQuery $productQuery, $parameters, $keywords = null)
    {
        if (isset($parameters['search_brief']) && $parameters['search_brief']) {
            $productQuery
                ->_or()
                ->filterByBrief("%{$keywords}%", \Criteria::LIKE)
                ->filterByStatusShelf(true);
        }

        if (isset($parameters['category_id']) && !isset($parameters['with_sub_category'])) {
            $productQuery
                ->useCategoryProductQuery()
                    ->filterByCategoryId($parameters['category_id'])
                ->endUse();
        }
        if (isset($parameters['category_id']) && isset($parameters['with_sub_category']) && $parameters['with_sub_category']) {
            if($category = CategoryQuery::create()
                ->findPk($parameters['category_id'])){
                $productQuery
                    ->useCategoryProductQuery()
                        ->useCategoryQuery()
                            ->filterByTreeLeft($category->getTreeLeft(), \Criteria::GREATER_EQUAL)
                            ->filterByTreeRight($category->getTreeRight(), \Criteria::LESS_EQUAL)
                        ->endUse()
                    ->endUse();
            }
        }

        $priceRange = array();
        if (isset($parameters['pricerange_start'])) {
            $priceRange['min'] = (int) $parameters['pricerange_start'];
        }
        if (isset($parameters['pricerange_end'])) {
            $priceRange['max'] = (int) $parameters['pricerange_end'];
        }
        if (count($priceRange) >= 1) {
            $productQuery->filterByPrice($priceRange);
        }
    }
}