<?php

namespace Widget\ProductStyleBundle\Controller\BackendAPI;

use Backend\BaseBundle\Controller\BackendAPI\BaseBackendAPIController;
use Backend\BaseBundle\Form\Type\APIFormTypeItem;
use JMS\DiExtraBundle\Annotation\InjectParams;
use Sensio\Bundle\FrameworkExtraBundle\Configuration\Method;
use Sensio\Bundle\FrameworkExtraBundle\Configuration\Route;
use Sensio\Bundle\FrameworkExtraBundle\Configuration\Security;
use Symfony\Component\EventDispatcher\EventDispatcher;
use Symfony\Component\EventDispatcher\EventDispatcherInterface;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;
use Widget\ProductBundle\Model\Product;
use Widget\ProductBundle\Model\ProductQuery;
use Widget\ProductStyleBundle\Event\ProductStyleStockCreateEvent;
use Widget\ProductStyleBundle\Event\ProductStyleStockDeleteEvent;
use Widget\ProductStyleBundle\Model\ProductStyle;
use Widget\ProductStyleBundle\Model\ProductStyleConfig;
use Widget\ProductStyleBundle\Model\ProductStyleConfigQuery;
use Widget\ProductStyleBundle\Model\ProductStyleLabel;
use Widget\ProductStyleBundle\Model\ProductStyleMix;
use Widget\ProductStyleBundle\Model\ProductStyleMixQuery;
use Widget\ProductStyleBundle\Model\ProductStyleQuery;
use Widget\ProductStyleBundle\Model\ProductStyleTemplate;
use Widget\ProductStyleBundle\Model\ProductStyleTemplateLog;
use Widget\ProductStyleBundle\Model\ProductStyleTemplateQuery;
use Widget\ProductStyleBundle\Model\ProductToStyleConfig;

/**
 * @Route("/productstyle")
 * @Security("has_role_or_superadmin('ROLE_PRODUCT')")
 */
class ProductStyleController extends BaseBackendAPIController
{
    /** @var  EventDispatcher */
    protected $dispatcher;

    public function getFormConfig()
    {
        return array(
            //new APIFormTypeItem('')
        );
    }

    /**
     * @InjectParams()
     */
    public function injectDispatcher(EventDispatcherInterface $eventDispatcher)
    {
        $this->dispatcher = $eventDispatcher;
    }

    /**
     * 產生組合試算
     * @Route("/query")
     * @Method({"POST"})
     * @Security("has_role_or_superadmin('ROLE_PRODUCT_WRITE')")
     */
    public function queryAction(Request $request)
    {
        $content = json_decode($request->getContent(), true);

        if (isset($content['default_style']) && $content['default_style'] === true){
            return $this->createJsonResponse(array('無屬性'));
        }

        $templateStyle = array();
        foreach ($content['template_ids'] as $id){
            $productStyleTemplate = ProductStyleTemplateQuery::create()->findPk($id);
            if (isset($productStyleTemplate->getConfig()['attrs'])){
                $templateStyle[] = $productStyleTemplate->getConfig()['attrs'];
            }
        }

        $permutation = $this->combinations($templateStyle);

        $styleArray = array();
        // 組回字串
        foreach ($permutation as $value) {
            if( is_array($value) ){
                $styleArray[] = implode("-", $value);
            }
            else {
                $styleArray[] = $value;
            }
        }
        return $this->createJsonResponse($styleArray);
    }

    /**
     * 新增組合至產品
     * @Route("s")
     * @Method({"POST"})
     * @Security("has_role_or_superadmin('ROLE_PRODUCT_WRITE')")
     */
    public function createAction(Request $request)
    {
        $content = json_decode($request->getContent(), true);
        if (count($content['template_ids']) == 0 && $content['default_style'] === true){
            $this->createHttpExceptionResponse(Response::HTTP_BAD_REQUEST, 'error.product_style.attr.missing_field');
        }
        $productId = $content['product_id'];
        $product = ProductQuery::create()->findPk($productId);

        if ($product == null){
            return $this->createHttpExceptionResponse(Response::HTTP_NOT_FOUND);
        }
        
        $defaultStyle = $content['default_style'];
        if($defaultStyle){
            $styleStockArray[] = $this->createProductStyleDefault($productId);
            $this->emitProductStyleStockCreateEvent($productId, $styleStockArray);
            $product = ProductQuery::create()->findPk($content['product_id']);
            return $this->createJsonSerializeResponse($product, array('detail'));
        }

        $productStyleTemplates = ProductStyleTemplateQuery::create()->findPks($content['template_ids']);
        if ($productStyleTemplates == null) {
            return $this->createHttpExceptionResponse(Response::HTTP_NOT_FOUND);
        }
        $templateStyles = array();
        // 組出 array 格式
        /** @var ProductStyleTemplate $productStyleTemplate */
        foreach ($productStyleTemplates as $productStyleTemplate){
            if ($productStyleTemplate->getConfig()){
                $templateStyles[] = $productStyleTemplate->getConfig()['attrs'];

                $styleLog = array(
                    'label' => $productStyleTemplate->getTemplateName(),
                    'attrs' => $productStyleTemplate->getConfig()['attrs'],
                );
                $styleLog['attrs']['keys'] = array_keys($styleLog['attrs']);

                $productStyleTemplateLog = new  ProductStyleTemplateLog();
                $productStyleTemplateLog->setConfig($productStyleTemplate->getConfig());
                $productStyleTemplateLog->save();
                // 寫入組合時的log
                $productToStyle = new ProductToStyleConfig();
                $productToStyle->setProductId($product->getId());
                $productToStyle->setProductStyleTemplateLogId($productStyleTemplateLog->getId());
                $productToStyle->setStyleLog($styleLog);
                $product->addProductToStyleConfig($productToStyle);
                $product->save();
            }
        }

        $permutation = $this->combinations($templateStyles);
        $styleStockArray = array();
        foreach ($permutation as $key => $value) {
            if(is_array($value)){
                $attrName = (implode("-", $value));
                $attrId = "{$productId}-{$key}";
                $styleStockArray[] = $this->createProductStyleMix($productId,$attrId, $attrName);
            }
            else {
                $attrId = "{$productId}-{$key}";
                $styleStockArray[] = $this->createProductStyleMix($productId,$attrId, $value);
            }
        }

        $this->emitProductStyleStockCreateEvent($productId, $styleStockArray);

        $product = ProductQuery::create()->findPk($content['product_id']);
        return $this->createJsonSerializeResponse($product, array('detail'));
    }

    /**
     * 讀取
     * @Route("/{id}")
     * @Method({"GET"})
     * @Security("has_role_or_superadmin('ROLE_PRODUCT_READ')")
     */
    public function readAction(Request $request, Product $product)
    {
        return $this->createJsonSerializeResponse($product, array('detail'));
    }

    /**
     * 修改
     * @Route("/{id}")
     * @Method({"PUT"})
     * @Security("has_role_or_superadmin('ROLE_PRODUCT_WRITE')")
     */
    public function updateAction(Request $request, Product $product)
    {

    }

    /**
     * 刪除 (請帶 product ID)
     * @Route("/{id}")
     * @Method({"DELETE"})
     * @Security("has_role_or_superadmin('ROLE_PRODUCT_WRITE')")
     */
    public function deleteAction(Product $product)
    {
        $product->getProductToStyleConfigs()->delete();
        $product->getProductStyleTemplateLogs()->delete();
        $event = new ProductStyleStockDeleteEvent($product->getId());
        $this->dispatcher->dispatch(ProductStyleStockDeleteEvent::EVENT_NAME, $event);
        return $this->createJsonResponse();
    }

    /**
     * 組出各種組合
     * @param $arrays
     * @param int $i
     * @return array
     */
    public function combinations($arrays, $i = 0) {
        if (!isset($arrays[$i])) {
            return array();
        }
        if ($i == count($arrays) - 1) {
            return $arrays[$i];
        }

        // get combinations from subsequent arrays
        $tmp = $this->combinations($arrays, $i + 1);

        $result = array();

        // concat each array from tmp with each element from $arrays[$i]
        foreach ($arrays[$i] as $key => $v) {
            foreach ($tmp as $tempKey => $t) {
                $result["$key-$tempKey"] = is_array($t) ?
                    array_merge(array($v), $t) :
                    array($v, $t);
            }
        }

        return $result;
    }

    /**
     * 建立產品屬性
     * @param $productId
     * @param $labelId
     * @param $attrName
     * @return ProductStyle
     */
    protected function createProductStyle($productId, $labelId, $attrName)
    {
        $productStyle = new ProductStyle();
        $productStyle->setName($attrName);
        $productStyle->setLabelId($labelId);
        $productStyle->setProductId($productId);
        $productStyle->save();
        return $productStyle;
    }

    /**
     * 建立產品屬性組合
     * @param $productId
     * @param $attrId
     * @param $attrMix
     * @return ProductStyleMix
     * @throws \Exception
     * @throws \PropelException
     */
    protected function createProductStyleMix($productId, $attrId, $attrMix)
    {

        $productStyleMix = new ProductStyleMix();
        $productStyleMix->setId($attrId);
        $productStyleMix->setProductId($productId);
        $productStyleMix->setName($attrMix);
        $productStyleMix->save();

        return $productStyleMix;
    }

    /**
     * 建立無屬性的產品
     */
    protected function createProductStyleDefault($productId)
    {
        $productStyleMix = new ProductStyleMix();
        $productStyleMix->setId($productId);
        $productStyleMix->setProductId($productId);
        $productStyleMix->setName("");
        $productStyleMix->save();

        return $productStyleMix;
    }
    
    protected function emitProductStyleStockCreateEvent($productId, $styleStockArray)
    {
        $productStyleStockCreateEvent = new ProductStyleStockCreateEvent($productId, $styleStockArray);
        $this->dispatcher->dispatch(ProductStyleStockCreateEvent::EVENT_NAME,$productStyleStockCreateEvent);        
    }
}