<?php
namespace Widget\PhotoBundle\Test\Controller\Backend;

use Backend\BaseBundle\Model\SiteUserQuery;
use Backend\BaseBundle\Tests\Fixture\BackendWebTestCase;
use Widget\PhotoBundle\Model\PhotoConfig;
use Widget\PhotoBundle\Model\PhotoConfigQuery;

class ConfigControllerTest extends BackendWebTestCase
{
    /**
     * 沒有權限不應該看到 menu
     */
    public function test_no_rights_should_not_see_menu_entry()
    {
        //arrange
        $slug = 'dgfactor';
        $user = SiteUserQuery::create()->findOneByLoginName('admin');
        $this->loginWithRoles($user);

        //act
        $crawler = $this->client->request('GET', $this->generateUrl('backend_base_backend_default_index', array('slug' => $slug)));
        $response = $this->client->getResponse();

        //assert
        $this->assertTrue($response->isOk());
        $this->assertEquals(0, $crawler->filter("#side-menu li a:contains('照片設定管理')")->count());
    }

    public function test_no_rights_should_not_enter_index()
    {
        //arrange
        $slug = 'dgfactor';
        $user = SiteUserQuery::create()->findOneByLoginName('admin');
        $this->loginWithRoles($user);

        //act
        $crawler = $this->client->request('GET', $this->generateUrl('widget_photo_backend_config_index', array('slug' => $slug)));
        $response = $this->client->getResponse();

        //assert
        $this->assertFalse($response->isOk());
    }

    public function test_has_rights_should_see_menu_entry()
    {
        //arrange
        $slug = 'dgfactor';
        $user = SiteUserQuery::create()->findOneByLoginName('admin');
        $roles = array('ROLE_PHOTO_CONFIG_READ');
        $this->loginWithRoles($user, $roles);

        //act
        $crawler = $this->client->request('GET', $this->generateUrl('backend_base_backend_default_index', array('slug' => $slug)));
        $response = $this->client->getResponse();

        //assert
        $this->assertTrue($response->isOk());
        $this->assertGreaterThanOrEqual(1, $crawler->filter("#side-menu li a:contains('照片設定管理')")->count());
    }

    public function test_has_rights_can_enter_index()
    {
        //arrange
        $slug = 'dgfactor';
        $user = SiteUserQuery::create()->findOneByLoginName('admin');
        $roles = array('ROLE_PHOTO_CONFIG_READ');
        $this->loginWithRoles($user, $roles);

        //act
        $crawler = $this->client->request('GET', $this->generateUrl('widget_photo_backend_config_index', array('slug' => $slug)));
        $response = $this->client->getResponse();

        //assert
        $this->assertTrue($response->isOk());
    }

    public function test_no_rights_should_not_enter_delete()
    {
        //arrange
        $slug = 'dgfactor';
        $user = SiteUserQuery::create()->findOneByLoginName('admin');
        $photoConfig = PhotoConfigQuery::create()->findOne();
        $roles = array('ROLE_PHOTO_CONFIG_READ');

        $tokenManager = $this->client->getContainer()->get('security.csrf.token_manager');
        $token = $tokenManager->getToken('delete', $roles);
        $this->loginWithRoles($user);

        //act
        $crawler = $this->client->request('GET', $url = $this->generateUrl('widget_photo_backend_config_delete', array('slug' => $slug, 'id' => $photoConfig->getId())).'?_token='.urlencode($token));
        $response = $this->client->getResponse();

        //assert
        $this->assertFalse($response->isOk());
    }

    public function test_has_rights_should_enter_delete()
    {
        //arrange
        $slug = 'dgfactor';
        $user = SiteUserQuery::create()->findOneByLoginName('admin');
        $photoConfig = PhotoConfigQuery::create()->findOne();
        $photoConfigArray = $photoConfig->toArray();

        $roles = array('ROLE_PHOTO_CONFIG_READ', 'ROLE_PHOTO_CONFIG_WRITE');
        $tokenManager = $this->client->getContainer()->get('security.csrf.token_manager');
        $token = $tokenManager->getToken('delete', $roles);
        $this->loginWithRoles($user, $roles);

        //act
        $crawler = $this->client->request('GET', $url = $this->generateUrl('widget_photo_backend_config_delete', array('slug' => $slug, 'id' => $photoConfig->getId())).'?_token='.urlencode($token));
        $response = $this->client->getResponse();

        //assert
        $this->assertTrue($response->isRedirect($this->generateUrl('widget_photo_backend_config_index', array('slug' => $slug))));
        $photoConfig = new PhotoConfig();
        $photoConfig->fromArray($photoConfigArray);
        $photoConfig->save();
    }

    public function test_no_rights_should_not_enter_edit()
    {
        //arrange
        $slug = 'dgfactor';
        $user = SiteUserQuery::create()->findOneByLoginName('admin');
        $photoConfig = PhotoConfigQuery::create()->findOne();
        $roles = array('ROLE_PHOTO_CONFIG_READ');
        $this->loginWithRoles($user);

        //act
        $crawler = $this->client->request('GET', $url = $this->generateUrl('widget_photo_backend_config_edit', array('slug' => $slug, 'id' => $photoConfig->getId())));
        $response = $this->client->getResponse();

        //assert
        $this->assertFalse($response->isOk());
    }

    public function test_has_rights_should_enter_edit()
    {
        //arrange
        $slug = 'dgfactor';
        $user = SiteUserQuery::create()->findOneByLoginName('admin');
        $photoConfig = PhotoConfigQuery::create()->findOne();
        $photoConfigArray = $photoConfig->toArray();

        $roles = array('ROLE_PHOTO_CONFIG_READ', 'ROLE_PHOTO_CONFIG_WRITE');
        $tokenManager = $this->client->getContainer()->get('security.csrf.token_manager');
        $this->loginWithRoles($user, $roles);

        //act
        $crawler = $this->client->request('GET', $url = $this->generateUrl('widget_photo_backend_config_edit', array('slug' => $slug, 'id' => $photoConfig->getId())));
        $response = $this->client->getResponse();

        //assert
        $this->assertTrue($response->isOk());
    }

}