<?php
namespace Widget\PhotoBundle\EventListener\ControllerConfig;

use Backend\BaseBundle\Form\Type\SubmitBackType;
use JMS\DiExtraBundle\Annotation\Service;
use JMS\DiExtraBundle\Annotation\Tag;
use JMS\DiExtraBundle\Annotation\Inject;
use JMS\DiExtraBundle\Annotation\InjectParams;
use Backend\BaseBundle\Event\Controller;
use Backend\BaseBundle\Form\Type;
use Symfony\Component\Form\FormBuilderInterface;
use Symfony\Component\Routing\Router;
use Symfony\Component\Validator\Constraints;
use Symfony\Component\Form\Extension\Core\Type as BaseType;
use Symfony\Component\Validator\Context\ExecutionContextInterface;
use Widget\PhotoBundle\Image\PhotoConfigFinder;
use Widget\PhotoBundle\Model\PhotoConfig;
use Widget\PhotoBundle\Model\PhotoConfigQuery;

/**
 * @Service
 * @Tag("kernel.event_listener", attributes = {"public": false, "event": Controller\ConfigEvent::EVENT_CONFIG, "method": "onConfig"})
 * @Tag("kernel.event_listener", attributes = {"public": false, "event": Controller\TypeRolesEvent::EVENT_TYPE_ROLES, "method": "onTypeRoles"})
 * @Tag("kernel.event_listener", attributes = {"public": false, "event": Controller\MenuEvent::EVENT_MENU, "method": "onMenu"})
 */
class ConfigListener
{
    const CONTROLLER_NAME = 'WidgetPhotoBundle:Backend/Config';

    /** @var  Router */
    protected $router;

    /** @var  PhotoConfigFinder */
    protected $configFinder;

    /**
     * @InjectParams()
     */
    public function __construct(Router $router)
    {
        $this->router = $router;
    }

    /**
     * @InjectParams({
     *     "configFinder" = @Inject("widget.photo_bundle.config_finder"),
     * })
     */
    public function injectConfigFinder(PhotoConfigFinder $configFinder)
    {
        $this->configFinder = $configFinder;
    }

    public function onMenu(Controller\MenuEvent $event)
    {
        $user = $event->getUser();
        if(!$user->hasRoleOrSuperAdmin('ROLE_PHOTO_CONFIG_READ')) {
            return;
        }
        $isActive = preg_match('/^widget_photo_backend_/i', $event->getRequestRouteName())>0;
        $slug = $user->getSiteSlug();
        $event->addMenu(self::CONTROLLER_NAME, array(
            'menu.photo_config' => array(
                'class' => 'fa fa-photo',
                'active' => $isActive,
                'target' => $this->router->generate('widget_photo_backend_config_index', array('slug' => $slug)),
            ),
        ));
    }

    public function onConfig(Controller\ConfigEvent $event)
    {
        if($event->getControllerName() != self::CONTROLLER_NAME){
            return;
        }
        $event->setConfig(array(
            'model' => 'PhotoConfig',
            'route' => array(
                'sort' => true,
                'index' => true,
                'edit' => true,
                'new' => true,
                'delete' => true,
            ),
            'index' => array(
                '＃' => 'Id',
                'index.photo_config.name' => 'Name',
                'index.photo_config.brief' => 'Brief',
                'index.photo_config.created_at' => 'CreatedAt',
            ),
            'security' => array(
                'new' =>  'ROLE_PHOTO_CONFIG_WRITE',
                'edit' => 'ROLE_PHOTO_CONFIG_WRITE',
                'delete' => 'ROLE_PHOTO_CONFIG_WRITE',
            ),
            'action' => array(
                'action.edit' => array(
                    'class' => 'fa fa-pencil-square-o',
                    'action' => 'edit'
                ),
                'action.delete' => array(
                    'class' => 'fa fa-trash',
                    'action' => 'delete',
                    'confirm' => 'message.confirm.delete',
                ),
            ),
            'form' => array(
                array(
                    'Name',
                    BaseType\TextType::class,
                    array(
                        'label' => 'form.label.edit.photo_config.name',
                        'required' => true,
                        'help' => 'form.label.help.photo_config.name',
                        'constraints' => array(
                            new Constraints\NotBlank(),
                            new Constraints\Regex(array(
                                'pattern' => '/^[0-9a-z_]+$/',
                                'message' => 'validator.photo_config.name.value_not_allow',
                            )),
                            new Constraints\Callback(function($value, ExecutionContextInterface $context){
                                $photoConfig = $context->getRoot()->getData();
                                $oldPhotoConfig = $this->configFinder->findConfig($photoConfig->getSiteId(), $photoConfig->getName());
                                if(!$oldPhotoConfig){
                                    return;
                                }
                                if($oldPhotoConfig->getName() == $photoConfig->getName() && $oldPhotoConfig->getId() != $photoConfig->getId()){
                                    $context->addViolation('validator.photo_config.name.duplicate');
                                }
                            }),
                        ),
                    ),
                ),
                array(
                    'Brief',
                    BaseType\TextType::class,
                    array(
                        'label' => 'form.label.edit.photo_config.brief',
                        'required' => true,
                        'constraints' => array(
                            new Constraints\NotBlank(),
                        ),
                    ),
                ),
                array(
                    'Crop',
                    BaseType\ChoiceType::class,
                    array(
                        'label' => 'form.label.edit.photo_config.crop',
                        'choices' => array(
                            true => 'form.choice.yes',
                            false => 'form.choice.no',
                        ),
                        'expanded' => true,
                        'required' => true,
                        'constraints' => array(
                            new Constraints\NotBlank(),
                        ),
                    ),
                ),
                array(
                    'Config',
                    BaseType\CollectionType::class,
                    array(
                        'label' => 'form.label.edit.photo_config.config',
                        'attr' => array(
                            'data-button-label' => 'form.label.edit.photo_config.button',
                        ),
                        'type' => new Type\BaseFormType(function(FormBuilderInterface $builder){
                            $builder
                                ->add('type', BaseType\ChoiceType::class, array(
                                    'label' => 'form.label.edit.photo_config_item.type',
                                    'choices' => array(
                                        'inbox' => 'form.choice.photo_config_item.inbox',
                                        'outbox' => 'form.choice.photo_config_item.outbox',
                                        'crop' => 'form.choice.photo_config_item.crop',
                                        'resizecrop' => 'form.choice.photo_config_item.resizecrop',
                                        'fit' => 'form.choice.photo_config_item.fit',
                                    ),
                                ))
                                ->add('suffix', BaseType\TextType::class, array(
                                    'label' => 'form.label.edit.photo_config_item.suffix',
                                    'constraints' => array(
                                        new Constraints\NotBlank(),
                                        new Constraints\NotEqualTo(array(
                                            'value' => 'origin',
                                            'message' => 'validator.photo_config.suffix.value_not_allow',
                                        )),
                                        new Constraints\Regex(array(
                                            'pattern' => '/^[a-z][0-9a-z]+$/',
                                            'message' => 'validator.photo_config.suffix.value_not_allow',
                                        )),
                                    ),
                                ))
                                ->add('width', BaseType\IntegerType::class, array(
                                    'label' => 'form.label.edit.photo_config_item.width',
                                    'constraints' => array(
                                        new Constraints\NotBlank(),
                                        new Constraints\GreaterThan(array(
                                            'value' => 0,
                                        )),
                                    ),
                                ))
                                ->add('height', BaseType\IntegerType::class, array(
                                    'label' => 'form.label.edit.photo_config_item.height',
                                    'constraints' => array(
                                        new Constraints\NotBlank(),
                                        new Constraints\GreaterThan(array(
                                            'value' => 0,
                                        )),
                                    ),
                                ))
                            ;
                        }, 'Config'),
                        'allow_add' => true,
                        'allow_delete'  => true,
                    ),
                ),
                array(
                    'submit',
                    SubmitBackType::class,
                    array(
                        'with_back' => true,
                        'with_reset' => true,
                        'back_route' => 'widget_photo_backend_config_index',
                    ),
                ),
            ),
        ));
    }

    public function onTypeRoles(Controller\TypeRolesEvent $event)
    {
        $event->addTypeRoles('ROLE_PHOTO', array(
            'ROLE_PHOTO_CONFIG_READ',
            'ROLE_PHOTO_CONFIG_WRITE'
        ));
    }
}