<?php
namespace Widget\PaymentBundle\Tests\Payment;

use Backend\BaseBundle\Tests\Fixture\BaseKernelTestCase;
use Symfony\Component\Process\Process;
use Widget\PaymentBundle\Event\GetProcesserEvent;
use Widget\PaymentBundle\Event\ProcessPaymentEvent;
use Widget\PaymentBundle\Event\QueryAvailablePaymentEvent;
use Widget\PaymentBundle\Event\UpdatePaymentTypeEvent;
use Widget\PaymentBundle\Event\VerifyPaymentEvent;
use Widget\PaymentBundle\Tests\AbstractKernelTestCase;
use Widget\PaymentBundle\Payment\Payment;
use Widget\PaymentBundle\Processer;
use Symfony\Component\EventDispatcher\EventDispatcher;

class PaymentTest extends BaseKernelTestCase
{
    public function test_createService()
    {
        $payment = $this->container->get('widget_payment.payment');
        $this->assertInstanceOf(Payment::class, $payment);
    }

    public function test_getProcesser()
    {
        //arrange
        $processer = $this->getMockBuilder(Processer\ProcesserBase::class)
            ->disableOriginalConstructor()
            ->getMockForAbstractClass();
        $payment = $this->getMockBuilder(Payment::class)
            ->disableOriginalConstructor()
            ->setMethods()
            ->getMock();
        $dispatcher = $this->getMockBuilder(EventDispatcher::class)
            ->setMethods(array('dispatch'))
            ->disableOriginalConstructor()
            ->getMock();
        $dispatcher
            ->expects($this->once())
            ->method('dispatch')
            ->willReturnCallback(function($eventName, GetProcesserEvent $event) use($processer){
                $this->assertEquals(GetProcesserEvent::EVENT_NAME, $eventName);
                $this->assertEquals('TestCredit', $event->getPaymentType());
                $event->setProcesser($processer);
            });
        $this->setObjectAttribute($payment, 'dispatcher', $dispatcher);

        //act
        $result = $payment->getProcesser("TestCredit");

        //assert
        $this->assertEquals($processer, $result);
    }

    public function test_processPayment()
    {
        //arrange
        $currency = 'CYN';
        $unit = 2;
        $extraInfo = array(
            'a' => 'a',
            'b' => 'b',
            'c' => 'c',
        );
        $twig = $this->getMockBuilder(\Twig_Environment::class)
            ->disableOriginalConstructor()
            ->setMethods()
            ->getMock();
        $payment = $this->getMockBuilder(Payment::class)
            ->disableOriginalConstructor()
            ->setMethods()
            ->getMock();
        $dispatcher = $this->getMockBuilder(EventDispatcher::class)
            ->setMethods(array('dispatch'))
            ->disableOriginalConstructor()
            ->getMock();
        $dispatcher
            ->expects($this->at(0))
            ->method('dispatch')
            ->willReturnCallback(function($eventName, UpdatePaymentTypeEvent $event) use($currency, $unit){
                $this->assertEquals(UpdatePaymentTypeEvent::EVENT_NAME, $eventName);
                $this->assertEquals('TestCredit', $event->getPaymentType());
                $this->assertEquals('o12345', $event->getOrdernumber());
                $this->assertEquals($currency, $event->getCurrency());
                $this->assertEquals($unit, $event->getUnit());
            });
        $dispatcher
            ->expects($this->at(1))
            ->method('dispatch')
            ->willReturnCallback(function($eventName, ProcessPaymentEvent $event) use($extraInfo, $currency, $unit){
                $this->assertEquals(ProcessPaymentEvent::EVENT_NAME, $eventName);
                $this->assertEquals('TestCredit', $event->getPaymentType());
                $this->assertEquals('o12345', $event->getOrdernumber());
                $this->assertEquals(100, $event->getAmount());
                $this->assertEquals($extraInfo, $event->getPaymentExtraInfo());
                $this->assertEquals($currency, $event->getCurrency());
                $this->assertEquals($unit, $event->getUnit());
            });
        $this->setObjectAttribute($payment, 'dispatcher', $dispatcher);
        $this->setObjectAttribute($payment, 'twig', $twig);

        //act
        $result = $payment->processPayment('TestCredit', 'o12345', 100, $extraInfo, $currency, $unit);

        //assert
        $this->assertInstanceOf(Processer\ProcesserResult::class, $result);
    }

    public function test_verifyPayment()
    {
        //arrange
        $ordernumber = 'o12345';
        $payment = $this->getMockBuilder(Payment::class)
            ->disableOriginalConstructor()
            ->setMethods()
            ->getMock();
        $dispatcher = $this->getMockBuilder(EventDispatcher::class)
            ->setMethods(array('dispatch'))
            ->disableOriginalConstructor()
            ->getMock();
        $dispatcher
            ->expects($this->once())
            ->method('dispatch')
            ->willReturnCallback(function($eventName, VerifyPaymentEvent $event) use($ordernumber){
                $this->assertEquals(VerifyPaymentEvent::EVENT_NAME, $eventName);
                $this->assertEquals($ordernumber, $event->getOrdernumber());
                $this->assertEquals('TestCredit', $event->getPaymentType());
            });
        $this->setObjectAttribute($payment, 'dispatcher', $dispatcher);

        //act
        $result = $payment->verifyPayment("TestCredit", $ordernumber);

        //assert
        $this->assertInstanceOf(VerifyPaymentEvent::class, $result);
    }
    
    public function test_queryAvailablePayment()
    {
        // arrange
        $amount = 500;
        $availablePayment = [];
        $payment = $this->getMockBuilder(Payment::class)
            ->disableOriginalConstructor()
            ->setMethods()
            ->getMock();
        $dispatcher = $this->getMockBuilder(EventDispatcher::class)
            ->setMethods(array('dispatch'))
            ->disableOriginalConstructor()
            ->getMock();
        $dispatcher
            ->expects($this->once())
            ->method('dispatch')
            ->willReturnCallback(function ($eventName , QueryAvailablePaymentEvent $event ) use($amount)  {
                $this->assertEquals(QueryAvailablePaymentEvent::EVENT_NAME, $eventName);
                $this->assertEquals($amount, $event->getAmount());
            });


        $this->setObjectAttribute($payment, 'dispatcher', $dispatcher);

        //act
        $result = $payment->queryAvailablePayment($amount);

        //assert
        $this->assertEquals($availablePayment, $result);
    }

    public function test_queryAvailablePayment_with_payment()
    {
        // arrange
        $amount = 500;
        $availablePayment = ['testPayment1', 'testPayment2'];
        $payment = $this->getMockBuilder(Payment::class)
            ->disableOriginalConstructor()
            ->setMethods()
            ->getMock();
        $dispatcher = $this->getMockBuilder(EventDispatcher::class)
            ->setMethods(array('dispatch'))
            ->disableOriginalConstructor()
            ->getMock();
        $dispatcher
            ->expects($this->once())
            ->method('dispatch')
            ->willReturnCallback(function ($eventName , QueryAvailablePaymentEvent $event ) use($amount)  {
                $this->assertEquals(QueryAvailablePaymentEvent::EVENT_NAME, $eventName);
                $this->assertEquals($amount, $event->getAmount());
                $event->addAvailablePayment('testPayment1');
                $event->addAvailablePayment('testPayment2');
            });


        $this->setObjectAttribute($payment, 'dispatcher', $dispatcher);

        //act
        $result = $payment->queryAvailablePayment($amount);

        //assert
        $this->assertEquals($availablePayment, $result);
    }

}