<?php
namespace Widget\PaymentBundle\Tests\EventListener;

use Backend\BaseBundle\Tests\Fixture\BaseWebTestCase;
use Widget\PaymentBundle\EventListener\PaymentListener;
use Widget\PaymentBundle\Event\ReceiveOrderEvent;
use Widget\PaymentBundle\Event\ReceivePaymentnumberEvent;
use Widget\PaymentBundle\Event\UpdatePaymentTypeEvent;
use Widget\PaymentBundle\Model\Payment;
use Widget\PaymentBundle\Model\PaymentPeer;
use Widget\PaymentBundle\Model\PaymentQuery;

class PaymentListenerTest extends BaseWebTestCase
{
    public function test_onReceivePaymentNumber_error_order_id()
    {
        //arrange
        $orderId = '7987456';
        $payment = PaymentQuery::create()->findPk($orderId);
        $paymentnumber = '800-6521782765376';
        $expiredAt = new \DateTime();
        $paymentInfo = '132174jsdgfdyhugfds';
        $event = new ReceivePaymentnumberEvent($orderId, $paymentnumber, $expiredAt, $paymentInfo);
        $listener = new PaymentListener();

        //act
        $listener->onReceivePaymentNumber($event);

        //assert
        $this->assertTrue($event->isPropagationStopped());
    }

    public function test_onReceivePaymentNumber_error_status()
    {
        //arrange
        $orderId = 'P0000';
        $payment = PaymentQuery::create()->findPk($orderId);
        $paymentData = $payment->toArray();
        $paymentnumber = '800-6521782765376';
        $expiredAt = new \DateTime();
        $paymentInfo = '132174jsdgfdyhugfds';
        $event = new ReceivePaymentnumberEvent($orderId, $paymentnumber, $expiredAt, $paymentInfo);
        $listener = new PaymentListener();

        //act
        $listener->onReceivePaymentNumber($event);

        //assert
        $this->assertTrue($event->isPropagationStopped());
        $payment->reload(true);
        $payment->fromArray($paymentData);
        $payment->save();
    }

    public function test_onReceivePaymentNumber_error_expired()
    {
        //arrange
        $orderId = 'P0001';
        $payment = PaymentQuery::create()->findPk($orderId);
        $paymentData = $payment->toArray();
        $paymentnumber = '800-6521782765376';
        $expiredAt = new \DateTime();
        $paymentInfo = '132174jsdgfdyhugfds';
        $event = new ReceivePaymentnumberEvent($orderId, $paymentnumber, $expiredAt, $paymentInfo);
        $listener = new PaymentListener();

        //act
        $listener->onReceivePaymentNumber($event);

        //assert
        $this->assertTrue($event->isPropagationStopped());
        $payment->reload(true);
        $payment->fromArray($paymentData);
        $payment->save();
    }

    public function test_onReceivePaymentNumber_valid_paymentnumber()
    {
        //arrange
        $orderId = 'P0001';
        $payment = PaymentQuery::create()->findPk($orderId);
        $paymentData = $payment->toArray();
        $paymentnumber = '800-6521782765376';
        $expiredAt = new \DateTime();
        $paymentInfo = '132174jsdgfdyhugfds';
        $event = new ReceivePaymentnumberEvent($orderId, $paymentnumber, $expiredAt, $paymentInfo);
        $listener = new PaymentListener();

        //act
        $listener->onReceivePaymentNumber($event);
        $payment = PaymentQuery::create()
            ->findPk($orderId);

        //assert
        $this->assertEquals($payment->getPaymentNumber(), $event->getPaymentnumber());
        $payment->reload(true);
        $payment->fromArray($paymentData);
        $payment->save();
    }

    public function test_onReceivePaymentNumber_valid_data()
    {
        //arrange
        $orderId = 'P0002';
        $payment = PaymentQuery::create()->findPk($orderId);
        $paymentData = $payment->toArray();
        $payment
            ->setPaymentStatus(PaymentPeer::PAYMENT_STATUS_INITIAL)
            ->setPaymentExpiredAt(time() + 86400)
            ->save();

        $paymentnumber = '800-6521782765376';
        $expiredAt = new \DateTime();
        $paymentInfo = '132174jsdgfdyhugfds';
        $event = new ReceivePaymentnumberEvent($orderId, $paymentnumber, $expiredAt, $paymentInfo);
        $listener = new PaymentListener();

        //act
        $listener->onReceivePaymentNumber($event);

        $payment = PaymentQuery::create()
            ->findPk($orderId);
        //assert
        $this->assertEquals($paymentnumber, $payment->getPaymentNumber());
        $this->assertEquals($expiredAt->format('U'), $payment->getPaymentExpiredAt('U'));
        $this->assertEquals($paymentInfo, $payment->getPaymentPaidInfo());
        $payment->reload(true);
        $payment->fromArray($paymentData);
        $payment->save();
    }

    public function test_onReceiveOrderEvent_error_order_id()
    {
        //arrange
        $orderId = 'P0000';
        $paymentType = 'TestPayment';
        $payment = PaymentQuery::create()->findPk($orderId);
        $paymentData = $payment->toArray();
        $paymentnumber = '800-6521782765376';
        $amount = 500;
        $status = true;
        $paymentInfo = '132174jsdgfdyhugfds';
        $event = new ReceiveOrderEvent($paymentnumber, $amount, $status, $paymentType, $paymentInfo);
        $listener = new PaymentListener();
        //act
        $listener->onReceiveOrderEvent($event);

        //assert
        $this->assertTrue($event->isPropagationStopped());
        $payment->reload(true);
        $payment->fromArray($paymentData);
        $payment->save();
    }

    public function test_onReceiveOrderEvent_error_status()
    {
        //arrange
        $orderId = 'P0000';
        $paymentType = 'TestPayment';
        $payment = PaymentQuery::create()->findPk($orderId);
        $paymentData = $payment->toArray();
        $paymentnumber = '800-6521782765376';
        $amount = 500;
        $status = false;
        $paymentInfo = '132174jsdgfdyhugfds';
        $event = new ReceiveOrderEvent($paymentnumber, $amount, $status, $paymentType, $paymentInfo);
        $listener = new PaymentListener();
        //act
        $listener->onReceiveOrderEvent($event);

        //assert
        $this->assertTrue($event->isPropagationStopped());
        $payment->reload(true);
        $payment->fromArray($paymentData);
        $payment->save();
    }

    public function test_onReceiveOrderEvent_error_expired()
    {
        //arrange
        $orderId = 'P0000';
        $paymentType = 'TestPayment';
        $payment = PaymentQuery::create()->findPk($orderId);
        $paymentData = $payment->toArray();
        $paymentnumber = '800-6521782765376';
        $amount = 500;
        $status = true;
        $paymentInfo = '132174jsdgfdyhugfds';
        $event = new ReceiveOrderEvent($paymentnumber, $amount, $status, $paymentType, $paymentInfo);
        $listener = new PaymentListener();
        //act
        $listener->onReceiveOrderEvent($event);

        //assert
        $this->assertTrue($event->isPropagationStopped());
        $payment->reload(true);
        $payment->fromArray($paymentData);
        $payment->save();
    }

    public function test_onReceiveOrderEvent_failed_payment()
    {
        //arrange
        $orderId = 'P0002';
        $paymentType = 'TestPayment';
        $payment = PaymentQuery::create()->findPk($orderId);
        $paymentData = $payment->toArray();
        $payment
            ->setPaymentStatus(PaymentPeer::PAYMENT_STATUS_INITIAL)
            ->setPaymentExpiredAt(time() + 86400)
            ->save();
        $amountForPay = 150;
        $paymentPaidAt = new \DateTime();
        $status = false;
        $paymentInfo = '800-6521782765376';
        $event = new ReceiveOrderEvent($orderId, $amountForPay, $status, $paymentType, $paymentInfo);
        $listener = new PaymentListener();
        
        //act
        $listener->onReceiveOrderEvent($event);
        $payment = PaymentQuery::create()
            ->findPk($orderId);

        //assert
        $this->assertEquals(PaymentPeer::PAYMENT_STATUS_FAILED, $payment->getPaymentStatus());
        $this->assertEquals($paymentInfo, $payment->getPaymentPaidInfo());
        $this->assertEquals($paymentPaidAt->format('U'), $payment->getPaymentPaidAt('U'));
        $this->assertEquals(null, $payment->getPaymentExpiredAt());
        $payment->reload(true);
        $payment->fromArray($paymentData);
        $payment->save();

    }

    public function test_onReceiveOrderEvent_valid_payment()
    {
        //arrange
        $paymentType = 'TestPayment';
        $orderId = 'P0002';
        $payment = PaymentQuery::create()->findPk($orderId);
        $paymentData = $payment->toArray();
        $payment
            ->setPaymentStatus(PaymentPeer::PAYMENT_STATUS_INITIAL)
            ->setPaymentExpiredAt(time() + 86400)
            ->save();
        $amountForPay = 150;
        $paymentPaidAt = new \DateTime();
        $status = true;
        $paymentInfo = '800-6521782765376';
        $event = new ReceiveOrderEvent($orderId, $amountForPay, $status, $paymentType, $paymentInfo);
        $listener = new PaymentListener();

        //act
        $listener->onReceiveOrderEvent($event);
        $payment = PaymentQuery::create()
            ->findPk($orderId);

        //assert
        $this->assertEquals(PaymentPeer::PAYMENT_STATUS_PAID, $payment->getPaymentStatus());
        $this->assertEquals($paymentInfo, $payment->getPaymentPaidInfo());
        $this->assertEquals($paymentPaidAt->format('U'), $payment->getPaymentPaidAt('U'));
        $this->assertEquals(null, $payment->getPaymentExpiredAt());
        $payment->reload(true);
        $payment->fromArray($paymentData);
        $payment->save();

    }

    public function test_onUpdatePaymentType_order_not_found()
    {
        //arrange
        $ordernumber = 'P0003';
        $paymentType = 'AllpayVacc';
        $amount = 100;
        $currency = 'CNY';
        $unit = 2;

        $payment = $this->getMockBuilder(Payment::class)
            ->setMethods(array('save', 'setPaymentType', 'setPaymentExpiredAt', 'setCurrency', 'setUnit'))
            ->getMock();
        $payment
            ->expects($this->never())
            ->method('setPaymentType');
        $payment
            ->expects($this->never())
            ->method('setPaymentExpiredAt');
        $payment
            ->expects($this->never())
            ->method('setCurrency');
        $payment
            ->expects($this->never())
            ->method('setUnit');
        $payment
            ->expects($this->never())
            ->method('save');

        $this->setObjectAttribute($payment, 'payment_type', $paymentType);

        $event = new UpdatePaymentTypeEvent($ordernumber, $paymentType, $amount, $currency, $unit);
        $listener = $this->getMockBuilder(PaymentListener::class)
            ->setMethods(array('findPayment'))
            ->getMock();
        $listener
            ->expects($this->once())
            ->method('findPayment')
            ->willReturnCallback(function($ordernumberForTest, $autoCreate) use($ordernumber){
                $this->assertEquals($ordernumber, $ordernumberForTest);
                $this->assertTrue($autoCreate);
                return null;
            });

        //act
        $listener->onUpdatePaymentType($event);

        //assert
        $this->assertFalse($event->isPropagationStopped());
    }

    public function test_onUpdatePaymentType_order_has_paymenttype_set()
    {
        //arrange
        $ordernumber = 'P0003';
        $paymentType = 'AllpayVacc';
        $amount = 100;
        $currency = 'CNY';
        $unit = 2;

        $payment = $this->getMockBuilder(Payment::class)
            ->setMethods(array('save', 'setPaymentType', 'setPaymentExpiredAt', 'setCurrency', 'setUnit'))
            ->getMock();
        $payment
            ->expects($this->never())
            ->method('setPaymentType');
        $payment
            ->expects($this->never())
            ->method('setPaymentExpiredAt');
        $payment
            ->expects($this->never())
            ->method('setCurrency');
        $payment
            ->expects($this->never())
            ->method('setUnit');
        $payment
            ->expects($this->never())
            ->method('save');

        $this->setObjectAttribute($payment, 'payment_type', $paymentType);

        $event = new UpdatePaymentTypeEvent($ordernumber, $paymentType, $amount, $currency, $unit);
        $listener = $this->getMockBuilder(PaymentListener::class)
            ->setMethods(array('findPayment'))
            ->getMock();
        $listener
            ->expects($this->once())
            ->method('findPayment')
            ->willReturnCallback(function($ordernumberForTest, $autoCreate) use($ordernumber, $payment){
                $this->assertEquals($ordernumber, $ordernumberForTest);
                $this->assertTrue($autoCreate);
                return $payment;
            });

        //act
        $listener->onUpdatePaymentType($event);

        //assert
        $this->assertFalse($event->isPropagationStopped());
    }

    public function test_onUpdatePaymentType_order()
    {
        //arrange
        $ordernumber = 'P0003';
        $paymentType = 'AllpayVacc';
        $amount = 100;
        $currency = 'CNY';
        $unit = 2;

        $payment = $this->getMockBuilder(Payment::class)
            ->setMethods(array('save', 'setPaymentType', 'setPaymentExpiredAt', 'setCurrency', 'setUnit'))
            ->getMock();
        $payment
            ->expects($this->once())
            ->method('setPaymentType')
            ->with($paymentType)
            ->willReturn($payment);
        $payment
            ->expects($this->once())
            ->method('setPaymentExpiredAt')
            ->willReturn($payment);

        $payment
            ->expects($this->once())
            ->method('setCurrency')
            ->with($currency)
            ->willReturn($payment);
        $payment
            ->expects($this->once())
            ->method('setUnit')
            ->with($unit)
            ->willReturn($payment);
        $payment
            ->expects($this->once())
            ->method('save')
            ->willReturn(1);

        $event = new UpdatePaymentTypeEvent($ordernumber, $paymentType, $amount, $currency, $unit);
        $listener = $this->getMockBuilder(PaymentListener::class)
            ->setMethods(array('findPayment'))
            ->getMock();
        $listener
            ->expects($this->once())
            ->method('findPayment')
            ->willReturnCallback(function($ordernumberForTest, $autoCreate) use($ordernumber, $payment){
                $this->assertEquals($ordernumber, $ordernumberForTest);
                $this->assertTrue($autoCreate);
                return $payment;
            });

        //act
        $listener->onUpdatePaymentType($event);

        //assert
    }

    public function test_findPayment_order_not_found()
    {
        //arrange
        $ordernumber = 'BAD_ORDERNUMBER';
        $listener = new PaymentListener();

        //act
        $result = $this->callObjectMethod($listener, 'findPayment', $ordernumber);

        //assert
        $this->assertNull($result);
    }

    public function test_findPayment_order_found_no_autocreate()
    {
        //arrange
        $ordernumber = 'P0003';
        $listener = new PaymentListener();

        //act
        $result = $this->callObjectMethod($listener, 'findPayment', $ordernumber);

        //assert
        $this->assertNull($result);
    }

    public function test_findPayment_order_found_with_autocreate()
    {
        //arrange
        $ordernumber = 'P0003';
        $listener = new PaymentListener();

        //act
        $result = $this->callObjectMethod($listener, 'findPayment', $ordernumber, true);

        //assert
        $this->assertInstanceOf(Payment::class, $result);
        $this->assertEquals($ordernumber, $result->getId());
    }

    public function test_findPayment_payment_found_with_no_autocreate()
    {
        //arrange
        $ordernumber = 'P0002';
        $listener = new PaymentListener();

        //act
        $result = $this->callObjectMethod($listener, 'findPayment', $ordernumber, false);

        //assert
        $this->assertInstanceOf(Payment::class, $result);
        $this->assertEquals($ordernumber, $result->getId());
        $this->assertFalse($result->isNew());
    }

    public function test_findPayment_payment_found_with_default_autocreate_false()
    {
        //arrange
        $ordernumber = 'P0002';
        $listener = new PaymentListener();

        //act
        $result = $this->callObjectMethod($listener, 'findPayment', $ordernumber);

        //assert
        $this->assertInstanceOf(Payment::class, $result);
        $this->assertEquals($ordernumber, $result->getId());
        $this->assertFalse($result->isNew());
    }

    public function test_findPayment_payment_found_with_autocreate_true()
    {
        //arrange
        $ordernumber = 'P0002';
        $listener = new PaymentListener();

        //act
        $result = $this->callObjectMethod($listener, 'findPayment', $ordernumber, true);

        //assert
        $this->assertInstanceOf(Payment::class, $result);
        $this->assertEquals($ordernumber, $result->getId());
        $this->assertFalse($result->isNew());
    }

}