<?php
namespace Widget\PaymentBundle\Test\Controller\API;

use Backend\BaseBundle\Tests\Fixture\BaseWebTestCase;
use Symfony\Component\HttpFoundation\Response;
use Widget\OrderBundle\Model\Order;
use Widget\OrderBundle\Token\OrderTokenRequest;
use Widget\PaymentBundle\Model\OrderReturnUrl;
use Widget\PaymentBundle\Model\OrderReturnUrlQuery;
use Widget\PaymentBundle\Payment\Payment;
use Widget\PaymentBundle\Processer\ProcesserResult;

class PaymentControllerTest extends BaseWebTestCase
{
    public function test_payAction_orderToken()
    {
        //arrange
        $okResponse = 'test ok';
        $ordernumber = 'A1234567';
        $amount  = 1000;
        $order = new Order();
        $order->setAmount($amount)->setAmountForPay($amount)->setId($ordernumber);
        $token = (string)$this->client->getContainer()->get('token_service')->sign(new OrderTokenRequest($order));
        $returnUrl = 'http://example.com/';
        $paymentType = 'AllpayVacc';
        $params = array(
            'return_url' => $returnUrl,
            'token' => $token,
            'payment_type' => $paymentType,
        );

        $paymentResult = $this->getMockBuilder(ProcesserResult::class)
            ->disableOriginalConstructor()
            ->setMethods(array('isProcessed', 'makeResponse'))
            ->getMock();
        $paymentResult
            ->expects($this->once())
            ->method('isProcessed')
            ->willReturn(true);
        $paymentResult
            ->expects($this->once())
            ->method('makeResponse')
            ->willReturn(new Response($okResponse));

        $payment = $this->getMockBuilder(Payment::class)
            ->disableOriginalConstructor()
            ->setMethods(array('processPayment'))
            ->getMock();
        $payment
            ->expects($this->once())
            ->method('processPayment')
            ->willReturnCallback(function($paymentTypeForTest, $ordernumberForTest, $amountForTest, $infoForTest) use($paymentResult, $paymentType, $ordernumber, $amount){
                $this->assertEquals($paymentType, $paymentTypeForTest);
                $this->assertEquals($ordernumber, $ordernumberForTest);
                $this->assertEquals($amount, $amountForTest);
                $this->assertEquals($ordernumber, $infoForTest);
                return $paymentResult;
            });

        $this->client->getContainer()->set('widget_payment.payment', $payment);

        //act
        $this->client->request('POST', $this->generateUrl('widget_payment_api_payment_pay'), $params);
        $response = $this->client->getResponse();
        $orderReturnUrl = OrderReturnUrlQuery::create()->findPk($ordernumber);

        //assert
        $this->assertTrue($response->isOk());
        $this->assertEquals($okResponse, $response->getContent());
        $this->assertNotNull($orderReturnUrl);
        $orderReturnUrl->delete();
    }

    public function test_payAction_paymentToken()
    {
        //arrange
        $okResponse = 'test ok';
        $ordernumber = 'A1234567';
        $amount  = 1000;
        $info = 'test payment info';
        $time = new \DateTime();
        $token = (string)$this->client->getContainer()->get('widget_order.payment.token')->createToken(
            $ordernumber,
            $amount,
            $info,
            $time
        );
        $returnUrl = 'http://example.com/';
        $paymentType = 'AllpayVacc';
        $params = array(
            'return_url' => $returnUrl,
            'token' => $token,
            'payment_type' => $paymentType,
        );

        $paymentResult = $this->getMockBuilder(ProcesserResult::class)
            ->disableOriginalConstructor()
            ->setMethods(array('isProcessed', 'makeResponse'))
            ->getMock();
        $paymentResult
            ->expects($this->once())
            ->method('isProcessed')
            ->willReturn(true);
        $paymentResult
            ->expects($this->once())
            ->method('makeResponse')
            ->willReturn(new Response($okResponse));

        $payment = $this->getMockBuilder(Payment::class)
            ->disableOriginalConstructor()
            ->setMethods(array('processPayment'))
            ->getMock();
        $payment
            ->expects($this->once())
            ->method('processPayment')
            ->willReturnCallback(function($paymentTypeForTest, $ordernumberForTest, $amountForTest, $infoForTest) use($paymentResult, $paymentType, $ordernumber, $amount, $info){
                $this->assertEquals($paymentType, $paymentTypeForTest);
                $this->assertEquals($ordernumber, $ordernumberForTest);
                $this->assertEquals($amount, $amountForTest);
                $this->assertEquals($info, $infoForTest);
                return $paymentResult;
            });

        $this->client->getContainer()->set('widget_payment.payment', $payment);

        //act
        $this->client->request('POST', $this->generateUrl('widget_payment_api_payment_pay'), $params);
        $response = $this->client->getResponse();
        $orderReturnUrl = OrderReturnUrlQuery::create()->findPk($ordernumber);

        //assert
        $this->assertTrue($response->isOk());
        $this->assertEquals($okResponse, $response->getContent());
        $this->assertNotNull($orderReturnUrl);
        $orderReturnUrl->delete();
    }

    public function test_payAction_bad_token()
    {
        //arrange
        $okResponse = 'test ok';
        $ordernumber = 'A1234567';
        $amount  = 1000;
        $info = 'test payment info';
        $time = new \DateTime();
        $token = 'bad token';
        $returnUrl = 'http://example.com/';
        $paymentType = 'AllpayVacc';
        $params = array(
            'return_url' => $returnUrl,
            'token' => $token,
            'payment_type' => $paymentType,
        );

        $paymentResult = $this->getMockBuilder(ProcesserResult::class)
            ->disableOriginalConstructor()
            ->setMethods(array('isProcessed', 'makeResponse'))
            ->getMock();
        $paymentResult
            ->expects($this->never())
            ->method('isProcessed')
            ->willReturn(true);
        $paymentResult
            ->expects($this->never())
            ->method('makeResponse')
            ->willReturn(new Response($okResponse));

        $payment = $this->getMockBuilder(Payment::class)
            ->disableOriginalConstructor()
            ->setMethods(array('processPayment'))
            ->getMock();
        $payment
            ->expects($this->never())
            ->method('processPayment')
            ->willReturnCallback(function($paymentTypeForTest, $ordernumberForTest, $amountForTest, $infoForTest) use($paymentResult, $paymentType, $ordernumber, $amount, $info){
                $this->assertEquals($paymentType, $paymentTypeForTest);
                $this->assertEquals($ordernumber, $ordernumberForTest);
                $this->assertEquals($amount, $amountForTest);
                $this->assertEquals($info, $infoForTest);
                return $paymentResult;
            });

        $this->client->getContainer()->set('widget_payment.payment', $payment);

        //act
        $this->client->request('POST', $this->generateUrl('widget_payment_api_payment_pay'), $params);
        $response = $this->client->getResponse();
        $orderReturnUrl = OrderReturnUrlQuery::create()->findPk($ordernumber);

        //assert
        $this->assertTrue($response->isNotFound());
        $this->assertNull($orderReturnUrl);
    }

    public function test_payAction_already_exists_paymentToken()
    {
        //arrange
        $okResponse = 'test ok';
        $ordernumber = 'A1234567';
        $amount  = 1000;
        $info = 'test payment info';
        $time = new \DateTime();
        $time->setTimestamp($time->getTimestamp() + 300);
        $token = (string)$this->client->getContainer()->get('widget_order.payment.token')->createToken(
            $ordernumber,
            $amount,
            $info,
            $time
        );
        $returnUrl = 'http://example.com/';
        $orderReturnUrl = new OrderReturnUrl();
        $orderReturnUrl
            ->setOrdernumber($ordernumber)
            ->setReturnUrl('http://bad.com/')
            ->save();

        $paymentType = 'AllpayVacc';
        $params = array(
            'return_url' => $returnUrl,
            'token' => $token,
            'payment_type' => $paymentType,
        );

        $paymentResult = $this->getMockBuilder(ProcesserResult::class)
            ->disableOriginalConstructor()
            ->setMethods(array('isProcessed', 'makeResponse'))
            ->getMock();
        $paymentResult
            ->expects($this->once())
            ->method('isProcessed')
            ->willReturn(true);
        $paymentResult
            ->expects($this->once())
            ->method('makeResponse')
            ->willReturn(new Response($okResponse));

        $payment = $this->getMockBuilder(Payment::class)
            ->disableOriginalConstructor()
            ->setMethods(array('processPayment'))
            ->getMock();
        $payment
            ->expects($this->once())
            ->method('processPayment')
            ->willReturnCallback(function($paymentTypeForTest, $ordernumberForTest, $amountForTest, $infoForTest) use($paymentResult, $paymentType, $ordernumber, $amount, $info){
                $this->assertEquals($paymentType, $paymentTypeForTest);
                $this->assertEquals($ordernumber, $ordernumberForTest);
                $this->assertEquals($amount, $amountForTest);
                $this->assertEquals($info, $infoForTest);
                return $paymentResult;
            });

        $this->client->getContainer()->set('widget_payment.payment', $payment);

        //act
        $this->client->request('POST', $this->generateUrl('widget_payment_api_payment_pay'), $params);
        $response = $this->client->getResponse();
        $orderReturnUrl->reload(true);

        //assert
        $this->assertTrue($response->isOk());
        $this->assertEquals($okResponse, $response->getContent());
        $this->assertEquals($returnUrl, $orderReturnUrl->getReturnUrl());
        $orderReturnUrl->delete();
    }

    public function test_payAction_already_exists_orderToken()
    {
        //arrange
        $okResponse = 'test ok';
        $ordernumber = 'A1234567';
        $amount  = 1000;
        $order = new Order();
        $order->setAmount($amount)->setAmountForPay($amount)->setId($ordernumber);
        $token = (string)$this->client->getContainer()->get('token_service')->sign(new OrderTokenRequest($order));
        $returnUrl = 'http://example.com/';
        $orderReturnUrl = new OrderReturnUrl();
        $orderReturnUrl
            ->setOrdernumber($ordernumber)
            ->setReturnUrl('http://bad.com/')
            ->save();

        $paymentType = 'AllpayVacc';
        $params = array(
            'return_url' => $returnUrl,
            'token' => $token,
            'payment_type' => $paymentType,
        );

        $paymentResult = $this->getMockBuilder(ProcesserResult::class)
            ->disableOriginalConstructor()
            ->setMethods(array('isProcessed', 'makeResponse'))
            ->getMock();
        $paymentResult
            ->expects($this->once())
            ->method('isProcessed')
            ->willReturn(true);
        $paymentResult
            ->expects($this->once())
            ->method('makeResponse')
            ->willReturn(new Response($okResponse));

        $payment = $this->getMockBuilder(Payment::class)
            ->disableOriginalConstructor()
            ->setMethods(array('processPayment'))
            ->getMock();
        $payment
            ->expects($this->once())
            ->method('processPayment')
            ->willReturnCallback(function($paymentTypeForTest, $ordernumberForTest, $amountForTest, $infoForTest) use($paymentResult, $paymentType, $ordernumber, $amount){
                $this->assertEquals($paymentType, $paymentTypeForTest);
                $this->assertEquals($ordernumber, $ordernumberForTest);
                $this->assertEquals($amount, $amountForTest);
                $this->assertEquals($ordernumber, $infoForTest);
                return $paymentResult;
            });

        $this->client->getContainer()->set('widget_payment.payment', $payment);

        //act
        $this->client->request('POST', $this->generateUrl('widget_payment_api_payment_pay'), $params);
        $response = $this->client->getResponse();
        $orderReturnUrl->reload(true);

        //assert
        $this->assertTrue($response->isOk());
        $this->assertEquals($okResponse, $response->getContent());
        $this->assertEquals($returnUrl, $orderReturnUrl->getReturnUrl());
        $orderReturnUrl->delete();
    }
}