<?php
namespace Widget\PaymentBundle\Payment;

use JMS\DiExtraBundle\Annotation as DI;
use Symfony\Component\EventDispatcher\EventDispatcherInterface;
use Symfony\Component\Routing\Router;
use Widget\PaymentBundle\Event\GetProcesserEvent;
use Widget\PaymentBundle\Event\ProcessPaymentEvent;
use Widget\PaymentBundle\Event\QueryAvailablePaymentEvent;
use Widget\PaymentBundle\Event\UpdatePaymentTypeEvent;
use Widget\PaymentBundle\Event\VerifyPaymentEvent;
use Widget\PaymentBundle\Processer;
use Widget\PaymentBundle\Exception;
use Widget\PaymentBundle\Processer\ProcesserResultInterface;

/**
 * @DI\Service("widget_payment.payment")
 */
class Payment
{
    const TokenType = 'order';

    /** @var EventDispatcherInterface  */
    protected $dispatcher;

    /** @var \Twig_Environment  */
    protected $twig;

    /**
     * @DI\InjectParams()
     */
    public function __construct(EventDispatcherInterface $eventDispatcher, \Twig_Environment $twig)
    {
        $this->dispatcher = $eventDispatcher;
        $this->twig = $twig;
    }

    /**
     * 取得金流處理
     * @param $paymentType
     * @return null|Processer\ProcesserBase
     */
    public function getProcesser($paymentType)
    {
        $event = new GetProcesserEvent($paymentType);
        $this->dispatcher->dispatch(GetProcesserEvent::EVENT_NAME, $event);
        return $event->getProcesser();
    }

    /**
     * 進行付款
     * @param $paymentType
     * @param $ordernumber
     * @param $amount
     * @param $paymentExtraInfo
     * @param $currency string 幣別
     * @param $unit int 貨幣單位 (對應小數位數例如 0.01元，表示 2)
     * @return ProcesserResultInterface
     */
    public function processPayment($paymentType, $ordernumber, $amount, $paymentExtraInfo, $currency, $unit)
    {
        $event = new ProcessPaymentEvent($paymentType, $ordernumber, $amount, $paymentExtraInfo, $currency, $unit);
        $event2 = new UpdatePaymentTypeEvent($ordernumber, $paymentType, $amount, $currency, $unit);
        $this->dispatcher->dispatch(UpdatePaymentTypeEvent::EVENT_NAME, $event2);
        $this->dispatcher->dispatch(ProcessPaymentEvent::EVENT_NAME, $event);
        $result = $event->getProcesserResult();
        $result->setTwig($this->twig);
        return $result;
    }

    /**
     * 查詢付款狀態
     * @param $paymentType
     * @param $ordernumber
     * @return VerifyPaymentEvent
     */
    public function verifyPayment($paymentType, $ordernumber)
    {
        $event = new VerifyPaymentEvent($paymentType, $ordernumber);
        $this->dispatcher->dispatch(VerifyPaymentEvent::EVENT_NAME, $event);
        return $event;
    }

    /**
     * 查詢可用付款方式
     * @param $amount
     * @return array 
     */
    public function queryAvailablePayment($amount)
    {
        $event = new QueryAvailablePaymentEvent($amount);
        $this->dispatcher->dispatch(QueryAvailablePaymentEvent::EVENT_NAME, $event);
        return $event->getAvailablePayment();
    }
}
