<?php
namespace Widget\PaymentBundle\EventListener;

use JMS\DiExtraBundle\Annotation as DI;
use Symfony\Component\EventDispatcher\EventDispatcherInterface;
use Widget\OrderBundle\Model\Order;
use Widget\OrderBundle\Model\OrderQuery;
use Widget\PaymentBundle\Event\PaymentErrorEvent;
use Widget\PaymentBundle\Event\ReceiveOrderEvent;
use Widget\PaymentBundle\Event\ReceivePaymentAfterCODEvent;
use Widget\PaymentBundle\Event\ReceivePaymentnumberEvent;
use Widget\PaymentBundle\Event\UpdatePaymentTypeEvent;
use Widget\PaymentBundle\Model\Payment;
use Widget\PaymentBundle\Model\PaymentPeer;
use Widget\PaymentBundle\Model\PaymentQuery;

/**
 * @DI\Service()
 */
class PaymentListener
{

    /** @var  EventDispatcherInterface */
    protected $eventDispatcher;

    /**
     * @DI\InjectParams()
     */
    public function injectEventDispatcher(EventDispatcherInterface $eventDispatcher)
    {
        $this->eventDispatcher = $eventDispatcher;
    }

    /**
     * @param $ordernumber
     * @return mixed|\Widget\PaymentBundle\Model\Payment|\Widget\PaymentBundle\Model\Payment[]
     */
    protected function findPayment($ordernumber, $autoCreate = false)
    {
        /** @var Order $order */
        $order = OrderQuery::create()
            ->leftJoinWithPayment()
            ->findPk($ordernumber);

        if(!$order){
            return null;
        }

        if(!$order->getPayment() && $autoCreate){
            $payment = new Payment();
            $payment->setOrder($order);
        }

        return $order->getPayment();
    }

    /**
     * 更新付款方式
     * @DI\Observe(UpdatePaymentTypeEvent::EVENT_NAME)
     */
    public function onUpdatePaymentType(UpdatePaymentTypeEvent $event)
    {
        $paymentType = $event->getPaymentType();

        $payment = $this->findPayment($event->getOrdernumber(), true);

        if(!$payment || $payment->getPaymentType() != null){
            return;
        }

        $payment
            ->setPaymentType($paymentType)
            ->setAmount($event->getAmount())
            ->setPaymentExpiredAt(time() + 30*60) //設定金流處理時間 30 分鐘
            ->setCurrency($event->getCurrency())
            ->setUnit($event->getUnit())
            ->save();
    }

    /**
     * 收到繳款代碼更新資料庫訂單狀態
     * @DI\Observe(ReceivePaymentnumberEvent::EVENT_NAME, priority = 1024)
     */
    public function onReceivePaymentNumber(ReceivePaymentnumberEvent $event)
    {
        $payment = $this->findPayment($event->getOrdernumber());
        
        if(!$payment){
            $event->stopPropagation();
            return;
        }
        if($payment->getPaymentStatus() != PaymentPeer::PAYMENT_STATUS_INITIAL){
            $event->stopPropagation();
            return;
        }
        if($payment->getPaymentExpiredAt('U') <= time() ){
            $event->stopPropagation();
            return;
        }

        $payment
            ->setPaymentNumber($event->getPaymentnumber())
            ->setPaymentStatus(PaymentPeer::PAYMENT_STATUS_PAYMENTNUMBER)
            ->setPaymentPaidInfo($event->getOriginInfo())
            ->setPaymentExpiredAt($event->getExpiredAt())
            ->save()
        ;
    }

    /**
     * 收到繳款代碼更新資料庫訂單狀態
     * @DI\Observe(ReceiveOrderEvent::EVENT_NAME, priority = 1024)
     */
    public function onReceiveOrderEvent(ReceiveOrderEvent $event)
    {
        $payment = $this->findPayment($event->getOrdernumber());
        
        if(!$payment){
            $event->stopPropagation();
            return;
        }

        if($payment->getPaymentStatus() == PaymentPeer::PAYMENT_STATUS_PAID 
            || $payment->getPaymentStatus() == PaymentPeer::PAYMENT_STATUS_FAILED
        ){
            $event->stopPropagation();
            return;
        }

        if($payment->getPaymentExpiredAt('U') <= time() ){
            $this->emitPaymentError($event->getOrdernumber(), 'expired', $event->getOriginInfo());
            $event->stopPropagation();
            return;
        }

        $payment
            ->setPaymentPaidInfo($event->getOriginInfo())
            ->setPaymentPaid($event->getStatus())
            ->setPaymentStatus($event->getStatus() ? PaymentPeer::PAYMENT_STATUS_PAID : PaymentPeer::PAYMENT_STATUS_FAILED)
            ->setPaymentPaidAt(time())
            ->setPaymentExpiredAt(null)
            ->save();
    }

    protected function emitPaymentError($ordernumber, $reason, $originInfo)
    {
        $event = new PaymentErrorEvent($ordernumber, $reason, $originInfo);
        $this->eventDispatcher->dispatch(PaymentErrorEvent::EVENT_NAME, $event);
    }
}