<?php
namespace Widget\PaymentBundle\Tests\Processer;

use Backend\BaseBundle\Tests\Fixture\BaseTestCase;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\HttpFoundation\RedirectResponse;
use Widget\PaymentBundle\Processer\ProcesserResult;

class ProcesserResultTest extends BaseTestCase
{
    public function test_setTwig()
    {
        //arrange
        $processerResult = new ProcesserResult();
        $twig = $this->getMockBuilder(\Twig_Environment::class)
            ->disableOriginalConstructor()
            ->setMethods()
            ->getMock();

        //act
        $processerResult->setTwig($twig);

        //assert
        $this->assertEquals($twig, $this->getObjectAttribute($processerResult, 'twig'));
    }

    public function isProcessed()
    {
        //arrange
        $processerResult = new ProcesserResult();
        $this->setObjectAttribute($processerResult, 'isProcessed', true);

        //act
        $result = $processerResult->isProcessed();

        //assert
        $this->assertTrue($result);
    }

    public function test_setForwardPost()
    {
        //arrange
        $processerResult = new ProcesserResult();
        $url = 'http://test.example.com/';
        $params = array_combine(range('a', 'f'), range('a', 'f'));

        //act
        $result = $processerResult->setForwardPost($url, $params);

        //assert
        $this->assertTrue($this->getObjectAttribute($processerResult, 'isProcessed'));
        $this->assertFalse($this->getObjectAttribute($processerResult, 'isRedirect'));
        $this->assertEquals($url, $this->getObjectAttribute($processerResult, 'target'));
        $this->assertEquals($params, $this->getObjectAttribute($processerResult, 'params'));
    }

    public function test_setForwardRedirect()
    {
        //arrange
        $processerResult = new ProcesserResult();
        $url = 'http://test.example.com/';
        $params = array_combine(range('a', 'f'), range('a', 'f'));

        //act
        $result = $processerResult->setForwardRedirect($url, $params);

        //assert
        $this->assertTrue($this->getObjectAttribute($processerResult, 'isProcessed'));
        $this->assertTrue($this->getObjectAttribute($processerResult, 'isRedirect'));
        $this->assertEquals($url, $this->getObjectAttribute($processerResult, 'target'));
        $this->assertEquals($params, $this->getObjectAttribute($processerResult, 'params'));
    }

    /**
     * @dataProvider dataProvider_makeRedirectResponse
     */
    public function test_makeRedirectResponse($resultUrl, $baseUrl, $params)
    {
        //arrange
        $processerResult = new ProcesserResult();

        //act
        $result = $this->callObjectMethod($processerResult, 'makeRedirectResponse', $baseUrl, $params);

        //assert
        $this->assertInstanceOf(RedirectResponse::class, $result);
        $this->assertEquals($resultUrl, $result->getTargetUrl());
    }

    public function dataProvider_makeRedirectResponse()
    {
        return array(
            array(
                'resultUrl' => 'http://a.b.c/?a=1&b=2&c=3',
                'baseUrl' => 'http://a.b.c/',
                'params' => array('a' => 1, 'b' => 2, 'c' => 3),
            ),
            array(
                'resultUrl' => 'http://a.b.c/?test=123&a=1&b=2&c=3',
                'baseUrl' => 'http://a.b.c/?test=123',
                'params' => array('a' => 1, 'b' => 2, 'c' => 3),
            ),
            array(
                'resultUrl' => 'http://a.b.c/?test=123&a=+123&b=%26test&c=3',
                'baseUrl' => 'http://a.b.c/?test=123',
                'params' => array('a' => ' 123', 'b' => '&test', 'c' => 3),
            ),
        );
    }

    public function test_makePostResponse()
    {
        //arrange
        $params = array('a' => 1, 'b' => 2, 'c' => 3);
        $url = 'http://example.com/';
        $twig = $this->getMockBuilder(\Twig_Environment::class)
            ->setMethods(array('render'))
            ->disableOriginalConstructor()
            ->getMock();
        $twig
            ->expects($this->once())
            ->method('render')
            ->willReturnCallback(function($template, $templateParams) use($url, $params){
                $this->assertEquals(array('url'=> $url, 'params'=> $params), $templateParams);
                $this->assertEquals('WidgetPaymentBundle:Front/Common:forwardPost.html.twig', $template);
                return 'test_content';
            });
        $processerResult = new ProcesserResult();
        $this->setObjectAttribute($processerResult, 'twig', $twig);

        //act
        $result = $this->callObjectMethod($processerResult, 'makePostResponse', $url, $params);

        //assert
        $this->assertInstanceOf(Response::class, $result);
        $this->assertEquals('test_content', $result->getContent());
    }

    public function test_makeResponse_isRedirect_true()
    {
        //arrange
        $target = 'http://example.com/';
        $params = array('a' => 1, 'b' => 2, 'c' => 3);
        $response = new RedirectResponse($target);
        $processerResult = $this->getMockBuilder(ProcesserResult::class)
            ->setMethods(array('makeRedirectResponse', 'makePostResponse'))
            ->getMock();
        $processerResult
            ->expects($this->once())
            ->method('makeRedirectResponse')
            ->willReturnCallback(function($thisTarget, $thisParams) use($response, $target, $params){
                $this->assertEquals($target, $thisTarget);
                $this->assertEquals($params, $thisParams);
                return $response;
            });
        $processerResult
            ->expects($this->never())
            ->method('makePostResponse')
            ->willReturn(null);
        $this->setObjectAttribute($processerResult, 'isRedirect', true);
        $this->setObjectAttribute($processerResult, 'target', $target);
        $this->setObjectAttribute($processerResult, 'params', $params);

        //act
        $result = $processerResult->makeResponse();

        //assert
        $this->assertEquals($response, $result);
    }

    public function test_makeResponse_isRedirect_false()
    {
        //arrange
        $target = 'http://example.com/';
        $params = array('a' => 1, 'b' => 2, 'c' => 3);
        $response = new Response($target);
        $processerResult = $this->getMockBuilder(ProcesserResult::class)
            ->setMethods(array('makeRedirectResponse', 'makePostResponse'))
            ->getMock();
        $processerResult
            ->expects($this->once())
            ->method('makePostResponse')
            ->willReturnCallback(function($thisTarget, $thisParams) use($response, $target, $params){
                $this->assertEquals($target, $thisTarget);
                $this->assertEquals($params, $thisParams);
                return $response;
            });
        $processerResult
            ->expects($this->never())
            ->method('makeRedirectResponse')
            ->willReturn(null);
        $this->setObjectAttribute($processerResult, 'isRedirect', false);
        $this->setObjectAttribute($processerResult, 'target', $target);
        $this->setObjectAttribute($processerResult, 'params', $params);

        //act
        $result = $processerResult->makeResponse();

        //assert
        $this->assertEquals($response, $result);
    }
}
