<?php
namespace Widget\Payment\SpgatewayBundle\Tests\Processer;

use Backend\BaseBundle\Tests\Fixture\BaseKernelTestCase;
use Symfony\Component\Routing\Router;
use Widget\Payment\SpgatewayBundle\Lib\AESCrypt;
use Widget\Payment\SpgatewayBundle\Lib\ShaHash;
use Widget\Payment\SpgatewayBundle\Processer\SpgatewayVaccProcesser;
use Widget\PaymentBundle\Event\GetProcesserEvent;
use Widget\PaymentBundle\Event\ProcessPaymentEvent;
use Widget\PaymentBundle\Event\QueryAvailablePaymentEvent;
use Widget\PaymentBundle\Processer\ProcesserResult;

class SpgatewayVaccProcesserTest extends BaseKernelTestCase
{
    public function test_inject()
    {
        //arrange

        //act
        $processer = $this->container->get('widget_payment.processer.spgateway_vacc');

        //assert
        $this->assertNotNull($this->getObjectAttribute($processer, 'config'));
        $this->assertInstanceOf(ShaHash::class, $this->getObjectAttribute($processer, 'shaHash'));
        $this->assertInstanceOf(AESCrypt::class, $this->getObjectAttribute($processer, 'crypt'));
    }

    public function test_paymentType()
    {
        //arrange
        $paymentType = 'SpgatewayVacc';

        //act
        $result = SpgatewayVaccProcesser::PROCESSER_NAME;

        //assert
        $this->assertEquals($paymentType, $result);
    }

    public function test_getProcesser_bad_payment_type()
    {
        //arrange
        $processer = $this->getMockBuilder(SpgatewayVaccProcesser::class)
            ->disableOriginalConstructor()
            ->setMethods()
            ->getMock();

        $event = $this->getMockBuilder(GetProcesserEvent::class)
            ->setMethods(array('getPaymentType', 'stopPropagation', 'setProcesser'))
            ->disableOriginalConstructor()
            ->getMock();

        $event
            ->expects($this->once())
            ->method('getPaymentType')
            ->willReturn('bad_payment')
            ;

        $event
            ->expects($this->never())
            ->method('stopPropagation')
        ;

        $event
            ->expects($this->never())
            ->method('setProcesser')
            ->with($processer)
        ;

        //act
        $processer->getProcesser($event);

        //assert
    }

    public function test_getProcesser()
    {
        //arrange
        $processer = $this->getMockBuilder(SpgatewayVaccProcesser::class)
            ->disableOriginalConstructor()
            ->setMethods()
            ->getMock();

        $event = $this->getMockBuilder(GetProcesserEvent::class)
            ->setMethods(array('getPaymentType', 'stopPropagation', 'setProcesser'))
            ->disableOriginalConstructor()
            ->getMock();

        $event
            ->expects($this->once())
            ->method('getPaymentType')
            ->willReturn(SpgatewayVaccProcesser::PROCESSER_NAME)
        ;

        $event
            ->expects($this->once())
            ->method('stopPropagation')
        ;

        $event
            ->expects($this->once())
            ->method('setProcesser')
            ->with($processer)
        ;

        //act
        $processer->getProcesser($event);

        //assert
    }

    public function test_process_null_config()
    {
        //arrange
        $config = array(
            'something'
        );

        $processer = $this->getMockBuilder(SpgatewayVaccProcesser::class)
            ->disableOriginalConstructor()
            ->setMethods()
            ->getMock();

        $processerResult = $this->getMockBuilder(ProcesserResult::class)
            ->disableOriginalConstructor()
            ->setMethods()
            ->getMock();

        $event = $this->getMockBuilder(ProcessPaymentEvent::class)
            ->setMethods(array('getPaymentType', 'stopPropagation', 'getProcesserResult'))
            ->disableOriginalConstructor()
            ->getMock();

        $event
            ->expects($this->never())
            ->method('getPaymentType')
            ->willReturn('bad_payment')
        ;

        $event
            ->expects($this->never())
            ->method('stopPropagation')
        ;

        $event
            ->expects($this->never())
            ->method('getProcesserResult')
            ->willReturn($processerResult)
        ;

        //act
        $processer->process($event);

        //assert
    }

    public function test_process_bad_payment()
    {
        //arrange
        $config = array(
            'something'
        );

        $processer = $this->getMockBuilder(SpgatewayVaccProcesser::class)
            ->disableOriginalConstructor()
            ->setMethods()
            ->getMock();

        $this->setObjectAttribute($processer, 'config', $config);

        $processerResult = $this->getMockBuilder(ProcesserResult::class)
            ->disableOriginalConstructor()
            ->setMethods()
            ->getMock();

        $event = $this->getMockBuilder(ProcessPaymentEvent::class)
            ->setMethods(array('getPaymentType', 'stopPropagation', 'getProcesserResult'))
            ->disableOriginalConstructor()
            ->getMock();

        $event
            ->expects($this->once())
            ->method('getPaymentType')
            ->willReturn('bad_payment')
        ;

        $event
            ->expects($this->never())
            ->method('stopPropagation')
        ;

        $event
            ->expects($this->never())
            ->method('getProcesserResult')
            ->willReturn($processerResult)
        ;

        //act
        $processer->process($event);

        //assert
    }

    public function test_process()
    {
        //arrange
        $config = array(
            'store_id' => '12345',
            'gateway_url' => 'some_url',
        );

        $params = array(
            'MerchantID' => '12345',
            'TradeInfo' => 'some_info',
            'TradeSha' => 'some_sha',
            'Version' => '1.4',
        );

        $ordernumber = 'test_ordernumber';
        $amount = 2000;
        $paymentExtraInfo = array('some_info');

        $processer = $this->getMockBuilder(SpgatewayVaccProcesser::class)
            ->disableOriginalConstructor()
            ->setMethods(array('createPaymentParams'))
            ->getMock();

        $processer
            ->expects($this->once())
            ->method('createPaymentParams')
            ->with($ordernumber, $amount, $paymentExtraInfo)
            ->willReturn($params);

        $this->setObjectAttribute($processer, 'config', $config);

        $processerResult = $this->getMockBuilder(ProcesserResult::class)
            ->disableOriginalConstructor()
            ->setMethods(array('setForwardPost'))
            ->getMock();

        $processerResult
            ->expects($this->once())
            ->method('setForwardPost')
            ->with($config['gateway_url'], $params)
            ;

        $event = $this->getMockBuilder(ProcessPaymentEvent::class)
            ->setMethods(array('getPaymentType', 'getOrdernumber', 'getAmount', 'getPaymentExtraInfo', 'stopPropagation', 'getProcesserResult'))
            ->disableOriginalConstructor()
            ->getMock();

        $event
            ->expects($this->once())
            ->method('getPaymentType')
            ->willReturn(SpgatewayVaccProcesser::PROCESSER_NAME)
        ;

        $event
            ->expects($this->once())
            ->method('getOrdernumber')
            ->willReturn($ordernumber)
        ;

        $event
            ->expects($this->once())
            ->method('getAmount')
            ->willReturn($amount)
        ;

        $event
            ->expects($this->once())
            ->method('getPaymentExtraInfo')
            ->willReturn($paymentExtraInfo)
        ;

        $event
            ->expects($this->once())
            ->method('stopPropagation')
        ;

        $event
            ->expects($this->once())
            ->method('getProcesserResult')
            ->willReturn($processerResult)
        ;

        //act
        $processer->process($event);

        //assert
    }

    public function test_onQueryAvailablePaymentEvent_等於10元()
    {
        //arrange
        $amount = 10;

        $processer = $this->getMockBuilder(SpgatewayVaccProcesser::class)
            ->disableOriginalConstructor()
            ->setMethods()
            ->getMock();

        $event = $this->getMockBuilder(QueryAvailablePaymentEvent::class)
            ->setMethods(array('getAmount', 'addAvailablePayment'))
            ->disableOriginalConstructor()
            ->getMock();

        $event
            ->expects($this->atLeastOnce())
            ->method('getAmount')
            ->willReturn($amount);

        $event
            ->expects($this->once())
            ->method('addAvailablePayment')
            ->with(SpgatewayVaccProcesser::PROCESSER_NAME);

        //act
        $processer->onQueryAvailablePaymentEvent($event);

        //assert
    }

    public function test_onQueryAvailablePaymentEvent_小於10元()
    {
        //arrange
        $amount = 5;

        $processer = $this->getMockBuilder(SpgatewayVaccProcesser::class)
            ->disableOriginalConstructor()
            ->setMethods()
            ->getMock();

        $event = $this->getMockBuilder(QueryAvailablePaymentEvent::class)
            ->setMethods(array('getAmount', 'addAvailablePayment'))
            ->disableOriginalConstructor()
            ->getMock();

        $event
            ->expects($this->atLeastOnce())
            ->method('getAmount')
            ->willReturn($amount);

        $event
            ->expects($this->never())
            ->method('addAvailablePayment')
            ->with(SpgatewayVaccProcesser::PROCESSER_NAME);

        //act
        $processer->onQueryAvailablePaymentEvent($event);

        //assert
    }

    public function test_createTradInfo()
    {
        //arrange
        $ordernumber = 'o1234567';
        $amount = 12345;
        $paymentExtraInfo = array(
            'some_info' => 'info'
        );
        $config = array(
            'store_id' => 'test_store_id',
            'expire_day' => 7,
        );
        $expireDate = new \DateTime();
        $expireDate->modify('+5days');
        $processer = $this->getMockBuilder(SpgatewayVaccProcesser::class)
            ->setMethods(array('getCurrentTime'))
            ->disableOriginalConstructor()
            ->getMock();
        $processer
            ->expects($this->once())
            ->method('getCurrentTime')
            ->willReturn(new \DateTime('2014-06-23 14:35:51'));
        $router = $this->container->get('router');
        $paymentInfoUrl = $router->generate('payment_spgateway_vacc_paymentinfo_receive', array(), Router::ABSOLUTE_URL);
        $notifyUrl = $router->generate('payment_spgateway_vacc_server_receive', array(), Router::ABSOLUTE_URL);
        $returnUrl = $router->generate('payment_spgateway_vacc_user_return', array('ordernumber' => $ordernumber), Router::ABSOLUTE_URL);
        $this->setObjectAttribute($processer, 'config', $config);
        $this->setObjectAttribute($processer, 'router', $router);

        //act
        $result = $this->callObjectMethod($processer, 'createTradInfo', $ordernumber, $amount, $paymentExtraInfo);

        //assert
        $this->assertLessThanOrEqual(2, time() - $result['TimeStamp']);
        unset($result['TimeStamp']);
        $this->assertEquals(array(
            'MerchantID' => $config['store_id'],
            'RespondType' => 'JSON',
            'Version' => '1.4',
            'MerchantOrderNo' => $ordernumber,
            'Amt' => $amount,
            'LoginType' => 0,
            'ItemDesc' => $ordernumber,
            'ExpireDate' => '20140629',
            'NotifyURL' => $notifyUrl,
            'CustomerURL' => $paymentInfoUrl,
            'ClientBackURL' => $returnUrl,
            'VACC' => 1,
        ), $result);
    }
}