<?php
namespace Widget\Payment\SpgatewayBundle\Tests\Controller\Front;

use Widget\Payment\SpgatewayBundle\Processer\SpgatewayBarcodeProcesser;
use Widget\PaymentBundle\Event\ReceiveOrderEvent;
use Widget\PaymentBundle\Event\ReceivePaymentnumberEvent;
use Widget\PaymentBundle\Event\ReturnUrlEvent;
use Widget\PaymentBundle\Payment\Payment;
use Widget\PaymentBundle\Tests\Controller\Front\PaymentFrontWebTestCase;

class SpgatewayBarcodeControllerTest extends PaymentFrontWebTestCase
{
    protected function makeMockPayment($paymentType, $processer)
    {
        $payment = $this->getMockBuilder(Payment::class)
            ->setMethods(array('getProcesser'))
            ->disableOriginalConstructor()
            ->getMock();
        $payment
            ->expects($this->once())
            ->method('getProcesser')
            ->willReturnCallback(function($paymentTypeForTest) use($paymentType, $processer){
                $this->assertEquals($paymentType, $paymentTypeForTest);
                return $processer;
            });
        return $payment;
    }

    public function test_barcodeReceiveOrderAction_bad_store_id()
    {
        //arrange
        $params = array(
          'MerchantID' =>  'test_store_id'
        );

        //act
        $this->client->request('POST', $this->generateUrl('payment_spgateway_barcode_server_receive'), $params);
        $response = $this->client->getResponse();

        //assert
        $this->assertTrue($response->isNotFound());
    }

    public function test_barcodeReceiveOrderAction_bad_verify()
    {
        //arrange
        $params = array(
            'MerchantID' => 'spgateway_barcode_store_id',
            'TradeInfo' => 'trad_info',
            'TradeSha' => 'trad_sha',
            'Version' => '1.4',
        );

        $processer = $this->getMockBuilder(SpgatewayBarcodeProcesser::class)
            ->setMethods(array('verify', 'emitReceiveOrder'))
            ->disableOriginalConstructor()
            ->getMock();

        $processer
            ->expects($this->never())
            ->method('verify')
            ->with($params['TradeInfo'], $params['TradeSha'])
            ->willReturn(false);

        $processer
            ->expects($this->never())
            ->method('emitReceiveOrder');
        $container = $this->client->getContainer();
        
        //$payment = $this->makeMockPayment('SpgatewayBarcode', $processer);
        //$container->set('widget_payment.payment', $payment);

        //act
        $this->client->request('POST', $this->generateUrl('payment_spgateway_barcode_server_receive'), $params);
        $response = $this->client->getResponse();

        //assert
        $this->assertTrue($response->isNotFound());
    }

    public function test_barcodeReceiveOrderAction_fail_status()
    {
        //arrange
        $eventFired = false;
        $container = $this->client->getContainer();
        $dispatcher = $container->get('event_dispatcher');

        $tradInfo = json_decode('{"Status":"MPG03009","Message":"\u6388\u6b0a\u6210\u529f","Result":{"MerchantID":"3430112","Amt":30,"TradeNo":"17012411541818970","MerchantOrderNo":"S_1485230044","RespondType":"JSON","IP":"59_124_92_194","EscrowBank":"KGI","PaymentType":"BARCODE","RespondCode":"00","Auth":"930637","Card6No":"400022","Card4No":"2222","TokenUseStatus":0,"InstFirst":30,"InstEach":0,"Inst":0,"ECI":"","PayTime":"2017-01-24_11:54:37"}}', true);

        $params = array(
            'MerchantID' => 'spgateway_barcode_store_id',
            'TradeInfo' => 'trad_info',
            'TradeSha' => 'trad_sha',
            'Version' => '1.4',
        );

        $processer = $this->getMockBuilder(SpgatewayBarcodeProcesser::class)
            ->setMethods(array('verify'))
            ->disableOriginalConstructor()
            ->getMock();
        $processer
            ->expects($this->once())
            ->method('verify')
            ->with($params['TradeInfo'], $params['TradeSha'])
            ->willReturn($tradInfo);
        
        $payment = $this->makeMockPayment('SpgatewayBarcode', $processer);
        $container->set('widget_payment.payment', $payment);
        $dispatcher->addListener(ReceiveOrderEvent::EVENT_NAME, function(ReceiveOrderEvent $event) use(&$eventFired, $tradInfo){
            $eventFired = true;
            $this->assertEquals($tradInfo['Result']['MerchantOrderNo'], $event->getOrdernumber());
            $this->assertEquals($tradInfo['Result']['Amt'], $event->getAmount());
            $this->assertEquals(false, $event->getStatus());
            $this->assertEquals($tradInfo, $event->getOriginInfo());
            $event->stopPropagation();
        }, 99999);
        $this->setObjectAttribute($processer, 'dispatcher', $dispatcher);

        //act
        $this->client->request('POST', $this->generateUrl('payment_spgateway_barcode_server_receive'), $params);
        $response = $this->client->getResponse();

        //assert
        $this->assertTrue($response->isOk());
        $this->assertTrue($eventFired);
    }

    public function test_barcodeReceiveOrderAction_ok()
    {
        //arrange
        $eventFired = false;
        $container = $this->client->getContainer();
        $dispatcher = $container->get('event_dispatcher');

        $tradInfo = json_decode('{"Status":"SUCCESS","Message":"\u6388\u6b0a\u6210\u529f","Result":{"MerchantID":"3430112","Amt":30,"TradeNo":"17012411541818970","MerchantOrderNo":"S_1485230044","RespondType":"JSON","IP":"59_124_92_194","EscrowBank":"KGI","PaymentType":"BARCODE","RespondCode":"00","Auth":"930637","Card6No":"400022","Card4No":"2222","TokenUseStatus":0,"InstFirst":30,"InstEach":0,"Inst":0,"ECI":"","PayTime":"2017-01-24_11:54:37"}}', true);

        $params = array(
            'MerchantID' => 'spgateway_barcode_store_id',
            'TradeInfo' => 'trad_info',
            'TradeSha' => 'trad_sha',
            'Version' => '1.4',
        );

        $processer = $this->getMockBuilder(SpgatewayBarcodeProcesser::class)
            ->setMethods(array('verify'))
            ->disableOriginalConstructor()
            ->getMock();
        $processer
            ->expects($this->once())
            ->method('verify')
            ->with($params['TradeInfo'], $params['TradeSha'])
            ->willReturn($tradInfo);

        $payment = $this->makeMockPayment('SpgatewayBarcode', $processer);
        $container->set('widget_payment.payment', $payment);
        $dispatcher->addListener(ReceiveOrderEvent::EVENT_NAME, function(ReceiveOrderEvent $event) use(&$eventFired, $tradInfo){
            $eventFired = true;
            $this->assertEquals($tradInfo['Result']['MerchantOrderNo'], $event->getOrdernumber());
            $this->assertEquals($tradInfo['Result']['Amt'], $event->getAmount());
            $this->assertEquals(true, $event->getStatus());
            $this->assertEquals($tradInfo, $event->getOriginInfo());
            $event->stopPropagation();
        }, 99999);
        $this->setObjectAttribute($processer, 'dispatcher', $dispatcher);

        //act
        $this->client->request('POST', $this->generateUrl('payment_spgateway_barcode_server_receive'), $params);
        $response = $this->client->getResponse();

        //assert
        $this->assertTrue($response->isOk());
        $this->assertTrue($eventFired);
    }

    public function test_barcodeUserReturnAction_ok()
    {
        //arrange
        $ordernumber = 'P1234567';
        $returnUrl = 'http://localhost:8888/';
        $container = $this->client->getContainer();
        $dispatcher = $container->get('event_dispatcher');

        $dispatcher->addListener(ReturnUrlEvent::EVENT_NAME, function(ReturnUrlEvent $event) use($returnUrl){
            $event->setUrl($returnUrl);
            $event->stopPropagation();
        }, 99999);

        //act
        $this->client->request('POST', $this->generateUrl('payment_spgateway_barcode_user_return', array('ordernumber' => $ordernumber)));
        $response = $this->client->getResponse();

        //assert
        $this->assertTrue($response->isRedirect($returnUrl));
    }

    public function test_barcodeUserReturnAction_not_ok()
    {
        //arrange
        $ordernumber = 'P1234567';
        $returnUrl = 'http://localhost:8888/';
        $container = $this->client->getContainer();
        $dispatcher = $container->get('event_dispatcher');

        $dispatcher->addListener(ReturnUrlEvent::EVENT_NAME, function(ReturnUrlEvent $event) {
            $event->stopPropagation();
        }, 99999);

        //act
        $this->client->request('POST', $this->generateUrl('payment_spgateway_barcode_user_return', array('ordernumber' => $ordernumber)));
        $response = $this->client->getResponse();

        //assert
        $this->assertTrue($response->isNotFound());
    }

    public function test_barcodeReceivePaymentInfoAction_bad_store_id()
    {
        //arrange
        $params = array(
            'MerchantID' => 'test_store_id',
        );

        //act
        $this->client->request('POST', $this->generateUrl('payment_spgateway_barcode_paymentinfo_receive'), $params);
        $response = $this->client->getResponse();

        //assert
        $this->assertTrue($response->isNotFound());
    }

    public function test_barcodeReceivePaymentInfoAction_bad_verify()
    {
        //arrange
        $params = array(
            'MerchantID' => 'spgateway_barcode_store_id',
            'TradeInfo' => 'trad_info',
            'TradeSha' => 'trad_sha',
            'Version' => '1.4',
        );

        $processer = $this->getMockBuilder(SpgatewayBarcodeProcesser::class)
            ->setMethods(array('verify', 'emitReceiveOrder'))
            ->disableOriginalConstructor()
            ->getMock();

        $processer
            ->expects($this->never())
            ->method('verify')
            ->with($params['TradeInfo'], $params['TradeSha'])
            ->willReturn(false);

        $processer
            ->expects($this->never())
            ->method('emitReceiveOrder');
        $container = $this->client->getContainer();

        //$payment = $this->makeMockPayment('SpgatewayBarcode', $processer);
        //$container->set('widget_payment.payment', $payment);

        //act
        $this->client->request('POST', $this->generateUrl('payment_spgateway_barcode_paymentinfo_receive'), $params);
        $response = $this->client->getResponse();

        //assert
        $this->assertTrue($response->isNotFound());
    }

    public function test_barcodeReceivePaymentInfoAction_fail_status_redirect_to()
    {
        //arrange
        $returnUrl = 'http://localhost:8888/';
        $eventFired = false;
        $container = $this->client->getContainer();
        $dispatcher = $container->get('event_dispatcher');

        $tradInfo = json_decode('{"Status":"MPG03009","Message":"\u53d6\u865f\u6210\u529f","Result":{"MerchantID":"3430112","Amt":40,"TradeNo":"17012412015573044","MerchantOrderNo":"S_1485230508","CheckCode":"5DDD45141A73460637D9B464A3F800997EB0C21D7483F354F3674474FB7D064A","PaymentType":"BARCODE","ExpireDate":"2017-01-31","ExpireTime":"23:59:59","CodeNo":"LCB70124004332"}}', true);

        $params = array(
            'MerchantID' => 'spgateway_barcode_store_id',
            'TradeInfo' => 'trad_info',
            'TradeSha' => 'trad_sha',
            'Version' => '1.4',
        );

        $processer = $this->getMockBuilder(SpgatewayBarcodeProcesser::class)
            ->setMethods(array('verify'))
            ->disableOriginalConstructor()
            ->getMock();
        $processer
            ->expects($this->once())
            ->method('verify')
            ->with($params['TradeInfo'], $params['TradeSha'])
            ->willReturn($tradInfo);

        $payment = $this->makeMockPayment('SpgatewayBarcode', $processer);
        $container->set('widget_payment.payment', $payment);
        $dispatcher->addListener(ReceivePaymentnumberEvent::EVENT_NAME, function(ReceivePaymentnumberEvent $event) use(&$eventFired, $tradInfo){
            $eventFired = true;
            $this->assertEquals($tradInfo['Result']['MerchantOrderNo'], $event->getOrdernumber());
            $this->assertNull($event->getPaymentnumber());
            $this->assertEquals($tradInfo, $event->getOriginInfo());
            $event->stopPropagation();
        }, 99999);

        $dispatcher->addListener(ReturnUrlEvent::EVENT_NAME, function(ReturnUrlEvent $event) use($returnUrl){
            $event->setUrl($returnUrl);
            $event->stopPropagation();
        }, 99999);

        $this->setObjectAttribute($processer, 'dispatcher', $dispatcher);

        //act
        $this->client->request('POST', $this->generateUrl('payment_spgateway_barcode_paymentinfo_receive'), $params);
        $response = $this->client->getResponse();

        //assert
        $this->assertTrue($response->isRedirect($returnUrl));
        $this->assertTrue($eventFired);
    }

    public function test_barcodeReceivePaymentInfoAction_fail_status_no_redirect()
    {
        //arrange
        $returnUrl = 'http://localhost:8888/';
        $eventFired = false;
        $container = $this->client->getContainer();
        $dispatcher = $container->get('event_dispatcher');

        $tradInfo = json_decode('{"Status":"MPG03009","Message":"\u53d6\u865f\u6210\u529f","Result":{"MerchantID":"3430112","Amt":40,"TradeNo":"17012412015573044","MerchantOrderNo":"S_1485230508","CheckCode":"5DDD45141A73460637D9B464A3F800997EB0C21D7483F354F3674474FB7D064A","PaymentType":"BARCODE","ExpireDate":"2017-01-31","ExpireTime":"23:59:59","CodeNo":"LCB70124004332"}}', true);

        $params = array(
            'MerchantID' => 'spgateway_barcode_store_id',
            'TradeInfo' => 'trad_info',
            'TradeSha' => 'trad_sha',
            'Version' => '1.4',
        );

        $processer = $this->getMockBuilder(SpgatewayBarcodeProcesser::class)
            ->setMethods(array('verify'))
            ->disableOriginalConstructor()
            ->getMock();
        $processer
            ->expects($this->once())
            ->method('verify')
            ->with($params['TradeInfo'], $params['TradeSha'])
            ->willReturn($tradInfo);

        $payment = $this->makeMockPayment('SpgatewayBarcode', $processer);
        $container->set('widget_payment.payment', $payment);
        $dispatcher->addListener(ReceivePaymentnumberEvent::EVENT_NAME, function(ReceivePaymentnumberEvent $event) use(&$eventFired, $tradInfo){
            $eventFired = true;
            $this->assertEquals($tradInfo['Result']['MerchantOrderNo'], $event->getOrdernumber());
            $this->assertNull($event->getPaymentnumber());
            $this->assertEquals($tradInfo, $event->getOriginInfo());
            $event->stopPropagation();
        }, 99999);

        $dispatcher->addListener(ReturnUrlEvent::EVENT_NAME, function(ReturnUrlEvent $event) use($returnUrl){
//            $event->setUrl($returnUrl);
            $event->stopPropagation();
        }, 99999);

        $this->setObjectAttribute($processer, 'dispatcher', $dispatcher);

        //act
        $this->client->request('POST', $this->generateUrl('payment_spgateway_barcode_paymentinfo_receive'), $params);
        $response = $this->client->getResponse();

        //assert
        $this->assertTrue($response->isNotFound());
        $this->assertTrue($eventFired);
    }

    public function test_barcodeReceivePaymentInfoAction_success_status_redirect_to()
    {
        //arrange
        $returnUrl = 'http://localhost:8888/';
        $eventFired = false;
        $container = $this->client->getContainer();
        $dispatcher = $container->get('event_dispatcher');

        $tradInfo = json_decode('{"Status":"SUCCESS","Message":"\u53d6\u865f\u6210\u529f","Result":{"MerchantID":"3430112","Amt":40,"TradeNo":"17012412015573044","MerchantOrderNo":"S_1485230508","CheckCode":"5DDD45141A73460637D9B464A3F800997EB0C21D7483F354F3674474FB7D064A","PaymentType":"BARCODE","ExpireDate":"2017-01-31","ExpireTime":"23:59:59","Barcode_1":"112345","Barcode_2":"212345","Barcode_3":"312345"}}', true);

        $params = array(
            'MerchantID' => 'spgateway_barcode_store_id',
            'TradeInfo' => 'trad_info',
            'TradeSha' => 'trad_sha',
            'Version' => '1.4',
        );

        $processer = $this->getMockBuilder(SpgatewayBarcodeProcesser::class)
            ->setMethods(array('verify'))
            ->disableOriginalConstructor()
            ->getMock();
        $processer
            ->expects($this->once())
            ->method('verify')
            ->with($params['TradeInfo'], $params['TradeSha'])
            ->willReturn($tradInfo);

        $payment = $this->makeMockPayment('SpgatewayBarcode', $processer);
        $container->set('widget_payment.payment', $payment);
        $dispatcher->addListener(ReceivePaymentnumberEvent::EVENT_NAME, function(ReceivePaymentnumberEvent $event) use(&$eventFired, $tradInfo){
            $eventFired = true;
            $this->assertEquals($tradInfo['Result']['MerchantOrderNo'], $event->getOrdernumber());
            $this->assertEquals("{$tradInfo['Result']['Barcode_1']} {$tradInfo['Result']['Barcode_2']} {$tradInfo['Result']['Barcode_3']}", $event->getPaymentnumber());
            $this->assertEquals($tradInfo, $event->getOriginInfo());
            $event->stopPropagation();
        }, 99999);

        $dispatcher->addListener(ReturnUrlEvent::EVENT_NAME, function(ReturnUrlEvent $event) use($returnUrl){
            $event->setUrl($returnUrl);
            $event->stopPropagation();
        }, 99999);

        $this->setObjectAttribute($processer, 'dispatcher', $dispatcher);

        //act
        $this->client->request('POST', $this->generateUrl('payment_spgateway_barcode_paymentinfo_receive'), $params);
        $response = $this->client->getResponse();

        //assert
        $this->assertTrue($response->isRedirect($returnUrl));
        $this->assertTrue($eventFired);
    }

    public function test_barcodeReceivePaymentInfoAction_success_status_no_redirect()
    {
        //arrange
        $returnUrl = 'http://localhost:8888/';
        $eventFired = false;
        $container = $this->client->getContainer();
        $dispatcher = $container->get('event_dispatcher');

        $tradInfo = json_decode('{"Status":"SUCCESS","Message":"\u53d6\u865f\u6210\u529f","Result":{"MerchantID":"3430112","Amt":40,"TradeNo":"17012412015573044","MerchantOrderNo":"S_1485230508","CheckCode":"5DDD45141A73460637D9B464A3F800997EB0C21D7483F354F3674474FB7D064A","PaymentType":"BARCODE","ExpireDate":"2017-01-31","ExpireTime":"23:59:59","Barcode_1":"112345","Barcode_2":"212345","Barcode_3":"312345"}}', true);

        $params = array(
            'MerchantID' => 'spgateway_barcode_store_id',
            'TradeInfo' => 'trad_info',
            'TradeSha' => 'trad_sha',
            'Version' => '1.4',
        );

        $processer = $this->getMockBuilder(SpgatewayBarcodeProcesser::class)
            ->setMethods(array('verify'))
            ->disableOriginalConstructor()
            ->getMock();
        $processer
            ->expects($this->once())
            ->method('verify')
            ->with($params['TradeInfo'], $params['TradeSha'])
            ->willReturn($tradInfo);

        $payment = $this->makeMockPayment('SpgatewayBarcode', $processer);
        $container->set('widget_payment.payment', $payment);
        $dispatcher->addListener(ReceivePaymentnumberEvent::EVENT_NAME, function(ReceivePaymentnumberEvent $event) use(&$eventFired, $tradInfo){
            $eventFired = true;
            $this->assertEquals($tradInfo['Result']['MerchantOrderNo'], $event->getOrdernumber());
            $this->assertEquals("{$tradInfo['Result']['Barcode_1']} {$tradInfo['Result']['Barcode_2']} {$tradInfo['Result']['Barcode_3']}", $event->getPaymentnumber());
            $this->assertEquals($tradInfo, $event->getOriginInfo());
            $event->stopPropagation();
        }, 99999);

        $dispatcher->addListener(ReturnUrlEvent::EVENT_NAME, function(ReturnUrlEvent $event) use($returnUrl){
//            $event->setUrl($returnUrl);
            $event->stopPropagation();
        }, 99999);

        $this->setObjectAttribute($processer, 'dispatcher', $dispatcher);

        //act
        $this->client->request('POST', $this->generateUrl('payment_spgateway_barcode_paymentinfo_receive'), $params);
        $response = $this->client->getResponse();

        //assert
        $this->assertTrue($response->isNotFound());
        $this->assertTrue($eventFired);
    }

    public function test_barcodePrintAction()
    {
        //arrange
        $ordernumber = 'A1234567';
        $url = 'http://example.com/';
        $processer = $this->getMockBuilder(SpgatewayBarcodeProcesser::class)
            ->setMethods(array('getBarCodePrintUrl'))
            ->getMock();
        $processer
            ->expects($this->once())
            ->method('getBarCodePrintUrl')
            ->with($ordernumber)
            ->willReturn($url);

        $this->client->getContainer()->set('widget_payment.processer.spgateway_barcode', $processer);

        //act
        $this->client->request('GET', $this->generateUrl('payment_spgateway_barcode_print', array('ordernumber' => $ordernumber)));
        $response = $this->client->getResponse();

        //assert
        $this->assertTrue($response->isRedirect($url));
    }
}
