<?php
namespace Widget\Payment\DigiflowBundle\Tests\Processer;

use Backend\BaseBundle\Tests\Fixture\BaseKernelTestCase;
use Widget\Payment\DigiflowBundle\Event\QuertPaymentResultEvent;
use Widget\Payment\DigiflowBundle\Lib\ShaHash;
use Widget\Payment\DigiflowBundle\Processer\DigiflowVaccProcesser;
use Widget\PaymentBundle\Event\GetProcesserEvent;
use Widget\PaymentBundle\Event\ProcessPaymentEvent;
use Widget\PaymentBundle\Event\QueryAvailablePaymentEvent;
use Widget\PaymentBundle\Event\ReceiveOrderEvent;
use Widget\PaymentBundle\Event\ReceivePaymentnumberEvent;
use Widget\PaymentBundle\Event\ReturnUrlEvent;
use Widget\PaymentBundle\Processer\ProcesserResult;

class DigiflowVaccProcesserTest extends BaseKernelTestCase
{
    public function test_inject()
    {
        //arrange
        //act
        $processer = $this->container->get('widget.payment.digiflow_bundle.processer.digiflow_vacc_processer');

        //assert
        $this->assertNotNull($this->getObjectAttribute($processer, 'config'));
        $this->assertInstanceOf(ShaHash::class, $this->getObjectAttribute($processer, 'shaHash'));
    }

    public function test_getPaymentExpiredAt()
    {
        //arrange
        $config = array(
            'expire_day' => 2,
        );

        $time = strtotime('2017-12-28 12:00:00');
        $processer = $this->getMockBuilder(DigiflowVaccProcesser::class)
            ->setMethods(array('getCurrentTime'))
            ->getMock();

        $processer
            ->expects($this->once())
            ->method('getCurrentTime')
            ->willReturn($time);

        $this->setObjectAttribute($processer, 'config', $config);

        //act
        $result = $this->callObjectMethod($processer, 'getPaymentExpiredAt');

        //assert
        $this->assertInstanceOf(\DateTime::class, $result);
        $this->assertEquals('2017-12-29 00:00:00', $result->format('Y-m-d 00:00:00'));
    }

    public function test_getProcesser()
    {
        //arrange
        $event = new GetProcesserEvent(DigiflowVaccProcesser::PROCESSER_NAME);
        $dispatcher = $this->container->get('event_dispatcher');

        //act
        $dispatcher->dispatch(GetProcesserEvent::EVENT_NAME, $event);

        //assert
        $this->assertTrue($event->isPropagationStopped());
        $this->assertInstanceOf(DigiflowVaccProcesser::class, $event->getProcesser());
    }

    public function test_process_invalid_payment()
    {
        //arrange
        $paymentType = 'BadVacc';
        $ordernumber = 'A12345';
        $amount = '1000';
        $paymentExtraInfo = array('a', 'b');
        $url = 'http://example.com/';

        $processer = $this->getMockBuilder(DigiflowVaccProcesser::class)
            ->setMethods(array('getPaymentParams'))
            ->getMock();
        $processer
            ->expects($this->never())
            ->method('getPaymentParams')
            ->with($ordernumber, $amount)
            ->willReturn($url);
        $event = new ProcessPaymentEvent($paymentType, $ordernumber,$amount, $paymentExtraInfo);

        //act
        $processer->process($event);

        //assert
    }

    public function test_process_no_config()
    {
        //arrange
        $paymentType = 'DigiflowVacc';
        $ordernumber = 'A12345';
        $amount = '1000';
        $paymentExtraInfo = array('a', 'b');
        $url = 'http://example.com/';

        $processer = $this->getMockBuilder(DigiflowVaccProcesser::class)
            ->setMethods(array('getPaymentParams'))
            ->getMock();
        $processer
            ->expects($this->never())
            ->method('getPaymentParams')
            ->with($ordernumber, $amount)
            ->willReturn($url);

        $event = new ProcessPaymentEvent($paymentType, $ordernumber,$amount, $paymentExtraInfo);

        //act
        $processer->process($event);

        //assert
    }

    public function test_process()
    {
        //arrange
        $paymentType = 'DigiflowVacc';
        $ordernumber = 'A12345';
        $amount = '1000';
        $paymentExtraInfo = array('a', 'b');
        $url = 'http://example.com/';
        $config = array('foo' => 'bar');
        $params = array(
            'a' => 'a',
            'b' => 'b',
            'c' => 'c',
        );

        $processer = $this->getMockBuilder(DigiflowVaccProcesser::class)
            ->setMethods(array('getPaymentParams', 'handlePayment'))
            ->getMock();
        $processer
            ->expects($this->once())
            ->method('getPaymentParams')
            ->with($ordernumber, $amount)
            ->willReturn($params);

        $processer
            ->expects($this->once())
            ->method('handlePayment')
            ->with($params)
            ->willReturn($url);

        $this->setObjectAttribute($processer, 'config', $config);

        $event = new ProcessPaymentEvent($paymentType, $ordernumber,$amount, $paymentExtraInfo);

        //act
        $processer->process($event);
        $response = $event->getProcesserResult()->makeResponse();

        //assert
        $this->assertTrue($event->isPropagationStopped());
        $this->assertTrue($response->isRedirect($url));
    }

    public function test_getPaymentParams()
    {
        //arrange
        $extData = 'some_ext_data';
        $time = strtotime('2017-12-29 12:00:00');
        $config = array(
            'merchant_id' => '12345',
            'terminal_id' => '54321',
            'expire_day' => '2',
            'key' => 'test_key',
        );
        $ordernumber = 'A1234567';
        $amount = 123;

        $params = array(
            'version' => '1.0',
            'merchant_id' => $config['merchant_id'],
            'terminal_id' => $config['terminal_id'],
            'order_no' => $ordernumber,
            'currency' => 'TWD',
            'order_desc' => $ordernumber,
            'order_amount' => $amount*100,
            'expiry_time' => '20171230000000',
            'payment_type' => '150',
            'timestamp' => $time*1000,
            'ext_data' => $extData,
        );

        $processer = $this->getMockBuilder(DigiflowVaccProcesser::class)
            ->setMethods(array('getCurrentTime', 'createExtData'))
            ->getMock();

        $processer
            ->expects($this->atLeastOnce())
            ->method('getCurrentTime')
            ->willReturn($time);
        $processer
            ->expects($this->once())
            ->method('createExtData')
            ->with($ordernumber)
            ->willReturn($extData);
        $this->setObjectAttribute($processer, 'config', $config);

        //act
        $result = $this->callObjectMethod($processer, 'getPaymentParams', $ordernumber, $amount);

        //assert
        $this->assertEquals($params, $result);
    }

    public function test_onReceivePaymentNumber_付款方式不是銀行虛擬帳號()
    {
        //arrange
        $paymentResult = array(
            'payment_type' => '111',
            'order_status' => 1,
            'order_no' => 'A12345',
            'bank' => '808',
            'account_no' => '123454321',
        );
        $event = new QuertPaymentResultEvent($paymentResult);
        $processer = $this->getMockBuilder(DigiflowVaccProcesser::class)
            ->setMethods(array('emitReceivePaymentnumber', 'getPaymentExpiredAt'))
            ->getMock();
        $processer
            ->expects($this->never())
            ->method('getPaymentExpiredAt')
            ->willReturn(new \DateTime('2017-12-29 12:00:00'));

        $processer
            ->expects($this->never())
            ->method('emitReceivePaymentnumber')
            ->willReturnCallback(function(ReceivePaymentnumberEvent $event) use($paymentResult){
                $this->assertEquals($paymentResult['order_no'], $event->getOrdernumber());
                $this->assertEquals("{$paymentResult['bank']}-{$paymentResult['account_no']}", $event->getPaymentnumber());
                $this->assertEquals(strtotime('2017-12-29 12:00:00'), $event->getExpiredAt()->format('U'));
                $this->assertEquals($paymentResult, $event->getOriginInfo());
            });

        //act
        $processer->onReceivePaymentNumber($event);

        //assert
        $this->assertFalse($event->isPropagationStopped());
    }

    public function test_onReceivePaymentNumber_付款方式是銀行虛擬帳號_但是已付款()
    {
        //arrange
        $paymentResult = array(
            'payment_type' => '115',
            'order_status' => 1,
            'order_no' => 'A12345',
            'bank' => '808',
            'account_no' => '123454321',
        );
        $event = new QuertPaymentResultEvent($paymentResult);
        $processer = $this->getMockBuilder(DigiflowVaccProcesser::class)
            ->setMethods(array('emitReceivePaymentnumber', 'getPaymentExpiredAt'))
            ->getMock();
        $processer
            ->expects($this->never())
            ->method('getPaymentExpiredAt')
            ->willReturn(new \DateTime('2017-12-29 12:00:00'));

        $processer
            ->expects($this->never())
            ->method('emitReceivePaymentnumber')
            ->willReturnCallback(function(ReceivePaymentnumberEvent $event) use($paymentResult){
                $this->assertEquals($paymentResult['order_no'], $event->getOrdernumber());
                $this->assertEquals("{$paymentResult['bank']}-{$paymentResult['account_no']}", $event->getPaymentnumber());
                $this->assertEquals(strtotime('2017-12-29 12:00:00'), $event->getExpiredAt()->format('U'));
                $this->assertEquals($paymentResult, $event->getOriginInfo());
            });

        //act
        $processer->onReceivePaymentNumber($event);

        //assert
        $this->assertFalse($event->isPropagationStopped());
    }

    public function test_onReceivePaymentNumber()
    {
        //arrange
        $paymentResult = array(
            'payment_type' => '150',
            'order_status' => 0,
            'order_no' => 'A12345',
            'bank' => '808',
            'account_no' => '123454321',
        );
        $event = new QuertPaymentResultEvent($paymentResult);
        $processer = $this->getMockBuilder(DigiflowVaccProcesser::class)
            ->setMethods(array('emitReceivePaymentnumber', 'getPaymentExpiredAt'))
            ->getMock();
        $processer
            ->expects($this->once())
            ->method('getPaymentExpiredAt')
            ->willReturn(new \DateTime('2017-12-29 12:00:00'));

        $processer
            ->expects($this->once())
            ->method('emitReceivePaymentnumber')
            ->willReturnCallback(function(ReceivePaymentnumberEvent $event) use($paymentResult){
                $this->assertEquals($paymentResult['order_no'], $event->getOrdernumber());
                $this->assertEquals("{$paymentResult['bank']}-{$paymentResult['account_no']}", $event->getPaymentnumber());
                $this->assertEquals(strtotime('2017-12-29 12:00:00'), $event->getExpiredAt()->format('U'));
                $this->assertEquals($paymentResult, $event->getOriginInfo());
            });

        //act
        $processer->onReceivePaymentNumber($event);

        //assert
        $this->assertTrue($event->isPropagationStopped());
    }


    public function test_onReceiveOrder_付款方式不是銀行虛擬帳號()
    {
        //arrange
        $paymentResult = array(
            'payment_type' => '114',
            'order_status' => 0,
            'order_no' => 'A12345',
            'order_amount' => 12300,
        );
        $event = new QuertPaymentResultEvent($paymentResult);
        $processer = $this->getMockBuilder(DigiflowVaccProcesser::class)
            ->setMethods(array('emitReceiveOrder'))
            ->getMock();
        $processer
            ->expects($this->never())
            ->method('emitReceiveOrder')
            ->willReturnCallback(function(ReceiveOrderEvent $event) use($paymentResult){
                $this->assertTrue($event->getStatus());
                $this->assertEquals($paymentResult['order_no'], $event->getOrdernumber());
                $this->assertEquals($paymentResult['order_amount']/100, $event->getAmount());
                $this->assertEquals(DigiflowVaccProcesser::PROCESSER_NAME, $event->getPaymentType());
                $this->assertEquals($paymentResult, $event->getOriginInfo());
            });

        //act
        $processer->onReceiveOrder($event);

        //assert
        $this->assertFalse($event->isPropagationStopped());
    }

    public function test_onReceiveOrder_付款方是銀行虛擬帳號_但是_尚未完成付款()
    {
        //arrange
        $paymentResult = array(
            'payment_type' => '150',
            'order_status' => 0,
            'order_no' => 'A12345',
            'order_amount' => 12300,
        );
        $event = new QuertPaymentResultEvent($paymentResult);
        $processer = $this->getMockBuilder(DigiflowVaccProcesser::class)
            ->setMethods(array('emitReceiveOrder'))
            ->getMock();
        $processer
            ->expects($this->never())
            ->method('emitReceiveOrder')
            ->willReturnCallback(function(ReceiveOrderEvent $event) use($paymentResult){
                $this->assertTrue($event->getStatus());
                $this->assertEquals($paymentResult['order_no'], $event->getOrdernumber());
                $this->assertEquals($paymentResult['order_amount']/100, $event->getAmount());
                $this->assertEquals(DigiflowVaccProcesser::PROCESSER_NAME, $event->getPaymentType());
                $this->assertEquals($paymentResult, $event->getOriginInfo());
            });

        //act
        $processer->onReceiveOrder($event);

        //assert
        $this->assertFalse($event->isPropagationStopped());
    }

    public function test_onReceiveOrder()
    {
        //arrange
        $paymentResult = array(
            'payment_type' => '150',
            'order_status' => 1,
            'order_no' => 'A12345',
            'order_amount' => 12300,
        );
        $event = new QuertPaymentResultEvent($paymentResult);
        $processer = $this->getMockBuilder(DigiflowVaccProcesser::class)
            ->setMethods(array('emitReceiveOrder'))
            ->getMock();
        $processer
            ->expects($this->once())
            ->method('emitReceiveOrder')
            ->willReturnCallback(function(ReceiveOrderEvent $event) use($paymentResult){
                $this->assertTrue($event->getStatus());
                $this->assertEquals($paymentResult['order_no'], $event->getOrdernumber());
                $this->assertEquals($paymentResult['order_amount']/100, $event->getAmount());
                $this->assertEquals(DigiflowVaccProcesser::PROCESSER_NAME, $event->getPaymentType());
                $this->assertEquals($paymentResult, $event->getOriginInfo());
            });

        //act
        $processer->onReceiveOrder($event);

        //assert
        $this->assertTrue($event->isPropagationStopped());
    }

    public function test_onQueryAvailablePaymentEvent_等於10元()
    {
        //arrange
        $amount = 10;

        $processer = $this->getMockBuilder(DigiflowVaccProcesser::class)
            ->disableOriginalConstructor()
            ->setMethods()
            ->getMock();

        $event = $this->getMockBuilder(QueryAvailablePaymentEvent::class)
            ->setMethods(array('getAmount', 'addAvailablePayment'))
            ->disableOriginalConstructor()
            ->getMock();

        $event
            ->expects($this->atLeastOnce())
            ->method('getAmount')
            ->willReturn($amount);

        $event
            ->expects($this->once())
            ->method('addAvailablePayment')
            ->with(DigiflowVaccProcesser::PROCESSER_NAME);

        //act
        $processer->onQueryAvailablePaymentEvent($event);

        //assert
    }

    public function test_onQueryAvailablePaymentEvent_小於10元()
    {
        //arrange
        $amount = 5;

        $processer = $this->getMockBuilder(DigiflowVaccProcesser::class)
            ->disableOriginalConstructor()
            ->setMethods()
            ->getMock();

        $event = $this->getMockBuilder(QueryAvailablePaymentEvent::class)
            ->setMethods(array('getAmount', 'addAvailablePayment'))
            ->disableOriginalConstructor()
            ->getMock();

        $event
            ->expects($this->atLeastOnce())
            ->method('getAmount')
            ->willReturn($amount);

        $event
            ->expects($this->never())
            ->method('addAvailablePayment')
            ->with(DigiflowVaccProcesser::PROCESSER_NAME);

        //act
        $processer->onQueryAvailablePaymentEvent($event);

        //assert
    }

}
