<?php
namespace Widget\Payment\DigiflowBundle\Tests\Processer;

use Backend\BaseBundle\Tests\Fixture\BaseKernelTestCase;
use Widget\Payment\DigiflowBundle\Event\QuertPaymentResultEvent;
use Widget\Payment\DigiflowBundle\Lib\ShaHash;
use Widget\Payment\DigiflowBundle\Processer\DigiflowCreditProcesser;
use Widget\PaymentBundle\Event\GetProcesserEvent;
use Widget\PaymentBundle\Event\ProcessPaymentEvent;
use Widget\PaymentBundle\Event\QueryAvailablePaymentEvent;
use Widget\PaymentBundle\Event\ReceiveOrderEvent;
use Widget\PaymentBundle\Event\ReturnUrlEvent;

class DigiflowCreditProcesserTest extends BaseKernelTestCase
{
    public function test_inject()
    {
        //arrange
        //act
        $processer = $this->container->get('widget.payment.digiflow_bundle.processer.digiflow_credit_processer');

        //assert
        $this->assertNotNull($this->getObjectAttribute($processer, 'config'));
        $this->assertInstanceOf(ShaHash::class, $this->getObjectAttribute($processer, 'shaHash'));
    }

    public function test_getProcesser()
    {
        //arrange
        $event = new GetProcesserEvent(DigiflowCreditProcesser::PROCESSER_NAME);
        $dispatcher = $this->container->get('event_dispatcher');

        //act
        $dispatcher->dispatch(GetProcesserEvent::EVENT_NAME, $event);

        //assert
        $this->assertTrue($event->isPropagationStopped());
        $this->assertInstanceOf(DigiflowCreditProcesser::class, $event->getProcesser());
    }

    public function test_process_invalid_payment()
    {
        //arrange
        $paymentType = 'BadCredit';
        $ordernumber = 'A12345';
        $amount = '1000';
        $paymentExtraInfo = array('a', 'b');
        $url = 'http://example.com/';

        $processer = $this->getMockBuilder(DigiflowCreditProcesser::class)
            ->setMethods(array('getPaymentParams'))
            ->getMock();
        $processer
            ->expects($this->never())
            ->method('getPaymentParams')
            ->with($ordernumber, $amount)
            ->willReturn($url);
        $event = new ProcessPaymentEvent($paymentType, $ordernumber,$amount, $paymentExtraInfo);

        //act
        $processer->process($event);

        //assert
    }

    public function test_process_no_config()
    {
        //arrange
        $paymentType = 'DigiflowCredit';
        $ordernumber = 'A12345';
        $amount = '1000';
        $paymentExtraInfo = array('a', 'b');
        $url = 'http://example.com/';

        $processer = $this->getMockBuilder(DigiflowCreditProcesser::class)
            ->setMethods(array('getPaymentParams'))
            ->getMock();
        $processer
            ->expects($this->never())
            ->method('getPaymentParams')
            ->with($ordernumber, $amount)
            ->willReturn($url);

        $event = new ProcessPaymentEvent($paymentType, $ordernumber,$amount, $paymentExtraInfo);

        //act
        $processer->process($event);

        //assert
    }

    public function test_process()
    {
        //arrange
        $paymentType = 'DigiflowCredit';
        $ordernumber = 'A12345';
        $amount = '1000';
        $paymentExtraInfo = array('a', 'b');
        $url = 'http://example.com/';
        $config = array('foo' => 'bar');
        $params = array(
            'a' => 'a',
            'b' => 'b',
            'c' => 'c',
        );

        $processer = $this->getMockBuilder(DigiflowCreditProcesser::class)
            ->setMethods(array('getPaymentParams', 'handlePayment'))
            ->getMock();
        $processer
            ->expects($this->once())
            ->method('getPaymentParams')
            ->with($ordernumber, $amount)
            ->willReturn($params);

        $processer
            ->expects($this->once())
            ->method('handlePayment')
            ->with($params)
            ->willReturn($url);

        $this->setObjectAttribute($processer, 'config', $config);

        $event = new ProcessPaymentEvent($paymentType, $ordernumber,$amount, $paymentExtraInfo);

        //act
        $processer->process($event);
        $response = $event->getProcesserResult()->makeResponse();

        //assert
        $this->assertTrue($event->isPropagationStopped());
        $this->assertTrue($response->isRedirect($url));
    }

    public function test_getPaymentParams()
    {
        //arrange
        $extData = 'some_ext_data';
        $time = strtotime('2017-12-28 12:00:00');
        $config = array(
            'merchant_id' => '12345',
            'terminal_id' => '54321',
            'key' => 'test_key',
        );
        $ordernumber = 'A1234567';
        $amount = 123;

        $params = array(
            'version' => '1.0',
            'merchant_id' => $config['merchant_id'],
            'terminal_id' => $config['terminal_id'],
            'order_no' => $ordernumber,
            'currency' => 'TWD',
            'order_desc' => $ordernumber,
            'order_amount' => $amount*100,
            'expiry_time' => '20171228123000',
            'payment_type' => '111',
            'timestamp' => $time*1000,
            'ext_data' => $extData,
        );

        $processer = $this->getMockBuilder(DigiflowCreditProcesser::class)
            ->setMethods(array('getCurrentTime', 'createExtData'))
            ->getMock();

        $processer
            ->expects($this->atLeastOnce())
            ->method('getCurrentTime')
            ->willReturn($time);
        $processer
            ->expects($this->once())
            ->method('createExtData')
            ->with($ordernumber)
            ->willReturn($extData);
        $this->setObjectAttribute($processer, 'config', $config);

        //act
        $result = $this->callObjectMethod($processer, 'getPaymentParams', $ordernumber, $amount);

        //assert
        $this->assertEquals($params, $result);
    }

    public function test_createPaymentParams()
    {
        //arrange
        $hash = 'test_hash';
        $params = array(
            'a' => 'a',
            'b' => 'b',
            'c' => 'c',
        );
        $processer = new DigiflowCreditProcesser();
        $shaHash = $this->getMockBuilder(ShaHash::class)
            ->setMethods(array('createHash'))
            ->disableOriginalConstructor()
            ->getMock();
        $shaHash
            ->expects($this->once())
            ->method('createHash')
            ->with($params)
            ->willReturn($hash);
        $this->setObjectAttribute($processer, 'shaHash', $shaHash);

        //act
        $result = $this->callObjectMethod($processer, 'createPaymentParams', $params);

        //assert
        $this->assertEquals(array(
            'a' => 'a',
            'b' => 'b',
            'c' => 'c',
            'sign' => $hash,
        ), $result);
    }

    public function test_handlePayment_fail()
    {
        //arrange
        $url = 'http://foo.bar/';

        $config = array(
            'gateway_url' => 'http://example.com/',
        );

        $params = array(
            'order_no' => 'A12345',
            'order_amount' => 123000,
            'c' => 'c',
        );

        $paramsWithSign = array(
            'a' => 'a',
            'b' => 'b',
            'c' => 'c',
            'sign' => 'test_hash',
        );

        $badResult = array(
            'return_code' => '000001',
            'return_msg' => 'bad',
        );

        $processer = $this->getMockBuilder(DigiflowCreditProcesser::class)
            ->setMethods(array('doPost', 'createPaymentParams', 'emitReceiveOrder', 'emitReturnUrl'))
            ->getMock();
        $processer
            ->expects($this->once())
            ->method('createPaymentParams')
            ->with($params)
            ->willReturn($paramsWithSign);
        $processer
            ->expects($this->once())
            ->method('doPost')
            ->with("{$config['gateway_url']}universal/order", $paramsWithSign)
            ->willReturn(json_encode($badResult));

        $processer
            ->expects($this->once())
            ->method('emitReceiveOrder')
            ->willReturnCallback(function(ReceiveOrderEvent $event) use($params, $badResult){
                $this->assertEquals($params['order_no'], $event->getOrdernumber());
                $this->assertEquals($params['order_amount']/100, $event->getAmount());
                $this->assertFalse($event->getStatus());
                $this->assertEquals($badResult, $event->getOriginInfo());
            });
        $processer
            ->expects($this->once())
            ->method('emitReturnUrl')
            ->willReturnCallback(function(ReturnUrlEvent $event) use($params, $url){
                $this->assertEquals($params['order_no'], $event->getOrdernumber());
                $event->setUrl($url);
            });

        $this->setObjectAttribute($processer, 'config', $config);

        //act
        $result = $this->callObjectMethod($processer, 'handlePayment', $params);

        //assert
        $this->assertEquals($url, $result);
    }

    public function test_handlePayment()
    {
        //arrange
        $url = 'http://foo.bar/';

        $config = array(
            'gateway_url' => 'http://example.com/',
        );

        $params = array(
            'order_no' => 'A12345',
            'order_amount' => 123000,
            'c' => 'c',
        );

        $paramsWithSign = array(
            'a' => 'a',
            'b' => 'b',
            'c' => 'c',
            'sign' => 'test_hash',
        );

        $badResult = array(
            'return_code' => '000000',
            'return_msg' => 'ok',
            'payment_url' => $url,
        );

        $processer = $this->getMockBuilder(DigiflowCreditProcesser::class)
            ->setMethods(array('doPost', 'createPaymentParams', 'emitReceiveOrder', 'emitReturnUrl'))
            ->getMock();
        $processer
            ->expects($this->once())
            ->method('createPaymentParams')
            ->with($params)
            ->willReturn($paramsWithSign);
        $processer
            ->expects($this->once())
            ->method('doPost')
            ->with("{$config['gateway_url']}universal/order", $paramsWithSign)
            ->willReturn(json_encode($badResult));

        $processer
            ->expects($this->never())
            ->method('emitReceiveOrder')
            ->willReturnCallback(function(ReceiveOrderEvent $event) use($params, $badResult){
                $this->assertEquals($params['order_no'], $event->getOrdernumber());
                $this->assertEquals($params['order_amount']/100, $event->getAmount());
                $this->assertFalse($event->getStatus());
                $this->assertEquals($badResult, $event->getOriginInfo());
            });
        $processer
            ->expects($this->never())
            ->method('emitReturnUrl')
            ->willReturnCallback(function(ReturnUrlEvent $event) use($params, $url){
                $this->assertEquals($params['order_no'], $event->getOrdernumber());
                $event->setUrl($url);
            });

        $this->setObjectAttribute($processer, 'config', $config);

        //act
        $result = $this->callObjectMethod($processer, 'handlePayment', $params);

        //assert
        $this->assertEquals($url, $result);
    }

    public function test_verify_bad_ext_data()
    {
        //arrange
        $ordernumber = 'A12345';
        $extData = 'some_ext_data';
        $hash = 'test_hash';
        $time = strtotime('2017-12-28 12:00:00');
        $config = array(
            'merchant_id' => '12345',
            'terminal_id' => '54321',
            'gateway_url' => 'http://foo.bat/',
            'key' => 'test_key',
        );
        $params = array(
            'version' => '1.0',
            'merchant_id' => $config['merchant_id'],
            'terminal_id' => $config['terminal_id'],
            'order_no' => $ordernumber,
            'timestamp' => $time*1000,
        );
        $paramsWithSign = array(
            'version' => '1.0',
            'merchant_id' => $config['merchant_id'],
            'terminal_id' => $config['terminal_id'],
            'order_no' => $ordernumber,
            'timestamp' => $time*1000,
            'sign' => $hash,
        );

        $resultData = array(
            'return_code' => '000000',
            'return_msg' => 'ok',
        );

        $processer = $this->getMockBuilder(DigiflowCreditProcesser::class)
            ->setMethods(array('doPost', 'createPaymentParams', 'createExtData', 'getCurrentTime'))
            ->getMock();

        $processer
            ->expects($this->never())
            ->method('getCurrentTime')
            ->willReturn($time);

        $processer
            ->expects($this->once())
            ->method('createExtData')
            ->with($ordernumber)
            ->willReturn('bad_ext_data');

        $processer
            ->expects($this->never())
            ->method('createPaymentParams')
            ->with($params)
            ->willReturn($paramsWithSign);

        $processer
            ->expects($this->never())
            ->method('doPost')
            ->with("{$config['gateway_url']}universal/query", $paramsWithSign)
            ->willReturn(json_encode($resultData));

        $this->setObjectAttribute($processer, 'config', $config);

        //act
        $result = $processer->verify($ordernumber, $extData);

        //assert
        $this->assertFalse($result);
    }

    public function test_verify_request_fail()
    {
        //arrange
        $ordernumber = 'A12345';
        $extData = 'some_ext_data';
        $hash = 'test_hash';
        $time = strtotime('2017-12-28 12:00:00');
        $config = array(
            'merchant_id' => '12345',
            'terminal_id' => '54321',
            'gateway_url' => 'http://foo.bat/',
            'key' => 'test_key',
        );
        $params = array(
            'version' => '1.0',
            'merchant_id' => $config['merchant_id'],
            'terminal_id' => $config['terminal_id'],
            'order_no' => $ordernumber,
            'timestamp' => $time*1000,
        );
        $paramsWithSign = array(
            'version' => '1.0',
            'merchant_id' => $config['merchant_id'],
            'terminal_id' => $config['terminal_id'],
            'order_no' => $ordernumber,
            'timestamp' => $time*1000,
            'sign' => $hash,
        );

        $resultData = array(
            'return_code' => '000001',
            'return_msg' => 'bad',
        );
        $processer = $this->getMockBuilder(DigiflowCreditProcesser::class)
            ->setMethods(array('doPost', 'createPaymentParams', 'createExtData', 'getCurrentTime'))
            ->getMock();

        $processer
            ->expects($this->once())
            ->method('getCurrentTime')
            ->willReturn($time);

        $processer
            ->expects($this->once())
            ->method('createExtData')
            ->with($ordernumber)
            ->willReturn($extData);

        $processer
            ->expects($this->once())
            ->method('createPaymentParams')
            ->with($params)
            ->willReturn($paramsWithSign);

        $processer
            ->expects($this->once())
            ->method('doPost')
            ->with("{$config['gateway_url']}universal/query", $paramsWithSign)
            ->willReturn(json_encode($resultData));

        $this->setObjectAttribute($processer, 'config', $config);

        //act
        $result = $processer->verify($ordernumber, $extData);

        //assert
        $this->assertFalse($result);
    }

    public function test_verify()
    {
        //arrange
        $ordernumber = 'A12345';
        $extData = 'some_ext_data';
        $hash = 'test_hash';
        $time = strtotime('2017-12-28 12:00:00');
        $config = array(
            'merchant_id' => '12345',
            'terminal_id' => '54321',
            'gateway_url' => 'http://foo.bat/',
            'key' => 'test_key',
        );
        $params = array(
            'version' => '1.0',
            'merchant_id' => $config['merchant_id'],
            'terminal_id' => $config['terminal_id'],
            'order_no' => $ordernumber,
            'timestamp' => $time*1000,
        );
        $paramsWithSign = array(
            'version' => '1.0',
            'merchant_id' => $config['merchant_id'],
            'terminal_id' => $config['terminal_id'],
            'order_no' => $ordernumber,
            'timestamp' => $time*1000,
            'sign' => $hash,
        );

        $resultData = array(
            'return_code' => '000000',
            'return_msg' => 'ok',
        );
        $processer = $this->getMockBuilder(DigiflowCreditProcesser::class)
            ->setMethods(array('doPost', 'createPaymentParams', 'createExtData', 'getCurrentTime'))
            ->getMock();

        $processer
            ->expects($this->once())
            ->method('getCurrentTime')
            ->willReturn($time);

        $processer
            ->expects($this->once())
            ->method('createExtData')
            ->with($ordernumber)
            ->willReturn($extData);

        $processer
            ->expects($this->once())
            ->method('createPaymentParams')
            ->with($params)
            ->willReturn($paramsWithSign);

        $processer
            ->expects($this->once())
            ->method('doPost')
            ->with("{$config['gateway_url']}universal/query", $paramsWithSign)
            ->willReturn(json_encode($resultData));

        $this->setObjectAttribute($processer, 'config', $config);

        //act
        $result = $processer->verify($ordernumber, $extData);

        //assert
        $this->assertEquals($resultData, $result);
    }

    public function test_onQueryPayment_付款方式不是信用卡()
    {
        //arrange
        $paymentResult = array(
            'payment_type' => '114',
            'order_status' => 0,
            'order_no' => 'A12345',
            'order_amount' => 12300,
        );
        $event = new QuertPaymentResultEvent($paymentResult);
        $processer = $this->getMockBuilder(DigiflowCreditProcesser::class)
            ->setMethods(array('emitReceiveOrder'))
            ->getMock();
        $processer
            ->expects($this->never())
            ->method('emitReceiveOrder')
            ->willReturnCallback(function(ReceiveOrderEvent $event) use($paymentResult){
                $this->assertTrue($event->getStatus());
                $this->assertEquals($paymentResult['order_no'], $event->getOrdernumber());
                $this->assertEquals($paymentResult['order_amount']/100, $event->getAmount());
                $this->assertEquals(DigiflowCreditProcesser::PROCESSER_NAME, $event->getPaymentType());
                $this->assertEquals($paymentResult, $event->getOriginInfo());
            });

        //act
        $processer->onQueryPayment($event);

        //assert
        $this->assertFalse($event->isPropagationStopped());
    }

    public function test_onQueryPayment_付款方是信用卡_但是_尚未完成付款()
    {
        //arrange
        $paymentResult = array(
            'payment_type' => '111',
            'order_status' => 0,
            'order_no' => 'A12345',
            'order_amount' => 12300,
        );
        $event = new QuertPaymentResultEvent($paymentResult);
        $processer = $this->getMockBuilder(DigiflowCreditProcesser::class)
            ->setMethods(array('emitReceiveOrder'))
            ->getMock();
        $processer
            ->expects($this->never())
            ->method('emitReceiveOrder')
            ->willReturnCallback(function(ReceiveOrderEvent $event) use($paymentResult){
                $this->assertTrue($event->getStatus());
                $this->assertEquals($paymentResult['order_no'], $event->getOrdernumber());
                $this->assertEquals($paymentResult['order_amount']/100, $event->getAmount());
                $this->assertEquals(DigiflowCreditProcesser::PROCESSER_NAME, $event->getPaymentType());
                $this->assertEquals($paymentResult, $event->getOriginInfo());
            });

        //act
        $processer->onQueryPayment($event);

        //assert
        $this->assertFalse($event->isPropagationStopped());
    }

    public function test_onQueryPayment()
    {
        //arrange
        $paymentResult = array(
            'payment_type' => '111',
            'order_status' => 1,
            'order_no' => 'A12345',
            'order_amount' => 12300,
        );
        $event = new QuertPaymentResultEvent($paymentResult);
        $processer = $this->getMockBuilder(DigiflowCreditProcesser::class)
            ->setMethods(array('emitReceiveOrder'))
            ->getMock();
        $processer
            ->expects($this->once())
            ->method('emitReceiveOrder')
            ->willReturnCallback(function(ReceiveOrderEvent $event) use($paymentResult){
                $this->assertTrue($event->getStatus());
                $this->assertEquals($paymentResult['order_no'], $event->getOrdernumber());
                $this->assertEquals($paymentResult['order_amount']/100, $event->getAmount());
                $this->assertEquals(DigiflowCreditProcesser::PROCESSER_NAME, $event->getPaymentType());
                $this->assertEquals($paymentResult, $event->getOriginInfo());
            });

        //act
        $processer->onQueryPayment($event);

        //assert
        $this->assertTrue($event->isPropagationStopped());
    }
    
    public function test_onQueryAvailablePaymentEvent_等於10元()
    {
        //arrange
        $amount = 10;

        $processer = $this->getMockBuilder(DigiflowCreditProcesser::class)
            ->disableOriginalConstructor()
            ->setMethods()
            ->getMock();

        $event = $this->getMockBuilder(QueryAvailablePaymentEvent::class)
            ->setMethods(array('getAmount', 'addAvailablePayment'))
            ->disableOriginalConstructor()
            ->getMock();

        $event
            ->expects($this->atLeastOnce())
            ->method('getAmount')
            ->willReturn($amount);

        $event
            ->expects($this->once())
            ->method('addAvailablePayment')
            ->with(DigiflowCreditProcesser::PROCESSER_NAME);

        //act
        $processer->onQueryAvailablePaymentEvent($event);

        //assert
    }

    public function test_onQueryAvailablePaymentEvent_小於10元()
    {
        //arrange
        $amount = 5;

        $processer = $this->getMockBuilder(DigiflowCreditProcesser::class)
            ->disableOriginalConstructor()
            ->setMethods()
            ->getMock();

        $event = $this->getMockBuilder(QueryAvailablePaymentEvent::class)
            ->setMethods(array('getAmount', 'addAvailablePayment'))
            ->disableOriginalConstructor()
            ->getMock();

        $event
            ->expects($this->atLeastOnce())
            ->method('getAmount')
            ->willReturn($amount);

        $event
            ->expects($this->never())
            ->method('addAvailablePayment')
            ->with(DigiflowCreditProcesser::PROCESSER_NAME);

        //act
        $processer->onQueryAvailablePaymentEvent($event);

        //assert
    }

}
