<?php
namespace Widget\Payment\CathayBundle\Tests\Controller\Front;

use Widget\PaymentBundle\Event\ReceiveOrderEvent;
use Widget\PaymentBundle\Event\ReturnUrlEvent;
use Widget\PaymentBundle\Event\VerifyAmountEvent;
use Widget\PaymentBundle\Tests\Controller\Front\PaymentFrontWebTestCase;

class CathayControllerTest extends PaymentFrontWebTestCase
{
    public function test_creditConfirmAction_bad_store_id()
    {
        //arrange
        
        $container = $this->client->getContainer();
        $ordernumber = 'o12345';
        $amount = 123;
        $storeid = 'test_store_id';
        $container
            ->get('event_dispatcher')
            ->addListener(VerifyAmountEvent::EVENT_NAME, function(VerifyAmountEvent $event){
                $this->fail('不應該觸發 Event 事件');
            });

        //act
        $this->client->request('GET', $this->generateUrl('payment_cathay_credit_confirm', array(
            'ordernumber' => $ordernumber,
            'amount' => $amount,
            'storeid' => $storeid,
        )));
        $response = $this->client->getResponse();

        //assert
        $this->assertTrue($response->isNotFound());
    }

    public function test_creditConfirmAction_event_emit_invalid()
    {
        //arrange
        
        $eventFired = false;
        $container = $this->client->getContainer();
        $ordernumber = 'o12345';
        $amount = 123;
        $storeid = 'cathay_credit_store_id';
        $container
            ->get('event_dispatcher')
            ->addListener(VerifyAmountEvent::EVENT_NAME, function(VerifyAmountEvent $event) use(&$eventFired, $ordernumber, $amount){
                $this->assertEquals($ordernumber, $event->getOrdernumber());
                $this->assertEquals($amount, $event->getAmount());
                $event->setValid(false);
                $event->stopPropagation();
                $eventFired = true;
            }, 9999999);

        //act
        $this->client->request('GET', $this->generateUrl('payment_cathay_credit_confirm', array(
            'ordernumber' => $ordernumber,
            'amount' => $amount,
            'storeid' => $storeid,
        )));
        $response = $this->client->getResponse();

        //assert
        $this->assertTrue($response->isOk());
        $this->assertTrue($eventFired);
        $this->assertEquals("<EPOS OrderConfirm='no' />", $response->getContent());
    }

    public function test_creditConfirmAction_event_emit_valid()
    {
        //arrange
        
        $eventFired = false;
        $container = $this->client->getContainer();
        $ordernumber = 'o12345';
        $amount = 123;
        $storeid = 'cathay_credit_store_id';
        $container
            ->get('event_dispatcher')
            ->addListener(VerifyAmountEvent::EVENT_NAME, function(VerifyAmountEvent $event) use(&$eventFired, $ordernumber, $amount){
                $this->assertEquals($ordernumber, $event->getOrdernumber());
                $this->assertEquals($amount, $event->getAmount());
                $event->setValid(true);
                $event->stopPropagation();
                $eventFired = true;
            }, 9999999);

        //act
        $this->client->request('GET', $this->generateUrl('payment_cathay_credit_confirm', array(
            'ordernumber' => $ordernumber,
            'amount' => $amount,
            'storeid' => $storeid,
        )));
        $response = $this->client->getResponse();

        //assert
        $this->assertTrue($eventFired);
        $this->assertTrue($response->isOk());
        $this->assertEquals("<EPOS OrderConfirm='yes' />", $response->getContent());
    }

    public function test_creditReceiveOrderAction_event_bad_store_id()
    {
        //arrange
        
        $container = $this->client->getContainer();
        $ordernumber = 'o12345';
        $amount = 123;
        $storeid = 'bad_store_id';
        $container
            ->get('event_dispatcher')
            ->addListener(ReceiveOrderEvent::EVENT_NAME, function(){
                $this->fail('不應該觸發 Event 事件');
            }, 9999999);

        //act
        $this->client->request('GET', $this->generateUrl('payment_cathay_credit_receive', array(
            'ordernumber' => $ordernumber,
            'amount' => $amount,
            'storeid' => $storeid,
            'JPOSauthamount' => 123,
            'JPOSresponsecode' => '00',
            'JPOSauthstatus' => 'aa',
            'JPOSapprovecode' => '123',
            'JPOStransdate' => '20160222',
            'JPOStranstime' => '142012',
        )));
        $response = $this->client->getResponse();

        //assert
        $this->assertTrue($response->isNotFound());
    }

    public function test_creditReceiveOrderAction_event_emit_success()
    {
        //arrange
        
        $responseEvent = null;
        $container = $this->client->getContainer();
        $returnUrl = 'http://localhost:8888/';
        $ordernumber = 'o12345';
        $amount = '123';
        $storeid = 'cathay_credit_store_id';
        $params = array(
            'ordernumber' => $ordernumber,
            'amount' => $amount,
            'storeid' => $storeid,
            'JPOSauthamount' => '123',
            'JPOSresponsecode' => '00',
            'JPOSauthstatus' => 'aa',
            'JPOSapprovecode' => '123',
            'JPOStransdate' => '20160222',
            'JPOStranstime' => '142012',
        );
        $diapatcher = $container->get('event_dispatcher');
        $diapatcher->addListener(ReceiveOrderEvent::EVENT_NAME, function(ReceiveOrderEvent $event) use(&$responseEvent, $params){
            $event->stopPropagation();
            $this->assertEquals('CathayCredit', $event->getPaymentType());
            $responseEvent = $event;
        }, 9999999);
        $diapatcher->addListener(ReturnUrlEvent::EVENT_NAME, function(ReturnUrlEvent $event) use($returnUrl){
            $event->stopPropagation();
            $event->setUrl($returnUrl);
        }, 9999999);

        //act
        $crawler = $this->client->request('GET', $this->generateUrl('payment_cathay_credit_receive', $params));
        $response = $this->client->getResponse();

        //assert
        $this->assertNotNull($responseEvent);
        $this->assertTrue($responseEvent->getStatus());
        $this->assertEquals($params['ordernumber'], $responseEvent->getOrdernumber());
        $this->assertEquals($params['amount'], $responseEvent->getAmount());
        $this->assertEquals($params, $responseEvent->getOriginInfo());
        $this->assertEquals("0;url=$returnUrl", $crawler->filter('meta[http-equiv="refresh"]')->attr('content'));
        $this->assertGreaterThan(0, preg_match('/location="(.*)";/i', trim($crawler->filter('script')->html()), $match));
        $this->assertEquals($returnUrl, $match[1]);
    }

    public function test_creditReceiveOrderAction_event_emit_fail()
    {
        //arrange
        
        $responseEvent = null;
        $container = $this->client->getContainer();
        $returnUrl = 'http://localhost:8888/';
        $ordernumber = 'o12345';
        $amount = '123';
        $storeid = 'cathay_credit_store_id';
        $params = array(
            'ordernumber' => $ordernumber,
            'amount' => $amount,
            'storeid' => $storeid,
            'JPOSauthamount' => '123',
            'JPOSresponsecode' => '01',
            'JPOSauthstatus' => 'aa',
            'JPOSapprovecode' => '123',
            'JPOStransdate' => '20160222',
            'JPOStranstime' => '142012',
        );
        $diapatcher = $container->get('event_dispatcher');
        $diapatcher->addListener(ReceiveOrderEvent::EVENT_NAME, function(ReceiveOrderEvent $event) use(&$responseEvent, $params){
            $event->stopPropagation();
            $responseEvent = $event;
            $this->assertEquals('CathayCredit', $event->getPaymentType());
        }, 9999999);

        $diapatcher->addListener(ReturnUrlEvent::EVENT_NAME, function(ReturnUrlEvent $event) use($returnUrl){
            $event->stopPropagation();
            $event->setUrl($returnUrl);
        }, 9999999);

        //act
        $crawler = $this->client->request('GET', $this->generateUrl('payment_cathay_credit_receive', $params));
        $response = $this->client->getResponse();

        //assert
        $this->assertNotNull($responseEvent);
        $this->assertFalse($responseEvent->getStatus());
        $this->assertEquals($params['ordernumber'], $responseEvent->getOrdernumber());
        $this->assertEquals($params['amount'], $responseEvent->getAmount());
        $this->assertEquals($params, $responseEvent->getOriginInfo());
        $this->assertEquals("0;url=$returnUrl", $crawler->filter('meta[http-equiv="refresh"]')->attr('content'));
        $this->assertGreaterThan(0, preg_match('/location="(.*)";/i', trim($crawler->filter('script')->html()), $match));
        $this->assertEquals($returnUrl, $match[1]);
    }
}
