<?php
namespace Widget\OrderBundle\Tests\Controller\API;

use Backend\BaseBundle\Model\Site;
use Backend\BaseBundle\Tests\Fixture\BaseWebTestCase;
use JMS\Serializer\SerializationContext;
use Symfony\Component\EventDispatcher\EventDispatcher;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;
use Widget\MemberBundle\Model\Member;
use Widget\MemberBundle\Model\MemberQuery;
use Widget\MemberBundle\Token\MemberAuthToken;
use Widget\OrderBundle\Controller\API\OrderController;
use Widget\OrderBundle\Event\DiscountChainEvent;
use Widget\OrderBundle\Event\DiscountModulesEvent;
use Widget\OrderBundle\Event\ProductAddEvent;
use Widget\OrderBundle\Event\ReceiveCreateOrderEvent;
use Widget\OrderBundle\Model\Order;
use Widget\OrderBundle\Model\OrderInfoQuery;
use Widget\OrderBundle\Model\OrderQuery;
use Widget\OrderBundle\Service\DiscountBuilder;
use Widget\OrderBundle\Service\OrderBuilder;
use Widget\StockBundle\EventListener\ProductAddEventListener;

class OrderControllerTest extends BaseWebTestCase
{
    public function setUp()
    {
        parent::setUp();
        \Propel::getConnection()->rollBack(); //reset previous transaction
    }

    /**
     * 沒登入的情況
     */
    public function test_queryAction_no_token()
    {
        //arrange
        $data = array(
            'order' => array(
                array(
                    'id'    => '123456',
                    'count' => 2,
                ),
            ),
        );

        //act
        $this->client->request(
            'POST',
            $this->generateUrl('widget_order_api_order_query'),
            array(),
            array(),
            array(),
            json_encode($data)
        );
        $response = $this->client->getResponse();

        //assert
        $this->assertTrue($response->isOk());
    }

    /**
     * 登入成功，且正確的情況
     *
     */
    public function test_queryAction_author_valid()
    {
        //arrange
        $data = array(
            'info' => array(
                'buyer'    => array(
                    'name'    => 'buyer',
                    'email'   => 'buyer@test.com',
                    'phone'   => '0912345678',
                    'zip'     => '231',
                    'address' => '台北市中正區延平南路96號',
                ),
                'receiver' => array(
                    'name'    => 'receiver',
                    'email'   => 'receiver@test.com',
                    'phone'   => '0987654321',
                    'zip'     => '132',
                    'address' => '台北市中正區重慶南路一段122號',
                ),
            ),
            'order' => array(
                array(
                    'id'    => '123456',
                    'count' => 2,
                ),
            ),
        );

        $origin = 'http://localhost';
        $secret = $this->client->getContainer()->getParameter('secret');
        $member = new Member();
        $member->setId(12345);
        $authToken = new MemberAuthToken($secret, $origin, $origin);
        $authToken->setMember($member);
        $token = $authToken->getJWTToken();

        //act
        $this->client->request(
            'POST',
            $this->generateUrl('widget_order_api_order_query'),
            array(),
            array(),
            array(
                'HTTP_ORIGIN' => $origin,
                'HTTP_AUTHORIZATION' => "Bearer $token",
            ),
            json_encode($data)
        );
        $response = $this->client->getResponse();

        //assert
        $this->assertTrue($response->isOk());
    }

    /**
     * 登入成功，資料回傳正確的情況
     */
    public function test_queryAction_valid_data()
    {
        //arrange
        $productData = array(
            '123456' => array(
                'name' => 'test product',
                'price' => 456,
            ),
        );
        $data = array(
            'info' => array(
                'buyer'    => array(
                    'name'    => 'buyer',
                    'email'   => 'buyer@test.com',
                    'phone'   => '0912345678',
                    'zip'     => '231',
                    'address' => '台北市中正區延平南路96號',
                ),
                'receiver' => array(
                    'name'    => 'receiver',
                    'email'   => 'receiver@test.com',
                    'phone'   => '0987654321',
                    'zip'     => '132',
                    'address' => '台北市中正區重慶南路一段122號',
                ),
            ),
            'order' => array(
                array(
                    'id'    => '123456',
                    'count' => 2,
                ),
            ),
            'discount' => array(
                'coupon_code' => '55667788'
            )
        );

        $eventDispatcher = $this->client->getContainer()->get('event_dispatcher');
        $eventDispatcher->addListener(ProductAddEvent::EVENT_NAME, function(ProductAddEvent $event) use($productData){
            $product = $productData[$event->getId()];
            $event->setProductPrice($product['price']);
            $event->setProductName($product['name']);
            $event->stopPropagation();
        }, 999999);

        // 折扣 Event
        $eventDispatcher->addListener(DiscountModulesEvent::EVENT_NAME,function(DiscountModulesEvent $event) use($data){
            $this->assertEquals($event->getParameters(), $data['discount']);
            $event->stopPropagation();
        }, 99999);

        $expectedData = array(
            'amount' => '912',
            'amount_for_pay' => '912',
            'order_items' => array(
                array(
                    'product_id' => '123456',
                    'name' => 'test product',
                    'unit_price' => 456,
                    'quantity' => 2
                ),
            ),
            'order_logs' => array(),
            'order_status' => null,
            'created_at' => null,
            'updated_at' => null,
            'payment' => null,
            'order_note' => null,
            'order_info' => null,
        );

        $origin = 'http://localhost';
        $secret = $this->client->getContainer()->getParameter('secret');
        $member = new Member();
        $member->setId(12345);
        $authToken = new MemberAuthToken($secret, $origin, $origin);
        $authToken->setMember($member);
        $token = $authToken->getJWTToken();

        //act
        $this->client->request(
            'POST',
            $this->generateUrl('widget_order_api_order_query'),
            array(),
            array(),
            array(
                'HTTP_ORIGIN' => $origin,
                'HTTP_AUTHORIZATION' => "Bearer $token",
            ),
            json_encode($data)
        );
        $response = $this->client->getResponse();
        $result = json_decode($response->getContent(), true);

        //assert
        $this->assertTrue($response->isOk());
        $this->assertEquals($expectedData, $result);
    }

    /**
     * 登入成功，但庫存不足
     */
    public function test_queryAction_stock_not_enough()
    {
        //arrange
        $productData = array(
            '123456' => array(
                'name' => 'test product',
                'price' => 456,
            ),
        );
        $data = array(
            'info' => array(
                'buyer'    => array(
                    'name'    => 'buyer',
                    'email'   => 'buyer@test.com',
                    'phone'   => '0912345678',
                    'zip'     => '231',
                    'address' => '台北市中正區延平南路96號',
                ),
                'receiver' => array(
                    'name'    => 'receiver',
                    'email'   => 'receiver@test.com',
                    'phone'   => '0987654321',
                    'zip'     => '132',
                    'address' => '台北市中正區重慶南路一段122號',
                ),
            ),
            'order' => array(
                array(
                    'id'    => '123456',
                    'count' => 2,
                ),
            ),
        );

        $eventDispatcher = $this->client->getContainer()->get('event_dispatcher');
        $eventDispatcher->addListener(ProductAddEvent::EVENT_NAME, function(ProductAddEvent $event) use($productData){
            $event->setError('out of stock');
            $event->stopPropagation();
        }, 999999);

        $expectedData = array(
            'amount' => 912,
            'amount_for_pay' => 912,
            'order_items' => array(
                array(
                    'name' => 'test product',
                    'unit_price' => 456,
                    'quantity' => 2
                ),
            ),
        );

        $origin = 'http://localhost';
        $secret = $this->client->getContainer()->getParameter('secret');
        $member = new Member();
        $member->setId(12345);
        $authToken = new MemberAuthToken($secret, $origin, $origin);
        $authToken->setMember($member);
        $token = $authToken->getJWTToken();

        //act
        $this->client->request(
            'POST',
            $this->generateUrl('widget_order_api_order_query'),
            array(),
            array(),
            array(
                'HTTP_ORIGIN' => $origin,
                'HTTP_AUTHORIZATION' => "Bearer $token",
            ),
            json_encode($data)
        );
        $response = $this->client->getResponse();
        $result = json_decode($response->getContent(), true);

        //assert
        $this->assertEquals(Response::HTTP_BAD_REQUEST, $response->getStatusCode());
    }


    /**
     * 錯誤的資料情況
     */
    public function test_createAction_error_data()
    {
        //arrange
        $productData = array(
            '123456' => array(
                'name' => 'test product',
                'price' => 456,
            ),
        );
        $data = array(
            'order' => array(
                array(
                    'id'    => '123456',
                    'count' => 'AGHHH',
                ),
            ),
        );

        $eventDispatcher = $this->client->getContainer()->get('event_dispatcher');
        $eventDispatcher->addListener(ProductAddEvent::EVENT_NAME, function(ProductAddEvent $event) use($productData){
            $product = $productData[$event->getId()];
            $event->setProductPrice($product['price']);
            $event->setProductName($product['name']);
            $event->stopPropagation();
        }, 999999);

        $origin = 'http://localhost';
        $secret = $this->client->getContainer()->getParameter('secret');
        $member = new Member();
        $member->setId(12345);
        $authToken = new MemberAuthToken($secret, $origin, $origin);
        $authToken->setMember($member);
        $token = $authToken->getJWTToken();

        //act
        $this->client->request(
            'POST',
            $this->generateUrl('widget_order_api_order_create'),
            array(),
            array(),
            array(
                'HTTP_ORIGIN' => $origin,
                'HTTP_AUTHORIZATION' => "Bearer $token",
            ),
            json_encode($data)
        );
        $response = $this->client->getResponse();

        //assert
        $this->assertEquals(Response::HTTP_BAD_REQUEST, $response->getStatusCode());
    }

    /**
     * 未登入&資料沒填E-mail
     */
    public function test_createAction_no_token_bad_param()
    {
        //arrange
        $productData = array(
            '123456' => array(
                'name' => 'test product',
                'price' => 456,
            ),
            '654321' => array(
                'name' => 'test discount',
                'price' => -100,
            ),
        );
        $data = array(
            'info' => array(
                'buyer'    => array(
                    'name'    => 'buyer',
                    'phone'   => '0912345678',
                    'zip'     => '231',
                    'address' => '台北市中正區延平南路96號',
                ),
                'receiver' => array(
                    'name'    => 'receiver',
                    'email'   => 'receiver@test.com',
                    'phone'   => '0987654321',
                    'zip'     => '132',
                    'address' => '台北市中正區重慶南路一段122號',
                ),
            ),
            'order' => array(
                array(
                    'id'    => '123456',
                    'count' => 10,
                ),
                array(
                    'id'    => '654321',
                    'count' => 1,
                ),
            ),
            'order_note' => '123456789',
            'discount' => array(
                'coupon_code' => 123456
            ),
        );

        $eventDispatcher = $this->client->getContainer()->get('event_dispatcher');
        $eventDispatcher->addListener(ProductAddEvent::EVENT_NAME, function(ProductAddEvent $event) use($productData){
            $product = $productData[$event->getId()];
            $event->setProductPrice($product['price']);
            $event->setProductName($product['name']);
            $event->stopPropagation();
        }, 999999);

        // 折扣 Event
        $eventDispatcher->addListener(DiscountModulesEvent::EVENT_NAME,function(DiscountModulesEvent $event) use($data){
            $this->assertEquals($event->getParameters(), $data['discount']);
            $event->stopPropagation();
        }, 99999);

        //act
        $this->client->request(
            'POST',
            $this->generateUrl('widget_order_api_order_create'),
            array(),
            array(),
            array(),
            json_encode($data)
        );
        
        $response = $this->client->getResponse();
        
        //assert
        $this->assertEquals(Response::HTTP_BAD_REQUEST, $response->getStatusCode());
    }

    /**
     * 未登入購買的狀態
     */
    public function test_createAction_no_token()
    {
        //arrange
        $productData = array(
            '123456' => array(
                'name' => 'test product',
                'price' => 456,
            ),
            '654321' => array(
                'name' => 'test discount',
                'price' => -100,
            ),
        );
        $data = array(
            'info' => array(
                'buyer'    => array(
                    'name'    => 'buyer',
                    'email'   => 'buyer@test.com',
                    'phone'   => '0912345678',
                    'zip'     => '231',
                    'address' => '台北市中正區延平南路96號',
                ),
                'receiver' => array(
                    'name'    => 'receiver',
                    'email'   => 'receiver@test.com',
                    'phone'   => '0987654321',
                    'zip'     => '132',
                    'address' => '台北市中正區重慶南路一段122號',
                ),
            ),
            'order' => array(
                array(
                    'id'    => '123456',
                    'count' => 10,
                ),
                array(
                    'id'    => '654321',
                    'count' => 1,
                ),
            ),
            'order_note' => '123456789',
            'discount' => array(
                'coupon_code' => 123456
            ),
        );

        $eventDispatcher = $this->client->getContainer()->get('event_dispatcher');
        $eventDispatcher->addListener(ProductAddEvent::EVENT_NAME, function(ProductAddEvent $event) use($productData){
            $product = $productData[$event->getId()];
            $event->setProductPrice($product['price']);
            $event->setProductName($product['name']);
            $event->stopPropagation();
        }, 999999);

        // 折扣 Event
        $eventDispatcher->addListener(DiscountModulesEvent::EVENT_NAME,function(DiscountModulesEvent $event) use($data){
            $this->assertEquals($event->getParameters(), $data['discount']);
            $event->stopPropagation();
        }, 99999);

        $expectedData = array(
            'amount' => 4560,
            'amount_for_pay' => 4460,
            'order_items' => array(
                array(
                    'id' => '123456',
                    'name' => 'test product',
                    'unit_price' => 456,
                    'quantity' => 10
                ),
                array(
                    'id' => '654321',
                    'name' => 'test discount',
                    'unit_price' => -100,
                    'quantity' => 1
                ),
            ),
            'order_note' => '123456789'
        );
        $responseData = new Order();

        //act
        $this->client->request(
            'POST',
            $this->generateUrl('widget_order_api_order_create'),
            array(),
            array(),
            array(),
            json_encode($data)
        );
        $con = \Propel::getConnection();
        $responseData->syncAmount($con);
        $response = $this->client->getResponse();
        $token = json_decode($response->getContent(), true)['token'];
        $payload = $this->client->getContainer()->get('widget_order.payment.token')->verifyToken($token);
        $order_note = OrderQuery::create()
            ->findOneById($payload->ordernumber);
        $member = MemberQuery::create()->findOneByEmail($data['info']['buyer']['email']);

        //assert
        $this->assertTrue($response->isOk());
        $this->assertNotNull($payload->ordernumber);
        $this->assertNotNull($member);
        $this->assertEquals($expectedData['amount_for_pay'], $payload->amount);
        $this->assertEquals($data['order_note'], $order_note->getOrderNote());
    }

    /**
     * 測試create的資料正確情況
     */
    public function test_createAction_valid_data()
    {
        //arrange
        $mailSend = false;
        $productData = array(
            '123456' => array(
                'name' => 'test product',
                'price' => 456,
            ),
            '654321' => array(
                'name' => 'test discount',
                'price' => -100,
            ),
        );
        $data = array(
            'info' => array(
                'buyer'    => array(
                    'name'    => 'buyer',
                    'email'   => 'buyer@test.com',
                    'phone'   => '0912345678',
                    'zip'     => '231',
                    'address' => '台北市中正區延平南路96號',
                ),
                'receiver' => array(
                    'name'    => 'receiver',
                    'email'   => 'receiver@test.com',
                    'phone'   => '0987654321',
                    'zip'     => '132',
                    'address' => '台北市中正區重慶南路一段122號',
                ),
            ),
            'order' => array(
                array(
                    'id'    => '123456',
                    'count' => 10,
                ),
                array(
                    'id'    => '654321',
                    'count' => 1,
                ),
            ),
            'order_note' => '123456789',
            'discount' => array(
                'coupon_code' => 123456
            ),
        );

        $eventDispatcher = $this->client->getContainer()->get('event_dispatcher');
        $eventDispatcher->addListener(ProductAddEvent::EVENT_NAME, function(ProductAddEvent $event) use($productData){
            $product = $productData[$event->getId()];
            $event->setProductPrice($product['price']);
            $event->setProductName($product['name']);
            $event->stopPropagation();
        }, 999999);

        // 折扣 Event
        $eventDispatcher->addListener(DiscountModulesEvent::EVENT_NAME,function(DiscountModulesEvent $event) use($data){
            $this->assertEquals($event->getParameters(), $data['discount']);
            $event->stopPropagation();
        }, 99999);

        $expectedData = array(
            'amount' => 4560,
            'amount_for_pay' => 4460,
            'order_items' => array(
                array(
                    'id' => '123456',
                    'name' => 'test product',
                    'unit_price' => 456,
                    'quantity' => 10
                ),
                array(
                    'id' => '654321',
                    'name' => 'test discount',
                    'unit_price' => -100,
                    'quantity' => 1
                ),
            ),
            'order_note' => '123456789'
        );
        $responseData = new Order();

        $origin = 'http://localhost';
        $secret = $this->client->getContainer()->getParameter('secret');
        $member = new Member();
        $member->setId(12345);
        $authToken = new MemberAuthToken($secret, $origin, $origin);
        $authToken->setMember($member);
        $token = $authToken->getJWTToken();

        //act
        $this->client->request(
            'POST',
            $this->generateUrl('widget_order_api_order_create'),
            array(),
            array(),
            array(
                'HTTP_ORIGIN' => $origin,
                'HTTP_AUTHORIZATION' => "Bearer $token",
            ),
            json_encode($data)
        );
        $con = \Propel::getConnection();
        $responseData->syncAmount($con);
        $response = $this->client->getResponse();
        $token = json_decode($response->getContent(), true)['token'];
        $payload = $this->client->getContainer()->get('widget_order.payment.token')->verifyToken($token);
        $order_note = OrderQuery::create()
            ->findOneById($payload->ordernumber);

        //assert
        $this->assertTrue($response->isOk());
        $this->assertNotNull($payload->ordernumber);
        $this->assertEquals($expectedData['amount_for_pay'], $payload->amount);
        $this->assertEquals($data['order_note'], $order_note->getOrderNote());
    }

    /**
     * 測試create的資料正確情況
     */
    public function test_createAction_send_mail()
    {
        //arrange
        $mailSend = false;
        $productData = array(
            '123456' => array(
                'name' => 'test product',
                'price' => 456,
            ),
            '654321' => array(
                'name' => 'test discount',
                'price' => -100,
            ),
        );
        $data = array(
            'info' => array(
                'buyer'    => array(
                    'name'    => 'buyer',
                    'email'   => 'buyer@test.com',
                    'phone'   => '0912345678',
                    'zip'     => '231',
                    'address' => '台北市中正區延平南路96號',
                ),
                'receiver' => array(
                    'name'    => 'receiver',
                    'email'   => 'receiver@test.com',
                    'phone'   => '0987654321',
                    'zip'     => '132',
                    'address' => '台北市中正區重慶南路一段122號',
                ),
            ),
            'order' => array(
                array(
                    'id'    => '123456',
                    'count' => 10,
                ),
                array(
                    'id'    => '654321',
                    'count' => 1,
                ),
            ),
            'order_note' => '123456789'
        );

        $eventDispatcher = $this->client->getContainer()->get('event_dispatcher');
        $eventDispatcher->addListener(ProductAddEvent::EVENT_NAME, function(ProductAddEvent $event) use($productData){
            $product = $productData[$event->getId()];
            $event->setProductPrice($product['price']);
            $event->setProductName($product['name']);
            $event->stopPropagation();
        }, 999999);

        // 折扣 Event
        $eventDispatcher->addListener(DiscountModulesEvent::EVENT_NAME,function(DiscountModulesEvent $event) use($data){
            $this->assertEquals($event->getParameters(), array());
            $event->stopPropagation();
        }, 99999);

        $expectedData = array(
            'amount' => 4560,
            'amount_for_pay' => 4460,
            'order_items' => array(
                array(
                    'id' => '123456',
                    'name' => 'test product',
                    'unit_price' => 456,
                    'quantity' => 10
                ),
                array(
                    'id' => '654321',
                    'name' => 'test discount',
                    'unit_price' => -100,
                    'quantity' => 1
                ),
            ),
            'order_note' => '123456789'
        );
        $responseData = new Order();

        $origin = 'http://localhost';
        $secret = $this->client->getContainer()->getParameter('secret');
        $member = new Member();
        $member->setId(12345);
        $authToken = new MemberAuthToken($secret, $origin, $origin);
        $authToken->setMember($member);
        $token = $authToken->getJWTToken();

        //act
        $this->client->request(
            'POST',
            $this->generateUrl('widget_order_api_order_create'),
            array(),
            array(),
            array(
                'HTTP_ORIGIN' => $origin,
                'HTTP_AUTHORIZATION' => "Bearer $token",
            ),
            json_encode($data)
        );
        $con = \Propel::getConnection();
        $receiveCreateOrderEvent = new ReceiveCreateOrderEvent($responseData, $con);
        $eventDispatcher->addListener(ReceiveCreateOrderEvent::EVENT_NAME, $receiveCreateOrderEvent);
        $responseData->syncAmount($con);
        $response = $this->client->getResponse();
        $token = json_decode($response->getContent(), true)['token'];
        $payload = $this->client->getContainer()->get('widget_order.payment.token')->verifyToken($token);

        //assert
        $this->assertTrue($response->isOk());
        $this->assertNotNull($payload->ordernumber);
        $this->assertEquals($expectedData['amount_for_pay'], $payload->amount);
    }

    /**
     * 購買時發現庫存不足
     */
    public function test_createAction_stock_not_enough()
    {
        //arrange
        $data = array(
            'info' => array(
                'buyer'    => array(
                    'name'    => 'buyer',
                    'email'   => 'buyer@test.com',
                    'phone'   => '0912345678',
                    'zip'     => '231',
                    'address' => '台北市中正區延平南路96號',
                ),
                'receiver' => array(
                    'name'    => 'receiver',
                    'email'   => 'receiver@test.com',
                    'phone'   => '0987654321',
                    'zip'     => '132',
                    'address' => '台北市中正區重慶南路一段122號',
                ),
            ),
            'order' => array(
                array(
                    'id'    => '123456',
                    'count' => 999999,
                ),
            )
        );

        $eventDispatcher = $this->client->getContainer()->get('event_dispatcher');

        $eventDispatcher->addListener(ProductAddEvent::EVENT_NAME, function(ProductAddEvent $event) {
            $event->setError('out of stock');
            $event->stopPropagation();
        }, 999999);

        $origin = 'http://localhost';
        $secret = $this->client->getContainer()->getParameter('secret');
        $member = new Member();
        $member->setId(12345);
        $authToken = new MemberAuthToken($secret, $origin, $origin);
        $authToken->setMember($member);
        $token = $authToken->getJWTToken();

        //act
        $this->client->request(
            'POST',
            $this->generateUrl('widget_order_api_order_create'),
            array(),
            array(),
            array(
                'HTTP_ORIGIN' => $origin,
                'HTTP_AUTHORIZATION' => "Bearer $token",
            ),
            json_encode($data)
        );

        $response = $this->client->getResponse();

        //assert
        $this->assertEquals(Response::HTTP_BAD_REQUEST, $response->getStatusCode());
    }

    public function test_queryOrderAction_null_token()
    {
        //arrange
        $ordernumber = 'P0001';
        $origin = 'http://localhost';
        $secret = $this->client->getContainer()->getParameter('secret');
        $member = new Member();
        $member->setId(1000);
        $authToken = new MemberAuthToken($secret, $origin, $origin);
        $authToken->setMember($member);
        $token = $authToken->getJWTToken();


        //act
        $this->client->request(
            'GET',
            $this->generateUrl('widget_order_api_order_queryorder', array('id' => $ordernumber))
        );
        $response = $this->client->getResponse();

        //assert
        $this->assertTrue($response->isForbidden());
    }

    public function test_queryOrderAction_bad_member()
    {
        //arrange
        $ordernumber = 'P0001';
        $origin = 'http://localhost';
        $secret = $this->client->getContainer()->getParameter('secret');
        $member = new Member();
        $member->setId(12345);
        $authToken = new MemberAuthToken($secret, $origin, $origin);
        $authToken->setMember($member);
        $token = $authToken->getJWTToken();


        //act
        $this->client->request(
            'GET',
            $this->generateUrl('widget_order_api_order_queryorder', array('id' => $ordernumber)),
            array(),
            array(),
            array(
                'HTTP_ORIGIN' => $origin,
                'HTTP_AUTHORIZATION' => "Bearer $token",
            )
        );
        $response = $this->client->getResponse();

        //assert
        $this->assertTrue($response->isForbidden());
    }

    public function test_queryOrderAction_order_not_found()
    {
        //arrange
        $ordernumber = 'BAD_ORDER';
        $origin = 'http://localhost';
        $secret = $this->client->getContainer()->getParameter('secret');
        $member = new Member();
        $member->setId(1000);
        $authToken = new MemberAuthToken($secret, $origin, $origin);
        $authToken->setMember($member);
        $token = $authToken->getJWTToken();


        //act
        $this->client->request(
            'GET',
            $this->generateUrl('widget_order_api_order_queryorder', array('id' => $ordernumber)),
            array(),
            array(),
            array(
                'HTTP_ORIGIN' => $origin,
                'HTTP_AUTHORIZATION' => "Bearer $token",
            )
        );
        $response = $this->client->getResponse();

        //assert
        $this->assertTrue($response->isNotFound());
    }

    public function test_queryOrderAction_author_valid()
    {
        //arrange
        $ordernumber = 'P0001';
        $origin = 'http://localhost';
        $secret = $this->client->getContainer()->getParameter('secret');
        $member = new Member();
        $member->setId(1000);
        $authToken = new MemberAuthToken($secret, $origin, $origin);
        $authToken->setMember($member);
        $token = $authToken->getJWTToken();

        //act
        $this->client->request(
            'GET',
            $this->generateUrl('widget_order_api_order_queryorder', array('id' => $ordernumber)),
            array(),
            array(),
            array(
                'HTTP_ORIGIN' => $origin,
                'HTTP_AUTHORIZATION' => "Bearer $token",
            )
        );
        $response = $this->client->getResponse();

        //assert
        $this->assertTrue($response->isOk());
    }

    public function test_queryAllOrderAction_order_not_found()
    {
        //arrange
        $ordernumber = 'BAD_ORDER';
        $origin = 'http://localhost';
        $secret = $this->client->getContainer()->getParameter('secret');
        $member = new Member();
        $member->setId(1000);
        $authToken = new MemberAuthToken($secret, $origin, $origin);
        $authToken->setMember($member);
        $token = $authToken->getJWTToken();


        //act
        $this->client->request(
            'GET',
            $this->generateUrl('widget_order_api_order_queryorder', array('id' => $ordernumber)),
            array(),
            array(),
            array(
                'HTTP_ORIGIN' => $origin,
                'HTTP_AUTHORIZATION' => "Bearer $token",
            )
        );
        $response = $this->client->getResponse();

        //assert
        $this->assertTrue($response->isNotFound());
    }

    public function test_queryAllOrderAction_data1_valid()
    {
        //arrange
        $origin = 'http://localhost';
        $secret = $this->client->getContainer()->getParameter('secret');
        $member = new Member();
        $member->setId(1000);
        $authToken = new MemberAuthToken($secret, $origin, $origin);
        $authToken->setMember($member);
        $token = $authToken->getJWTToken();
        $authData = $authToken->getAuthData();

        //act
        $orders = OrderQuery::create()
            ->filterByMemberId($authData['uid'])
            ->find();

        $this->client->request(
            'GET',
            $this->generateUrl('widget_order_api_order_queryallorder'),
            array(),
            array(),
            array(
                'HTTP_ORIGIN' => $origin,
                'HTTP_AUTHORIZATION' => "Bearer $token",
            )
        );
        $response = $this->client->getResponse();
        $result = json_decode($response->getContent(), true);

        //assert
        $this->assertTrue($response->isOk());
        $this->assertEquals(count($orders), $result['pager']['rows']);
        $this->assertEquals(4, $result['pager']['pages']);
        $this->assertEquals(1, $result['pager']['page']);
        $this->assertEquals(10, count($result['data']));
        $this->assertEquals($this->makeRange(29, 20), array_column($result['data'], 'id'));
    }

    public function test_queryAllOrderAction_data2_valid()
    {
        //arrange
        $origin = 'http://localhost';
        $secret = $this->client->getContainer()->getParameter('secret');
        $member = new Member();
        $member->setId(1000);
        $authToken = new MemberAuthToken($secret, $origin, $origin);
        $authToken->setMember($member);
        $token = $authToken->getJWTToken();
        $authData = $authToken->getAuthData();
        $page = 2;



        //act
        $orders = OrderQuery::create()
            ->filterByMemberId($authData['uid'])
            ->find();

        $this->client->request(
            'GET',
            $this->generateUrl('widget_order_api_order_queryallorder', array('page' => $page)),
            array(),
            array(),
            array(
                'HTTP_ORIGIN' => $origin,
                'HTTP_AUTHORIZATION' => "Bearer $token",
            )
        );
        $response = $this->client->getResponse();
        $result = json_decode($response->getContent(), true);

        //assert
        $this->assertTrue($response->isOk());
        $this->assertEquals(count($orders), $result['pager']['rows']);
        $this->assertEquals(4, $result['pager']['pages']);
        $this->assertEquals(2, $result['pager']['page']);
        $this->assertEquals(10, count($result['data']));
        $this->assertEquals($this->makeRange(19, 10), array_column($result['data'], 'id'));
    }

    public function test_queryAllOrderAction_data3_invalid()
    {
        //arrange
        $origin = 'http://localhost';
        $secret = $this->client->getContainer()->getParameter('secret');
        $member = new Member();
        $member->setId(1000);
        $authToken = new MemberAuthToken($secret, $origin, $origin);
        $authToken->setMember($member);
        $token = $authToken->getJWTToken();
        $authData = $authToken->getAuthData();
        $page = -1;



        //act
        $orders = OrderQuery::create()
            ->filterByMemberId($authData['uid'])
            ->find();

        $this->client->request(
            'GET',
            $this->generateUrl('widget_order_api_order_queryallorder', array('page' => $page)),
            array(),
            array(),
            array(
                'HTTP_ORIGIN' => $origin,
                'HTTP_AUTHORIZATION' => "Bearer $token",
            )
        );
        $response = $this->client->getResponse();
        $result = json_decode($response->getContent(), true);

        //assert
        $this->assertTrue($response->isOk());
        $this->assertEquals(count($orders), $result['pager']['rows']);
        $this->assertEquals(4, $result['pager']['pages']);
        $this->assertEquals(1, $result['pager']['page']);
        $this->assertEquals(10, count($result['data']));
        $this->assertEquals($this->makeRange(29, 20), array_column($result['data'], 'id'));
    }

    public function test_queryAllOrderAction_data4_invalid()
    {
        //arrange
        $origin = 'http://localhost';
        $secret = $this->client->getContainer()->getParameter('secret');
        $member = new Member();
        $member->setId(1000);
        $authToken = new MemberAuthToken($secret, $origin, $origin);
        $authToken->setMember($member);
        $token = $authToken->getJWTToken();
        $authData = $authToken->getAuthData();
        $page = 999999999;



        //act
        $orders = OrderQuery::create()
            ->filterByMemberId($authData['uid'])
            ->find();

        $this->client->request(
            'GET',
            $this->generateUrl('widget_order_api_order_queryallorder', array('page' => $page)),
            array(),
            array(),
            array(
                'HTTP_ORIGIN' => $origin,
                'HTTP_AUTHORIZATION' => "Bearer $token",
            )
        );
        $response = $this->client->getResponse();
        $result = json_decode($response->getContent(), true);

        //assert
        $this->assertTrue($response->isOk());
        $this->assertEquals(count($orders), $result['pager']['rows']);
        $this->assertEquals(4, $result['pager']['pages']);
        $this->assertEquals(4, $result['pager']['page']);
    }

    public function test_createOrderInfo_bad_valid()
    {
        // arrange
        $orderController = $this->getMockBuilder(OrderController::class)
            ->setMethods(array('validOrderInfo'))
            ->disableOriginalConstructor()
            ->getMock();

        $expectResponse = new Response("error", 400);
        $orderController
            ->expects($this->once())
            ->method('validOrderInfo')
            ->willReturn($expectResponse);

        $orderId = "OrderTest54321";

        $order = new Order();
        $order->setId($orderId);

        $parameters = array(
            'buyer'    => array(
                'name'    => 'buyer',
                'email'   => 'buyer@test.com',
                'phone'   => '0912345678',
                'zip'     => '231',
                'address' => '台北市中正區延平南路96號',
            ),
            'receiver' => array(
                'name'    => 'receiver',
                'email'   => 'receiver@test.com',
                'phone'   => '0987654321',
                'zip'     => '132',
                'address' => '台北市中正區重慶南路一段122號',
            ),
        );

        // act
        $response = $this->callObjectMethod($orderController, 'createOrderInfo', $orderId, $parameters);

        // assert
        $this->assertEquals($expectResponse, $response);

    }

    public function test_createOrderInfo_ok()
    {
        // arrange
        $orderController = $this->getMockBuilder(OrderController::class)
            ->setMethods(array('validOrderInfo'))
            ->disableOriginalConstructor()
            ->getMock();
        $orderController
            ->expects($this->exactly(2))
            ->method('validOrderInfo')
            ->willReturn(null);

        $orderId = "OrderTest54321";

        $order = new Order();
        $order->setId($orderId);
        $order->save();

        $parameters = array(
            'buyer'    => array(
                'name'    => 'buyer',
                'email'   => 'buyer@test.com',
                'phone'   => '0912345678',
                'zip'     => '231',
                'address' => '台北市中正區延平南路96號',
            ),
            'receiver' => array(
                'name'    => 'receiver',
                'email'   => 'receiver@test.com',
                'phone'   => '0987654321',
                'zip'     => '132',
                'address' => '台北市中正區重慶南路一段122號',
            ),
            'pay_after' => 'ATM'
        );

        // act
        $this->callObjectMethod($orderController, 'createOrderInfo', $orderId, $parameters);
        $orderInfoData = OrderInfoQuery::create()
            ->findPk($orderId);

        // assert
        $this->assertEquals($parameters['buyer']['name'], $orderInfoData->getBuyerName());
        $this->assertEquals($parameters['receiver']['name'], $orderInfoData->getReceiverName());
        $this->assertEquals($parameters['pay_after'], $orderInfoData->getPayAfter());

        // 刪掉測試用資料
        $orderInfoData->delete();
        $order->delete();
    }

    public function test_createOrderInfo_ok_empty_payafter()
    {
        // arrange
        $orderController = $this->getMockBuilder(OrderController::class)
            ->setMethods(array('validOrderInfo'))
            ->disableOriginalConstructor()
            ->getMock();
        $orderController
            ->expects($this->exactly(2))
            ->method('validOrderInfo')
            ->willReturn(null);

        $orderId = "OrderTest54321";

        $order = new Order();
        $order->setId($orderId);
        $order->save();

        $parameters = array(
            'buyer'    => array(
                'name'    => 'buyer',
                'email'   => 'buyer@test.com',
                'phone'   => '0912345678',
                'zip'     => '231',
                'address' => '台北市中正區延平南路96號',
            ),
            'receiver' => array(
                'name'    => 'receiver',
                'email'   => 'receiver@test.com',
                'phone'   => '0987654321',
                'zip'     => '132',
                'address' => '台北市中正區重慶南路一段122號',
            ),
            'pay_after' => ''
        );

        // act
        $this->callObjectMethod($orderController, 'createOrderInfo', $orderId, $parameters);
        $orderInfoData = OrderInfoQuery::create()
            ->findPk($orderId);

        // assert
        $this->assertEquals($parameters['buyer']['name'], $orderInfoData->getBuyerName());
        $this->assertEquals($parameters['receiver']['name'], $orderInfoData->getReceiverName());
        $this->assertEquals($parameters['pay_after'], null);

        // 刪掉測試用資料
        $orderInfoData->delete();
        $order->delete();
    }

    public function test_validOrderInfo_blank_name()
    {
        // arrange
        $orderController = $this->getMockBuilder(OrderController::class)
            ->disableOriginalConstructor()
            ->setMethods()
            ->getMock();
        $validator = $this->client->getContainer()->get('validator');
        $errorConvertor = $this->client->getContainer()->get('api.validator.error.convertor');

        $this->setObjectAttribute($orderController, 'validator', $validator);
        $this->setObjectAttribute($orderController, 'errorConvertor', $errorConvertor);

        $info = array(
            'name' => '',
            'email' => 'Bad Email',
            'phone' => '09132456',
            'zip' => '12345',
            'address' => '台中市大大路',
        );

        // act
        $result = $this->callObjectMethod($orderController, 'validOrderInfo', $info);

        // assert
        $this->assertTrue(is_object($result));
        $this->assertEquals(Response::HTTP_BAD_REQUEST, $this->getObjectAttribute($result, 'statusCode'));
    }

    public function test_validOrderInfo_bad_email()
    {
        // arrange
        $orderController = $this->getMockBuilder(OrderController::class)
            ->disableOriginalConstructor()
            ->setMethods()
            ->getMock();
        $validator = $this->client->getContainer()->get('validator');
        $errorConvertor = $this->client->getContainer()->get('api.validator.error.convertor');

        $this->setObjectAttribute($orderController, 'validator', $validator);
        $this->setObjectAttribute($orderController, 'errorConvertor', $errorConvertor);

        $info = array(
            'name' => 'Name',
            'email' => 'Bad Email',
            'phone' => '09132456',
            'zip' => '12345',
            'address' => '台中市大大路',
        );

        // act
        $result = $this->callObjectMethod($orderController, 'validOrderInfo', $info);

        // assert
        $this->assertTrue(is_object($result));
        $this->assertEquals(Response::HTTP_BAD_REQUEST, $this->getObjectAttribute($result, 'statusCode'));
    }

    public function test_validOrderInfo_blank_phone()
    {
        // arrange
        $orderController = $this->getMockBuilder(OrderController::class)
            ->disableOriginalConstructor()
            ->setMethods()
            ->getMock();
        $validator = $this->client->getContainer()->get('validator');
        $errorConvertor = $this->client->getContainer()->get('api.validator.error.convertor');

        $this->setObjectAttribute($orderController, 'validator', $validator);
        $this->setObjectAttribute($orderController, 'errorConvertor', $errorConvertor);

        $info = array(
            'name' => 'Name',
            'email' => 'yang@dgfactor.com.tw',
            'phone' => '',
            'zip' => '12345',
            'address' => '台中市大大路',
        );

        // act
        $result = $this->callObjectMethod($orderController, 'validOrderInfo', $info);

        // assert
        $this->assertTrue(is_object($result));
        $this->assertEquals(Response::HTTP_BAD_REQUEST, $this->getObjectAttribute($result, 'statusCode'));
    }

    public function test_validOrderInfo_blank_address()
    {
        // arrange
        $orderController = $this->getMockBuilder(OrderController::class)
            ->disableOriginalConstructor()
            ->setMethods()
            ->getMock();
        $validator = $this->client->getContainer()->get('validator');
        $errorConvertor = $this->client->getContainer()->get('api.validator.error.convertor');

        $this->setObjectAttribute($orderController, 'validator', $validator);
        $this->setObjectAttribute($orderController, 'errorConvertor', $errorConvertor);

        $info = array(
            'name' => 'Name',
            'email' => 'yang@dgfactor.com.tw',
            'phone' => '09132456',
            'zip' => '12345',
            'address' => '',
        );

        // act
        $result = $this->callObjectMethod($orderController, 'validOrderInfo', $info);

        // assert
        $this->assertTrue(is_object($result));
        $this->assertEquals(Response::HTTP_BAD_REQUEST, $this->getObjectAttribute($result, 'statusCode'));
    }

    public function test_validOrderInfo_ok()
    {
        // arrange
        $orderController = $this->getMockBuilder(OrderController::class)
            ->disableOriginalConstructor()
            ->setMethods()
            ->getMock();
        $validator = $this->client->getContainer()->get('validator');
        $errorConvertor = $this->client->getContainer()->get('api.validator.error.convertor');

        $this->setObjectAttribute($orderController, 'validator', $validator);
        $this->setObjectAttribute($orderController, 'errorConvertor', $errorConvertor);

        $info = array(
            'name' => 'Name',
            'email' => 'yang@dgfactor.com.tw',
            'phone' => '09132456',
            'zip' => '12345',
            'address' => '台中市大大路',
        );

        // act
        $result = $this->callObjectMethod($orderController, 'validOrderInfo', $info);

        // assert
        $this->assertTrue(is_null($result));
    }

    public function test_validOrderInfo_allow_extra_fields()
    {
        // arrange
        $orderController = $this->getMockBuilder(OrderController::class)
            ->disableOriginalConstructor()
            ->setMethods()
            ->getMock();
        $validator = $this->client->getContainer()->get('validator');
        $errorConvertor = $this->client->getContainer()->get('api.validator.error.convertor');

        $this->setObjectAttribute($orderController, 'validator', $validator);
        $this->setObjectAttribute($orderController, 'errorConvertor', $errorConvertor);

        $info = array(
            'name' => 'Name',
            'email' => 'yang@dgfactor.com.tw',
            'phone' => '09132456',
            'zip' => '12345',
            'address' => '台中市大大路',
            'phone2' => '123',
            'good' => true
        );

        // act
        $result = $this->callObjectMethod($orderController, 'validOrderInfo', $info);

        // assert
        $this->assertTrue(is_null($result));
    }
    
    protected function makeRange($start, $end)
    {
        $array = array();
        foreach(range($start, $end) as $i){
            $array[] = sprintf('P00%02d', $i);
        }
        return $array;
    }

    public function test_runDiscount()
    {
        //arrange
        $con = \Propel::getConnection();
        $order = new Order();
        $controller = $this->getMockBuilder(OrderController::class)
            ->disableOriginalConstructor()
            ->setMethods(null)
            ->getMock();
        $discontBuilder = $this->getMockBuilder(DiscountBuilder::class)
            ->disableOriginalConstructor()
            ->setMethods(array('run'))
            ->getMock();
        $discontBuilder
            ->expects($this->once())
            ->method('run')
            ->willReturnCallback(function(Order $orderForTest, \PropelPDO $conForTest = null) use($con, $order){
                $this->assertEquals($con, $conForTest);
                $this->assertEquals($order, $orderForTest);
            });

        $eventDispatcher = $this->getMockBuilder(EventDispatcher::class)
            ->disableOriginalConstructor()
            ->setMethods(array('dispatch'))
            ->getMock();
        $eventDispatcher
            ->expects($this->once())
            ->method('dispatch')
            ->willReturnCallback(function($eventName, DiscountChainEvent $event) use($discontBuilder){
                $this->assertEquals(DiscountChainEvent::EVENT_NAME, $eventName);
                $this->assertEquals($discontBuilder, $event->getDiscountBuilder());
            });
        $this->setObjectAttribute($controller, 'eventDispatcher', $eventDispatcher);
        $this->setObjectAttribute($controller, 'discountBuilder', $discontBuilder);

        //act
        $this->callObjectMethod($controller, 'runDiscount', $order, $con);

        //assert
    }

    public function test_createOrder()
    {
        //arrange
        $memberId = 1234567;
        $con = \Propel::getConnection();
        $params = array(
            'order' => array(
                '1' => 5,
                '2' => 6,
                '3' => 7,
            )
        );
        $errors = array();

        $order = $this->getMockBuilder(Order::class)
            ->disableOriginalConstructor()
            ->setMethods(array('syncAmount'))
            ->getMock();
        $order
            ->expects($this->once())
            ->method('syncAmount')
            ->willReturn(null);

        $controller = $this->getMockBuilder(OrderController::class)
            ->disableOriginalConstructor()
            ->setMethods(array('runDiscount', 'findMemberIdOrCreate'))
            ->getMock();

        $controller
            ->expects($this->once())
            ->method('findMemberIdOrCreate')
            ->willReturn($memberId);

        $controller
            ->expects($this->never())
            ->method('runDiscount')
            ->willReturnCallback(function ($orderForTest, $conForTest) use ($order, $con) {
                $this->assertEquals($order, $orderForTest);
                $this->assertEquals($con, $conForTest);
            });


        $orderBuilder = $this->getMockBuilder(OrderBuilder::class)
            ->disableOriginalConstructor()
            ->setMethods(array('create', 'add', 'getOrder', 'getErrors'))
            ->getMock();
        $orderBuilder
            ->expects($this->atLeastOnce())
            ->method('getErrors')
            ->willReturn($errors);

        $orderBuilder
            ->expects($this->once())
            ->method('create')
            ->willReturnCallback(function($memberIdForTest, $conForTest) use($orderBuilder, $memberId, $con){
                $this->assertEquals($memberId, $memberIdForTest);
                $this->assertEquals($con, $conForTest);
                return $orderBuilder;
            });
        $orderBuilder
            ->expects($this->exactly(count($params['order'])))
            ->method('add')
            ->willReturn($orderBuilder)
            ;
        $orderBuilder
            ->expects($this->once())
            ->method('getOrder')
            ->willReturn($order)
            ;
        $this->setObjectAttribute($controller, 'orderBuilder', $orderBuilder);

        //act
        $result = $this->callObjectMethod($controller, 'createOrder', $memberId, $params, $con);

        //assert
        $this->assertEquals($order, $result);
    }

    /**
     *
     * @expectedException \Backend\BaseBundle\Exception\ErrorResponseException
     */
    public function test_createOrder_hasError()
    {
        //arrange
        $memberId = 1234567;
        $con = \Propel::getConnection();
        $params = array(
            'order' => array(
                '1' => 5,
                '2' => 6,
                '3' => 7,
            )
        );
        $errors = array(
            'error1',
            'error2'
        );

        $order = $this->getMockBuilder(Order::class)
            ->disableOriginalConstructor()
            ->setMethods(array('syncAmount'))
            ->getMock();

        $order
            ->expects($this->never())
            ->method('syncAmount')
            ->willReturn(null);

        $controller = $this->getMockBuilder(OrderController::class)
            ->disableOriginalConstructor()
            ->setMethods(array('runDiscount', 'findMemberIdOrCreate'))
            ->getMock();

        $controller
            ->expects($this->once())
            ->method('findMemberIdOrCreate')
            ->willReturn($memberId);

        $controller
            ->expects($this->never())
            ->method('runDiscount')
            ->willReturnCallback(function($orderForTest, $conForTest) use($order, $con){
                $this->assertEquals($order, $orderForTest);
                $this->assertEquals($con, $conForTest);
            });

        $orderBuilder = $this->getMockBuilder(OrderBuilder::class)
            ->disableOriginalConstructor()
            ->setMethods(array('create', 'add', 'getOrder', 'getErrors'))
            ->getMock();

        $orderBuilder
            ->expects($this->atLeastOnce())
            ->method('getErrors')
            ->willReturn($errors);

        $orderBuilder
            ->expects($this->once())
            ->method('create')
            ->willReturnCallback(function($memberIdForTest, $conForTest) use($orderBuilder, $memberId, $con){
                $this->assertEquals($memberId, $memberIdForTest);
                $this->assertEquals($con, $conForTest);
                return $orderBuilder;
            });
        $orderBuilder
            ->expects($this->exactly(count($params['order'])))
            ->method('add')
            ->willReturn($orderBuilder)
        ;
        $orderBuilder
            ->expects($this->never())
            ->method('getOrder')
            ->willReturn($order)
        ;
        $this->setObjectAttribute($controller, 'orderBuilder', $orderBuilder);

        //act
        $result = $this->callObjectMethod($controller, 'createOrder', $memberId, $params, $con);

        //assert
        $this->assertEquals($order, $result);
    }
}