<?php

namespace Widget\OrderBundle\EventListener;

use Backend\BaseBundle\Service\CustomMailerService;
use Backend\BaseBundle\Service\SendMailService;
use Backend\BaseBundle\SiteConfig\SiteConfigBuilder;
use JMS\DiExtraBundle\Annotation as DI;
use JMS\DiExtraBundle\Annotation\Inject;
use JMS\DiExtraBundle\Annotation\InjectParams;
use Widget\MemberBundle\Model\MemberQuery;
use Widget\OrderBundle\Event\ReceiveCreateOrderEvent;
use Widget\OrderBundle\Model\OrderQuery;
use Widget\OrderBundle\Model\OrderStatusQuery;
use Widget\PaymentBundle\Event\ReceiveOrderEvent;
use Widget\PaymentBundle\Event\ReceivePaymentnumberEvent;

/**
 * @DI\Service()
 */
class OrderSendMailListener
{
    /** @var  CustomMailerService */
    protected $customMailerService;

    /** @var  SiteConfigBuilder */
    protected $siteConfigBuilder;

    /** @var  SendMailService */
    protected $sendMailService;

    /**
     * @InjectParams({
     *    "customMailerService" = @Inject("site_custom_mailer"),
     * })
     */
    public function injectCustomMailerService(CustomMailerService $customMailerService)
    {
        $this->customMailerService = $customMailerService;
    }

    /**
     * @InjectParams({
     *    "siteConfigBuilder" = @Inject("backend_base.site_config_builder"),
     * })
     */
    public function injectSiteConfigBuilder(SiteConfigBuilder $siteConfigBuilder)
    {
        $this->siteConfigBuilder = $siteConfigBuilder;
    }

    /**
     * @InjectParams({
     *    "sendMailService" = @Inject("backend_base.send_mail"),
     *     })
     */
    public function injectSendMailService(SendMailService $sendMailService)
    {
        $this->sendMailService  = $sendMailService;
    }

    protected function twigBuilder(array $templates)
    {
        return new \Twig_Environment(new \Twig_Loader_Array($templates));
    }

    /**
     * 付款成功寄信通知
     * @DI\Observe("event.widget.payment.receive.order")
     */
    public function onReceiveOrderStatusEvent(ReceiveOrderEvent $event)
    {
        if(!$event->getStatus()){
            $event->stopPropagation();
            return;
        }

        $order = OrderQuery::create()->findPk($event->getOrdernumber());
        $orderStatus = $order->getOrderStatus();
        $buyer = $order->getOrderInfo();
        $member = MemberQuery::create()->findOneById($order->getMemberId());

        $group = array(
            'element' => 'widget_order',
            'name' => 'PaymentSuccess'
        );
        $to = array(
            'email' => array(
                $member->getEmail(),
                $buyer->getBuyerEmail(),
            ),
            'name' => array(
                $member->getName(),
                $buyer->getBuyerName(),
            )
        );
        $templateName = "payment_success";
        $templateSubject = array(
            'ordernumber' => $order->getId(),
            'orderStatus' => $orderStatus->getName()
        );
        $templateBody =  array(
            'ordernumber' => $order->getId(),
            'amountForPay' => $order->getAmountForPay(),
            'amount' => $order->getAmount(),
            'paidAt' => $order->getPayment()->getPaymentPaidAt('%Y-%m-%d %T'),
            'paymentType' => $order->getPayment()->getPaymentType(),
        );

        $this->sendMailService->send($group, $to, $templateName, $templateSubject, $templateBody);
    }

    /**
     * 繳款通知寄信
     * @DI\Observe("event.widget.payment.receive.paymentnumber")
     */
    public function onReceivePaymentnumberEvent(ReceivePaymentnumberEvent $event)
    {
        $order = OrderQuery::create()->findPk($event->getOrdernumber());
        $orderStatus = OrderStatusQuery::create()
            ->findOneById($order->getOrderStatusId());
        if(!$order->getOrderStatus() || !$order->getOrderStatus()->getStatus()){
            $event->stopPropagation();
            return;
        }
        $buyer = $order->getOrderInfo();
        $member = MemberQuery::create()
            ->findOneById($order->getMemberId());

        $group = array(
            'element' => 'widget_order',
            'name' => 'Payment'
        );
        $to = array(
            'email' => array(
                $member->getEmail(),
                $buyer->getBuyerEmail()
            ),
            'name' => array(
                $member->getName(),
                $buyer->getBuyerName()
            )
        );
        $templateName = "payment";
        $templateSubject = array(
            'ordernumber' => $order->getId(),
            'orderStatus' => $orderStatus->getName(),
        );
        $templateBody = array(
            'ordernumber' => $order->getId(),
            'paymentnumber' =>$event->getPaymentnumber(),
            'amountForPay' => $order->getAmountForPay(),
            'amount' => $order->getAmount(),
            'createdAt' => $order->getCreatedAt(),
            'expiredAt' => $event->getExpiredAt(),
            'paymentType' => $order->getPayment()->getPaymentType(),
        );
        $this->sendMailService->send($group, $to, $templateName, $templateSubject, $templateBody);
    }

    /**
     * 建立訂單寄信通知
     * @DI\Observe(ReceiveCreateOrderEvent::EVENT_NAME)
     */
    public function onReceiveCreateOrderEvent(ReceiveCreateOrderEvent $event)
    {
        $order = OrderQuery::create()->findPk($event->getOrder()->getId());
        $orderStatus = OrderStatusQuery::create()
            ->findOneById($order->getOrderStatusId());
        if(!$order->getOrderStatus()){
            $event->stopPropagation();
            return;
        }
        $member = MemberQuery::create()
            ->findOneById($order->getMemberId());
        $buyer = $order->getOrderInfo();

        $group = array(
            'element' => 'widget_order',
            'name' => 'statusChange'
        );
        $to = array(
            'email' => array(
                $member->getEmail(),
                $buyer->getBuyerEmail()
            ),
            'name' => array(
                $member->getName(),
                $buyer->getBuyerName()
            )
        );
        $templateName = "status_change";
        $templateSubject = array(
            'ordernumber' => $order->getId(),
            'orderStatus' => $orderStatus->getName(),
        );
        $templateBody = array(
            'ordernumber' => $order->getId(),
            'orderStatus' => $orderStatus->getName(),
            'amountForPay' => $order->getAmountForPay(),
            'amount' => $order->getAmount(),
            'createdAt' => $order->getCreatedAt(),
        );
        $this->sendMailService->send($group, $to, $templateName, $templateSubject, $templateBody);
    }
}
