<?php
namespace Widget\OrderBundle\Controller\BackendAPI;

use Backend\BaseBundle\Controller\BackendAPI\BaseBackendAPIController;
use Backend\BaseBundle\Form\Type\APIFormTypeItem;
use Backend\BaseBundle\Propel\I18n;
use Sensio\Bundle\FrameworkExtraBundle\Configuration\Method;
use Sensio\Bundle\FrameworkExtraBundle\Configuration\Route;
use Sensio\Bundle\FrameworkExtraBundle\Configuration\Security;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\Validator\Constraints\Callback;
use Symfony\Component\Validator\Constraints\NotBlank;
use Symfony\Component\Validator\Context\ExecutionContextInterface;
use Widget\OrderBundle\Model\OrderStatus;
use Widget\OrderBundle\Model\OrderStatusPeer;
use Widget\OrderBundle\Model\OrderStatusQuery;


/**
 * @Route("/orderstatus")
 * @Security("has_role_or_superadmin('ROLE_ORDER')")
 */
class OrderStatusController extends BaseBackendAPIController
{

    /**
     * 欄位表單設定
     * 主要是USER送過來的欄位
     * @return APIFormTypeItem[]
     */
    protected function getFormConfig()
    {
        return array(
            (new APIFormTypeItem('name'))->setOptions(array(
                'constraints' => array(
                    new NotBlank(array(
                        'message' => 'error.order_status.name.missing_field',
                    )),
                ),
            )),
            new APIFormTypeItem('status'),
            new APIFormTypeItem('default_status'),
        );
    }

    /**
     * 新增
     * @Route("s")
     * @Method({"POST"})
     * @Security("has_role_or_superadmin('ROLE_ORDER_WRITE')")
     */
    public function createAction(Request $request)
    {
        $orderStatus = new OrderStatus();
        if ($orderStatus instanceof I18n){
            $locale = $request->query->get('_locale');
            $orderStatus->setLocale($locale);
        }
        $serializer = $this->doProcessForm($orderStatus, $request->getContent());
        $this->setDefaultStatus($orderStatus);
        return $serializer;
    }

    /**
     * 搜尋
     * @Route("s")
     * @Method({"GET"})
     * @Security("has_role_or_superadmin('ROLE_ORDER_READ')")
     */
    public function searchAction(Request $request)
    {
        $query = OrderStatusQuery::create();
        if ($query instanceof I18n){
            $locale = $request->query->get('_locale');
            $query->joinOrderStatusI18n($locale);
        }
        return $this->doSearch($request->query->all(), $query->distinct(), OrderStatusPeer::class);
    }

    /**
     * 讀取
     * @Route("/{id}")
     * @Method({"GET"})
     * @Security("has_role_or_superadmin('ROLE_ORDER_READ')")
     */
    public function readAction(Request $request, OrderStatus $orderStatus)
    {
        if ($orderStatus instanceof I18n){
            $locale = $request->query->get('_locale');
            $orderStatus->setLocale($locale);
        }
        return $this->createJsonSerializeResponse($orderStatus, array('detail'));
    }

    /**
     * 更新
     * @Route("/{id}")
     * @Method({"PUT"})
     * @Security("has_role_or_superadmin('ROLE_ORDER_WRITE')")
     */
    public function updateAction(Request $request, OrderStatus $orderStatus)
    {
        if ($orderStatus instanceof I18n){
            $locale = $request->query->get('_locale');
            $orderStatus->setLocale($locale);
        }
        $serializer = $this->doProcessForm($orderStatus, $request->getContent());
        $this->setDefaultStatus($orderStatus);
        return $serializer;
    }

    /**
     * 刪除
     * @Route("/{id}")
     * @Method({"DELETE"})
     * @Security("has_role_or_superadmin('ROLE_ORDER_WRITE')")
     */
    public function deleteAction(OrderStatus $orderStatus)
    {
        $this->deleteObject($orderStatus);
        return $this->createJsonResponse();
    }

    /**
     * @Route("s/all")
     * @Method({"GET"})
     * @Security("has_role_or_superadmin('ROLE_ORDER_READ')")
     */
    public function listAction(Request $request)
    {
        $orderStatus = OrderStatusQuery::create()->find();
        return $this->createJsonSerializeResponse($orderStatus, array('list'));

    }

    /**
     * 預設訂單狀態為是 將其他改為否
     * @param OrderStatus $orderStatus
     */
    protected function setDefaultStatus($orderStatus)
    {
        if ($orderStatus->getDefaultStatus()) {
            /** @var OrderStatus $status */
            foreach (OrderStatusQuery::create()->filterByDefaultStatus(true)->find() as $status) {
                $status->setDefaultStatus(false)->save();
            }

            OrderStatusQuery::create()->findPk($orderStatus->getId())->setDefaultStatus(true)->save();
        }
    }
}