<?php
namespace Widget\OrderBundle\Tests\Controller\BackendAPI;


use Backend\BaseBundle\Model\SiteUserQuery;
use Backend\BaseBundle\Tests\Fixture\BackendWebTestCase;
use Symfony\Component\EventDispatcher\EventDispatcherInterface;
use Widget\OrderBundle\Event\OrderStatusChangeEvent;
use Widget\OrderBundle\Event\ReceiveCreateOrderEvent;
use Widget\OrderBundle\Model\Order;
use Widget\OrderBundle\Model\OrderQuery;
use Widget\OrderBundle\Model\OrderStatusQuery;

class OrderControllerTest extends BackendWebTestCase
{
    public function test_updateAction()
    {
        //arrange
        $loginName = 'admin';
        $origin = 'http://localhost';
        $token = $this->createToken(SiteUserQuery::create()->findOneByLoginName($loginName), $origin);

        $params = array(
            'payment_status' => 'p',
        );

        $order = OrderQuery::create()->findOneById('P0001');
        $originOrderArray = $order->toArray();

        //act
        $this->client->request(
            'PUT',
            $this->generateUrl('widget_order_backendapi_order_update', array('id' => $order->getId())),
            array(),
            array(),
            array(
                'HTTP_ORIGIN' => $origin,
                'HTTP_AUTHORIZATION' => "Bearer $token",
            ),
            json_encode($params)
        );
        $response = $this->client->getResponse();

        $order->reload(true);

        //assert
        $this->assertTrue($response->isOk());
        $this->assertEquals($params['payment_status'], $order->getPaymentStatus());

        //reset
        $order->fromArray($originOrderArray);
        $order->save();
    }

    public function test_updateAction_change_status()
    {
        //arrange
        $loginName = 'admin';
        $origin = 'http://localhost';
        $token = $this->createToken(SiteUserQuery::create()->findOneByLoginName($loginName), $origin);
        $newOrderStatus = OrderStatusQuery::create()->filterByStatus(false)->findOne();
        $hasTrigger = false;
        $params = array(
            'payment_status' => 'p',
            'order_status_id' => $newOrderStatus->getId()
        );

        $order = OrderQuery::create()->findOneById('P0001');
        $originOrderArray = $order->toArray();

        /** @var EventDispatcherInterface $eventDispatcher */
        $eventDispatcher = $this->client->getContainer()->get('event_dispatcher');
        $eventDispatcher->addListener(OrderStatusChangeEvent::EVENT_NAME, function(OrderStatusChangeEvent $event) use(&$hasTrigger, $order){
            $hasTrigger = true;
            $this->assertInstanceOf(Order::class, $event->getOrder());
            $this->assertInstanceOf(\PropelPDO::class, $event->getCon());
            $event->stopPropagation();
        }, 99999);

        //act
        $this->client->request(
            'PUT',
            $this->generateUrl('widget_order_backendapi_order_update', array('id' => $order->getId())),
            array(),
            array(),
            array(
                'HTTP_ORIGIN' => $origin,
                'HTTP_AUTHORIZATION' => "Bearer $token",
            ),
            json_encode($params)
        );
        $response = $this->client->getResponse();
        $order->reload(true);

        //assert
        $this->assertTrue($response->isOk());
        $this->assertTrue($hasTrigger);
        $this->assertEquals($params['payment_status'], $order->getPaymentStatus());
        $this->assertEquals($params['order_status_id'], $order->getOrderStatusId());

        //reset
        $order->fromArray($originOrderArray);
        $order->save();
    }

    public function test_searchAction()
    {
        //arrange
        $loginName = 'admin';
        $origin = 'http://localhost';
        $token = $this->createToken(SiteUserQuery::create()->findOneByLoginName($loginName), $origin);
        $params = array(
            'page' => 1,
            'sort' => array(
                'id' => 'asc'
            ),
            'search' => array(
                'id' => 'P0000',
            ),
        );

        //act
        $this->client->request(
            'GET',
            $this->generateUrl('widget_order_backendapi_order_search', $params),
            array(),
            array(),
            array(
                'HTTP_ORIGIN' => $origin,
                'HTTP_AUTHORIZATION' => "Bearer $token",
            )
        );
        $response = $this->client->getResponse();
        $result = json_decode($response->getContent(), true);

        //assert
        $this->assertTrue($response->isOk());
        $this->assertEquals(1, $result['pager']['page']);
        $this->assertEquals(1, $result['pager']['pages']);
        $this->assertEquals(1, $result['pager']['rows']);
        $this->assertEquals(1, count($result['data']));
        $this->assertArrayHasKey('id', $result['data'][0]);
        $this->assertArrayHasKey('member_id', $result['data'][0]);
        $this->assertArrayHasKey('amount', $result['data'][0]);
        $this->assertArrayHasKey('amount_for_pay', $result['data'][0]);
        $this->assertArrayHasKey('order_status', $result['data'][0]);
        $this->assertArrayHasKey('order_items', $result['data'][0]);
        $this->assertArrayHasKey('created_at', $result['data'][0]);
        $this->assertArrayHasKey('updated_at', $result['data'][0]);
        $this->assertArrayHasKey('payment', $result['data'][0]);
        $this->assertArrayHasKey('order_note', $result['data'][0]);
        $this->assertArrayHasKey('order_info', $result['data'][0]);
    }

    public function test_searchAction_buyer_name()
    {
        //arrange
        $loginName = 'admin';
        $origin = 'http://localhost';
        $token = $this->createToken(SiteUserQuery::create()->findOneByLoginName($loginName), $origin);
        $params = array(
            'page' => 1,
            'sort' => array(
                'created_at' => 'desc'
            ),
            'search' => array(
                'order_info.buyer_name' => 'bubble',
            ),
        );

        //act
        $this->client->request(
            'GET',
            $this->generateUrl('widget_order_backendapi_order_search', $params),
            array(),
            array(),
            array(
                'HTTP_ORIGIN' => $origin,
                'HTTP_AUTHORIZATION' => "Bearer $token",
            )
        );
        $response = $this->client->getResponse();
        $result = json_decode($response->getContent(), true);

        //assert
        $this->assertTrue($response->isOk());
        $this->assertEquals(1, $result['pager']['page']);
        $this->assertEquals(1, $result['pager']['pages']);
        $this->assertEquals(1, $result['pager']['rows']);
        $this->assertEquals(1, count($result['data']));
        $this->assertArrayHasKey('id', $result['data'][0]);
        $this->assertArrayHasKey('member_id', $result['data'][0]);
        $this->assertArrayHasKey('amount', $result['data'][0]);
        $this->assertArrayHasKey('amount_for_pay', $result['data'][0]);
        $this->assertArrayHasKey('order_status', $result['data'][0]);
        $this->assertArrayHasKey('order_items', $result['data'][0]);
        $this->assertArrayHasKey('created_at', $result['data'][0]);
        $this->assertArrayHasKey('updated_at', $result['data'][0]);
        $this->assertArrayHasKey('payment', $result['data'][0]);
        $this->assertArrayHasKey('order_note', $result['data'][0]);
        $this->assertArrayHasKey('order_info', $result['data'][0]);
    }

    public function test_readAction()
    {
        //arrange
        $loginName = 'admin';
        $origin = 'http://localhost';
        $user = SiteUserQuery::create()->findOneByLoginName($loginName);
        $token = $this->createToken($user, $origin);

        $order = OrderQuery::create()->findOne();

        //act
        $this->client->request(
            'GET',
            $this->generateUrl('widget_order_backendapi_order_read', array('id'=>$order->getId())),
            array(),
            array(),
            array(
                'HTTP_ORIGIN' => $origin,
                'HTTP_AUTHORIZATION' => "Bearer $token",
            )
        );
        $response = $this->client->getResponse();

        //assert
        $this->assertTrue($response->isOk());

        //reset
    }

    public function test_sendMailAction()
    {
        //arrange
        $loginName = 'admin';
        $origin = 'http://localhost';
        $user = SiteUserQuery::create()->findOneByLoginName($loginName);
        $token = $this->createToken($user, $origin);

        $order = OrderQuery::create()->findOne();
        $orderId = $order->getId();
        $eventDispatched = false;
        $dispatcher = $this->client->getContainer()->get('event_dispatcher');
        $dispatcher->addListener(ReceiveCreateOrderEvent::EVENT_NAME, function(ReceiveCreateOrderEvent $event) use(&$eventDispatched, $orderId){
            $event->stopPropagation();
            $this->assertEquals($orderId, $event->getOrder()->getId());
            $eventDispatched = true;
        }, 99999);
        //act
        $this->client->request(
            'GET',
            $this->generateUrl('widget_order_backendapi_order_sendmail', array('id'=>$order->getId())),
            array(),
            array(),
            array(
                'HTTP_ORIGIN' => $origin,
                'HTTP_AUTHORIZATION' => "Bearer $token",
            )
        );
        $response = $this->client->getResponse();

        //assert
        $this->assertTrue($response->isOk());
        $this->assertTrue($eventDispatched);
    }

}