<?php

namespace Widget\OrderBundle\Controller\BackendAPI;

use Backend\BaseBundle\Controller\API\BaseController;
use Sensio\Bundle\FrameworkExtraBundle\Configuration\Method;
use Sensio\Bundle\FrameworkExtraBundle\Configuration\Route;
use Sensio\Bundle\FrameworkExtraBundle\Configuration\Security;
use Widget\MemberBundle\Model\MemberPeer;
use Widget\MemberBundle\Model\MemberQuery;
use Widget\OrderBundle\Model\Order;
use Widget\OrderBundle\Model\OrderPeer;
use Widget\OrderBundle\Model\OrderQuery;
use Widget\OrderBundle\Model\OrderStatus;
use Widget\OrderBundle\Model\OrderStatusQuery;
use Widget\PaymentBundle\Model\PaymentPeer;
use Widget\PaymentBundle\Payment\Payment;

/**
 * @Route("/order")
 * @Security("has_role_or_superadmin('ROLE_ORDER')")
 */
class OrderDashboardController extends BaseController
{
    /**
     * @Route("s/dashboard")
     * @Method({"GET"})
     * @Security("has_role_or_superadmin('ROLE_ORDER_READ')")
     */
    public function dashBoardAction()
    {
        $statisticsData = array(
            // 最新n筆訂單 OrderList
            "last" => $this->getLastOrder(),
            // 未付款訂單
            "hasNotPaid" => $this->getHasNotPaid(),

            // 付款失敗訂單
            "paidFail" => $this->getPaidFail(),

            // 各訂單狀態
            "statusCount" => $this->getStatusCont(),

            "topOrder" => $this->getTopOrder(),
        );

        return $this->createJsonResponse($statisticsData);
    }

    protected function getTimeIntervalTimestamp()
    {
        return strtotime("-1 month");
    }

    protected function getDashBoardLimit()
    {
        return 5;
    }

    protected function getLastOrder()
    {
        $orders = OrderQuery::create()
            ->joinWithOrderInfo()
            ->leftJoinWithOrderStatus()
            ->leftJoinWithPayment()
            ->filterByCreatedAt(array('min' => $this->getTimeIntervalTimestamp()))
            ->orderByCreatedAt(\Criteria::DESC)
            ->limit($this->getDashBoardLimit())
            ->find();
        return $this->getSerializerArray($orders);
    }

    protected function getHasNotPaid()
    {
        if (!class_exists(Payment::class)) {
            return null;
        }
        $orders = OrderQuery::create()
            ->joinWithOrderInfo()
            ->leftJoinWithOrderStatus()
            ->leftJoinWithPayment()
            ->usePaymentQuery()
                ->filterByPaymentStatus(PaymentPeer::PAYMENT_STATUS_INITIAL)
            ->endUse()
            ->orderByCreatedAt(\Criteria::DESC)
            ->limit($this->getDashBoardLimit())
            ->find();
        return $this->getSerializerArray($orders);
    }

    protected function getPaidFail()
    {
        $orders = OrderQuery::create()
            ->joinWithOrderInfo()
            ->leftJoinWithOrderStatus()
            ->leftJoinWithPayment()
            ->usePaymentQuery()
                ->filterByPaymentStatus(PaymentPeer::PAYMENT_STATUS_FAILED)
            ->endUse()
            ->orderByCreatedAt(\Criteria::DESC)
            ->limit($this->getDashBoardLimit())
            ->find();
        return $this->getSerializerArray($orders);
    }

    protected function getTopOrder()
    {
        $amountColumn = OrderPeer::AMOUNT;
        $amountForPayColumn = OrderPeer::AMOUNT_FOR_PAY;
        $memberIdColumn = OrderPeer::MEMBER_ID;
        $orders = OrderQuery::create()
            ->withColumn("SUM({$amountColumn})", 'amount_sum')
            ->withColumn("SUM({$amountForPayColumn})", 'amount_for_pay_sum')
            ->select(array($memberIdColumn, 'amount_sum', 'amount_for_pay_sum'))
            ->groupBy($memberIdColumn)
            ->orderBy('amount_sum', \Criteria::DESC)
            ->limit($this->getDashBoardLimit())
            ->find();

        $memberIds = array();
        foreach ($orders as $order){
            $memberIds[] = $order[$memberIdColumn];
        }
        $membersArray = MemberQuery::create()
            ->findPks($memberIds)
            ->toArray('id', false, \BasePeer::TYPE_FIELDNAME);
        $responseData = array();
        foreach ($orders as $order){
            $memberId = $order[$memberIdColumn];
            // 避免訂單有member_id member table 卻沒有 的狀態 但是應該不會發生
            if (! isset($membersArray[$memberId])) {
                continue;
            }
            $member = $membersArray[$memberId];
            $responseData[] = array(
                "id" => $member['id'],
                "email" => $member['email'],
                "name" => $member['name'],
                "amount" => $order['amount_sum'],
                "amount_for_pay" => $order['amount_for_pay_sum'],
            );
        }

        return $responseData;
    }

    protected function getStatusCont()
    {
        $orderStatusArray = OrderStatusQuery::create()
            ->find()
            ->toArray('id', false, \BasePeer::TYPE_FIELDNAME);
        $statusColumn = OrderPeer::ORDER_STATUS_ID;
        $orders = OrderQuery::create()
            ->withColumn("COUNT(*)", 'status_count')
            ->select(array($statusColumn))
            ->groupByOrderStatusId()
            ->find();
        $responseData = array();

        // 各狀態先設一個空的
        foreach ($orderStatusArray as $orderStatus){
            $responseData[$orderStatus['id']] = array(
                'name' => $orderStatus['name'],
                'count' => 0
            );
        }
        // 把統計資料塞進去
        foreach ($orders as $order){
            // null 就不統計了
            if (!$order[$statusColumn]) {
                continue;
            }
            $orderStatusId = $order[$statusColumn];
            $responseData[$orderStatusId] = array(
                'name' => $orderStatusArray[$orderStatusId]['name'],
                'count' => $order['status_count']
            );
        }

        return array_values($responseData);
    }

    protected function getSerializerArray($object)
    {
        return json_decode($this->createJsonSerializeResponse($object, array('list'))->getContent(), true);
    }

}