<?php
namespace Widget\OrderBundle\Controller\BackendAPI;

use Backend\BaseBundle\Controller\BackendAPI\BaseBackendAPIController;
use Backend\BaseBundle\Form\Type\APIFormTypeItem;
use JMS\DiExtraBundle\Annotation\Inject;
use Sensio\Bundle\FrameworkExtraBundle\Configuration\Method;
use Sensio\Bundle\FrameworkExtraBundle\Configuration\Route;
use Sensio\Bundle\FrameworkExtraBundle\Configuration\Security;
use Symfony\Component\EventDispatcher\EventDispatcherInterface;
use Symfony\Component\HttpFoundation\Request;
use Widget\OrderBundle\Event\OrderStatusChangeEvent;
use Widget\OrderBundle\Event\ReceiveCreateOrderEvent;
use Widget\OrderBundle\Model\Order;
use Widget\OrderBundle\Model\OrderPeer;
use Widget\OrderBundle\Model\OrderQuery;

/**
 * @Route("/order")
 * @Security("has_role_or_superadmin('ROLE_ORDER')")
 */
class OrderController extends BaseBackendAPIController
{

    /**
     * @Inject()
     * @var EventDispatcherInterface
     */
    protected $eventDispatcher;    
    
    /**
     * 欄位表單設定
     * 主要是USER送過來的欄位
     * @return APIFormTypeItem[]
     */
    protected function getFormConfig()
    {
        return array(
            new APIFormTypeItem('member_id'),
            new APIFormTypeItem('amount'),
            new APIFormTypeItem('amount_for_pay'),
            new APIFormTypeItem('payment_status'),
            new APIFormTypeItem('order_status_id'),
            new APIFormTypeItem('order_note'),
        );
    }

    /**
     * 新增
     * 後台訂單無需新增功能
     * @Route("s")
     * @Method({"POST"})
     * @Security("has_role_or_superadmin('ROLE_ORDER_WRITE')")
     */
    public function createAction(Request $request)
    {
//        return $this->doProcessForm(new Order(), $request->getContent());
    }

    /**
     * 搜尋
     * @Route("s")
     * @Method({"GET"})
     * @Security("has_role_or_superadmin('ROLE_ORDER_READ')")
     */
    public function searchAction(Request $request)
    {
        $query = OrderQuery::create()
            ->leftJoinWithOrderInfo()
            ->leftJoinWithOrderStatus()
            ->leftJoinWithPayment()
        ;
        return $this->doSearch($request->query->all(), $query->distinct(), OrderPeer::class);
    }

    /**
     * 讀取
     * @Route("/{id}")
     * @Method({"GET"})
     * @Security("has_role_or_superadmin('ROLE_ORDER_READ')")
     */
    public function readAction(Order $order)
    {
        return $this->createJsonSerializeResponse($order, array('detail'));
    }

    /**
     * 更新
     * @Route("/{id}")
     * @Method({"PUT"})
     * @Security("has_role_or_superadmin('ROLE_ORDER_WRITE')")
     */
    public function updateAction(Request $request, Order $order)
    {
        $oldOrderStatusId = $order->getOrderStatusId();
        $response = $this->doProcessForm($order, $request->getContent());
        
        $this->emitOrderStatusChangeEvent($oldOrderStatusId, $order);
        return $response;
    }

    /**
     * 寄通知信件
     * @Route("/send/{id}")
     * @Method({"GET"})
     * @Security("has_role_or_superadmin('ROLE_ORDER_WRITE')")
     */    
    public function sendMailAction(Order $order, Request $request)
    {
        $receiveCreatOrder = new ReceiveCreateOrderEvent($order);
        $this->eventDispatcher->dispatch(ReceiveCreateOrderEvent::EVENT_NAME, $receiveCreatOrder);
        return $this->createJsonResponse();
    }

    protected function emitOrderStatusChangeEvent($oldOrderStatusId,Order $order)
    {
        if ($oldOrderStatusId === $order->getOrderStatusId()) {
            return;
        }

        $con = \Propel::getConnection();
        $con->query("SET TRANSACTION ISOLATION LEVEL SERIALIZABLE;");

        $orderStatusChangeEvent = new OrderStatusChangeEvent($order, $con);
        /** @var EventDispatcherInterface $eventDispatcher */
        $eventDispatcher = $this->get('event_dispatcher');
        $eventDispatcher->dispatch(OrderStatusChangeEvent::EVENT_NAME, $orderStatusChangeEvent);
        $con->commit();
    }
    
}