<?php
namespace Widget\OrderBundle\Tests\Service;

use Backend\BaseBundle\Tests\Fixture\BaseTestCase;
use Widget\OrderBundle\Model\Order;
use Widget\OrderBundle\Service\Discount\DiscountInterface;
use Widget\OrderBundle\Service\DiscountBuilder;

class MockDiscount implements DiscountInterface
{

    public function process(Order $order, \PropelPDO $con = null)
    {
    }

    public function isProcessed()
    {
    }

    public function injectSiteId($siteId)
    {
    }
}

class DiscountBuilderTest extends BaseTestCase
{
    /**
     * @expectedException \TypeError
     */
    public function test_add_bad_object()
    {
        //arrange
        $builder = new DiscountBuilder();

        //act
        $builder->add(new \stdClass());

        //assert
        $this->fail();
    }

    public function test_add()
    {
        //arrange
        $builder = new DiscountBuilder();
        $discount = new MockDiscount();
        $discount1 = new MockDiscount();
        $discount2 = new MockDiscount();

        //act
        $builder
            ->add($discount)
            ->add($discount1)
            ->add($discount2);

        //assert
        $this->assertEquals(array(
            $discount,
            $discount1,
            $discount2
        ), $this->getObjectAttribute($builder, 'discounts'));
    }

    /**
     * @expectedException \TypeError
     */
    public function test_addExclude_bad_object()
    {
        //arrange
        $builder = new DiscountBuilder();

        //act
        $builder->addExclude(new \stdClass());

        //assert
        $this->fail();
    }

    public function test_addExclude()
    {
        //arrange
        $builder = new DiscountBuilder();
        $discount = new MockDiscount();
        $discount1 = new MockDiscount();
        $discount2 = new MockDiscount();

        //act
        $builder
            ->addExclude($discount)
            ->addExclude($discount1)
            ->addExclude($discount2);

        //assert
        $this->assertEquals(array(
            $discount,
            $discount1,
            $discount2
        ), $this->getObjectAttribute($builder, 'excludeDiscounts'));
    }

    public function test_run()
    {
        //arrange
        $executedDiscounts = array();
        $order = new Order();
        $discountBuilder = $this->getMockBuilder(DiscountBuilder::class)
            ->setMethods(array('runDiscounts', 'runExcludeDiscounts'))
            ->disableOriginalConstructor()
            ->getMock();
        $discountBuilder
            ->expects($this->once())
            ->method('runExcludeDiscounts')
            ->willReturnCallback(function(Order $orderForTest) use($order, &$executedDiscounts){
                $this->assertEquals($order, $orderForTest);
                $executedDiscounts[] = 'runExcludeDiscounts';
            });
        $discountBuilder
            ->expects($this->once())
            ->method('runDiscounts')
            ->willReturnCallback(function(Order $orderForTest) use($order, &$executedDiscounts){
                $this->assertEquals($order, $orderForTest);
                $executedDiscounts[] = 'runDiscounts';
            });

        //act
        $discountBuilder->run($order);

        //assert
        $this->assertEquals(array('runExcludeDiscounts', 'runDiscounts'), $executedDiscounts);
    }

    public function test_runDiscounts()
    {
        //arrange
        $order = new Order();
        $discount1 = $this->getMockBuilder(MockDiscount::class)->setMethods(array('process'))->getMock();
        $discount1
            ->expects($this->once())
            ->method('process')
            ->willReturnCallback(function(Order $orderForTest) use($order){
                $this->assertEquals($order, $orderForTest);
            });

        $discount2 = $this->getMockBuilder(MockDiscount::class)->setMethods(array('process'))->getMock();
        $discount2
            ->expects($this->once())
            ->method('process')
            ->willReturnCallback(function(Order $orderForTest) use($order){
                $this->assertEquals($order, $orderForTest);
            });

        $discount3 = $this->getMockBuilder(MockDiscount::class)->setMethods(array('process'))->getMock();
        $discount3
            ->expects($this->once())
            ->method('process')
            ->willReturnCallback(function(Order $orderForTest) use($order){
                $this->assertEquals($order, $orderForTest);
            });

        $discountBuilder = new DiscountBuilder();
        $this->setObjectAttribute($discountBuilder, 'discounts', array($discount1, $discount2, $discount3));

        //act
        $discountBuilder->run($order);

        //assert
    }

    protected function makeMockDiscount(Order $order, $isProcessed, $runs)
    {
        $discount = $this->getMockBuilder(MockDiscount::class)->setMethods(array('process', 'isProcessed'))->getMock();
        $discount
            ->expects($runs)
            ->method('process')
            ->willReturnCallback(function(Order $orderForTest) use($order) {
                $this->assertEquals($order, $orderForTest);
            });
        $discount
            ->expects(clone $runs)
            ->method('isProcessed')
            ->willReturn($isProcessed);
        return $discount;
    }

    public function test_runExcludeDiscounts()
    {
        //arrange
        $order = new Order();
        $discounts = array(
            $this->makeMockDiscount($order, false, $this->once()),
            $this->makeMockDiscount($order, false, $this->once()),
            $this->makeMockDiscount($order, false, $this->once()),
        );

        $discountBuilder = new DiscountBuilder();
        $this->setObjectAttribute($discountBuilder, 'excludeDiscounts', $discounts);

        //act
        $discountBuilder->run($order);

        //assert
    }

    public function test_runExcludeDiscounts_processed()
    {
        //arrange
        $order = new Order();
        $discounts = array(
            $this->makeMockDiscount($order, false, $this->once()),
            $this->makeMockDiscount($order, true, $this->once()),
            $this->makeMockDiscount($order, false, $this->never()),
        );

        $discountBuilder = new DiscountBuilder();
        $this->setObjectAttribute($discountBuilder, 'excludeDiscounts', $discounts);

        //act
        $discountBuilder->run($order);

        //assert
    }

}
