<?php
namespace Widget\OrderBundle\Tests\Model;

use Backend\BaseBundle\Tests\Fixture\BaseTestCase;
use Backend\BaseBundle\Tests\Fixture\BaseWebTestCase;
use Widget\OrderBundle\Model\Order;
use Widget\OrderBundle\Model\OrderItem;
use Widget\ProductBundle\Model\ProductQuery;

class OrderTest extends BaseWebTestCase
{

    /**
     * 1.mock需要call的function 之後false是表示先不要去做function裡面的事情
     * 2.之後塞一個假資料讓setId去call一次function，之後回傳setId做完function回傳的值
     * 3.之後再去call createOrderNumber function並給他假資料，測試是否有呼叫function，資料回傳則回傳一開始設定的假資料
     * 4.最後比對兩個function回傳的結果是否相同
     */
    public function test___construct()
    {
        //arrange
        $ordernumber = '123456';
        $order = $this->getMockBuilder(Order::class)
            ->setMethods(array('setId', 'createOrderNumber'))
            ->disableOriginalConstructor()
            ->getMock();
        $order
            ->expects($this->once())
            ->method('setId')
            ->willReturnCallback(function($id) use($order, $ordernumber){
                $this->assertEquals($ordernumber, $id);
                return $order;
            });
        $order
            ->expects($this->once())
            ->method('createOrderNumber')
            ->willReturn($ordernumber);

        //act
        $reflect = new \ReflectionClass($order);
        $reflect->getConstructor()->invoke($order);

        //assert
    }

    /**
     * 1.先產生一個假的order服務
     * 2.實際去呼叫order服務，並同一時間跑數次來測試
     * 3.最後排除重複的Id
     * 4.並對照是否都不重複
     */
    public function test_createOrderNumber()
    {
        //arrange
        $max = 1000;
        $ordernumbers = array();
        $order = $this->getMockBuilder(Order::class)
            ->disableOriginalConstructor()
            ->setMethods()
            ->getMock();

        //act
        for($i=0; $i<$max; $i++) {
            $ordernumbers[] = $this->callObjectMethod($order, 'createOrderNumber');
        }

        //assert
        $ordernumbers = array_unique($ordernumbers);
        $this->assertEquals($max, count($ordernumbers));
    }

    /**
     * 1.給他一個orderItems的組合大資料
     * 2.把getItem先mock起來，之後把orderItem丟回去
     * 3.直接去callsyncAmout（是public所以可以直接call）
     * 4.最後要對照資料是否正確，不過這邊的amount跟amount_for_pay是protected的所以需要用getObject硬去call出來
     * @dataProvider dataProvider_test_syncAmount
     */
    public function test_syncAmount($dataSet, $amount, $amountForPay)
    {
        //arrange
        $orderItems = array();
        foreach ($dataSet as $data) {
            $orderItem = new OrderItem();
            $orderItem->fromArray($data, \BasePeer::TYPE_PHPNAME);
            $orderItems[] = $orderItem;
        }

        $order = $this->getMockBuilder(Order::class)
            ->setMethods(array('getOrderItems'))
            ->disableOriginalConstructor()
            ->getMock();
        $order
            ->expects($this->once())
            ->method('getOrderItems')
            ->willReturn($orderItems);

        //act
        $order->syncAmount();

        //assert
        $this->assertEquals($amount, $this->getObjectAttribute($order, 'amount'));
        $this->assertEquals($amountForPay, $this->getObjectAttribute($order, 'amount_for_pay'));
    }

    public function test_addProducts()
    {
        // arrange
        $order = $this->getMockBuilder(Order::class)
            ->disableOriginalConstructor()
            ->setMethods(null)
            ->getMock();
        $product = new OrderItem();
        $product->setId(1);
        $product->setProductName('Name');
        $product->setUnitPrice(5566);
        $product->setQuantity(1);

        $product2 = new OrderItem();
        $product2->setId(2);
        $product2->setProductName('Name2');
        $product2->setUnitPrice(11132);
        $product2->setQuantity(1);
        // act
        $order->addProducts($product);
        $order->addProducts($product2);

        $products = $this->getObjectAttribute($order, 'products');
        $this->assertEquals($product, $products[0]);
        $this->assertEquals($product2, $products[1]);
    }

    public function test_getProducts()
    {
        // arrange
        $order = $this->getMockBuilder(Order::class)
            ->disableOriginalConstructor()
            ->setMethods(null)
            ->getMock();

        $product = new OrderItem();
        $product->setId(1);
        $product->setProductName('Name');
        $product->setUnitPrice(5566);
        $product->setQuantity(1);

        $product2 = new OrderItem();
        $product2->setId(2);
        $product2->setProductName('Name2');
        $product2->setUnitPrice(11132);
        $product2->setQuantity(1);
        $products = array(
            $product,
            $product2
        );

        $this->setObjectAttribute($order, 'products', $products);

        // act
        $orderProducts = $order->getProducts();
        // assert
        $this->assertEquals($products, $orderProducts);
    }

    public function test_getShipments()
    {
        // arrange
        $order = $this->getMockBuilder(Order::class)
            ->disableOriginalConstructor()
            ->setMethods(null)
            ->getMock();

        $shipment = new OrderItem();
        $shipment->setId(null);
        $shipment->setProductName('運費');
        $shipment->setUnitPrice(50);
        $shipment->setQuantity(1);

        $shipment2 = new OrderItem();
        $shipment2->setId(null);
        $shipment2->setProductName('運費');
        $shipment2->setUnitPrice(50);
        $shipment2->setQuantity(1);

        // act
        $order->addShipments($shipment);
        $order->addShipments($shipment2);

        $products = $this->getObjectAttribute($order, 'shipments');
        $this->assertEquals($shipment, $products[0]);
        $this->assertEquals($shipment2, $products[1]);
    }

    public function test_setProtectItems()
    {
        $order = new Order();
        $order->setId('5566');

        $product = new OrderItem();
        $product->setId(1);
        $product->setProductName('Name');
        $product->setUnitPrice(5566);
        $product->setQuantity(1);

        $product2 = new OrderItem();
        $product2->setId(2);
        $product2->setProductName('Name2');
        $product2->setUnitPrice(11132);
        $product2->setQuantity(1);

        $shipment = new OrderItem();
        $shipment->setId(null);
        $shipment->setProductName('運費');
        $shipment->setUnitPrice(50);
        $shipment->setQuantity(1);

        $discount = new OrderItem();
        $discount->setId(null);
        $discount->setProductName('折扣1');
        $discount->setUnitPrice(-500);
        $discount->setQuantity(1);

        $discount2 = new OrderItem();
        $discount2->setId(null);
        $discount2->setProductName('折扣2');
        $discount2->setUnitPrice(-500);
        $discount2->setQuantity(1);

        $order->addProducts($product);
        $order->addProducts($product2);
        $order->addShipments($shipment);
        $order->addDiscounts($discount);
        $order->addDiscounts($discount2);

        // act

        $order->setProtectItems();
        $orderItem = $order->getOrderItems();

        // arrange
        $this->assertEquals($product->getProductName(), $orderItem[0]->getProductName());
        $this->assertEquals($product2->getProductName(), $orderItem[1]->getProductName());
        $this->assertEquals($shipment->getProductName(), $orderItem[2]->getProductName());
        $this->assertEquals($discount->getProductName(), $orderItem[3]->getProductName());
        $this->assertEquals($discount2->getProductName(), $orderItem[4]->getProductName());
    }

    public function test_getOrderAmount()
    {
        $order = new Order();
        $this->injectOrderProductShipmentDiscount($order);

        // act
        $orderAmount = $order->getOrderAmount();

        // assert
        $this->assertEquals(16748, $orderAmount);
    }

    public function test_getOrderAmountForPay()
    {
        $order = new Order();
        $this->injectOrderProductShipmentDiscount($order);

        // act
        $orderAmountForPay = $order->getOrderAmountForPay();

        // assert
        $this->assertEquals(15748, $orderAmountForPay);
    }

    public function test_getOrderNoShipmentAmount()
    {
        $order = new Order();
        $this->injectOrderProductShipmentDiscount($order);

        // act
        $orderAmountForPay = $order->getOrderNoShipmentAmount();

        // assert
        $this->assertEquals(15698, $orderAmountForPay);
    }

    /**
     * 1.一大組測試資料的時候可以用dataProvider來組合資料
     */
    public function dataProvider_test_syncAmount()
    {
        return array(
            array(
                'dataSet' => array(
                    array(
                        'UnitPrice' => 100,
                        'Quantity' => 1,
                    ),
                    array(
                        'UnitPrice' => 200,
                        'Quantity' => 2,
                    ),
                    array(
                        'UnitPrice' => 300,
                        'Quantity' => 3,
                    )
                ),
                'amount' => 1400,
                'amountForPay' => 1400,
            ),
            array(
                'dataSet' => array(
                    array(
                        'UnitPrice' => 100,
                        'Quantity' => 1,
                    ),
                    array(
                        'UnitPrice' => 200,
                        'Quantity' => 2,
                    ),
                    array(
                        'UnitPrice' => -20,
                        'Quantity' => 10,
                    ),
                    array(
                        'UnitPrice' => 300,
                        'Quantity' => 3,
                    )
                ),
                'amount' => 1400,
                'amountForPay' => 1200,
            ),
            array(
                'dataSet' => array(
                    array(
                        'UnitPrice' => 100,
                        'Quantity' => 1,
                    ),
                    array(
                        'UnitPrice' => 200,
                        'Quantity' => 2,
                    ),
                    array(
                        'UnitPrice' => -20,
                        'Quantity' => 10,
                    ),
                    array(
                        'UnitPrice' => -30,
                        'Quantity' => 5,
                    ),
                    array(
                        'UnitPrice' => 300,
                        'Quantity' => 3,
                    )
                ),
                'amount' => 1400,
                'amountForPay' => 1050,
            ),
        );
    }

    protected function injectOrderProductShipmentDiscount(Order $order)
    {
        $order->setId('5566');

        $product = new OrderItem();
        $product->setId(1);
        $product->setProductName('產品1號');
        $product->setUnitPrice(5566);
        $product->setQuantity(1);

        $product2 = new OrderItem();
        $product2->setId(2);
        $product2->setProductName('產品2號');
        $product2->setUnitPrice(11132);
        $product2->setQuantity(1);

        $shipment = new OrderItem();
        $shipment->setId(null);
        $shipment->setProductName('運費');
        $shipment->setUnitPrice(50);
        $shipment->setQuantity(1);

        $discount = new OrderItem();
        $discount->setId(null);
        $discount->setProductName('折扣1');
        $discount->setUnitPrice(-500);
        $discount->setQuantity(1);

        $discount2 = new OrderItem();
        $discount2->setId(null);
        $discount2->setProductName('折扣2');
        $discount2->setUnitPrice(-500);
        $discount2->setQuantity(1);

        $order->addProducts($product);
        $order->addProducts($product2);
        $order->addShipments($shipment);
        $order->addDiscounts($discount);
        $order->addDiscounts($discount2);


    }
}