<?php

namespace Widget\OrderBundle\Model;

use Widget\OrderBundle\Model\om\BaseOrder;

class Order extends BaseOrder
{
    protected $orderLogComment;

    protected $products = array();

    protected $shipments = array();

    protected $discounts = array();

    public function __construct()
    {
        $this->setId($this->createOrderNumber());
    }

    protected function createOrderNumber()
    {
        $milisec = floor(microtime(true)*10000) % 10000;
        $sec = ((strftime('%k') * 60) + strftime('%M')) * 60 + strftime('%S');
        $times = sprintf('A%s%05d%04d%03d',
            strftime("%y%j"),
            $sec,
            $milisec,
            rand(0, 999)
        );
        usleep(100);
        return $times;
    }

    public function calculateProductAmount(\PropelPDO $con = null)
    {
        $amount = 0;
        foreach($this->getOrderItems(null, $con) as $orderItem){
            if(!$orderItem->getProductId()){
                continue;
            }
            if($orderItem->getUnitPrice()<=0){
                continue;
            }
            $amount += $orderItem->getUnitPrice() * $orderItem->getQuantity();
        }
        return $amount;
    }

    public function syncAmount(\PropelPDO $con = null)
    {
        $amount = 0;
        $amountDiscount = 0;
        foreach($this->getOrderItems(null, $con) as $orderItem){
            $tmpAmount = $orderItem->getUnitPrice() * $orderItem->getQuantity();
            if($tmpAmount > 0){
                $amount += $tmpAmount;
            }
            else{
                $amountDiscount += $tmpAmount;
            }
        }
        $this->setAmount($amount);
        $this->setAmountForPay($amount + $amountDiscount);
    }

    public function getPayment(\PropelPDO $con = null)
    {
        if(property_exists(parent::class, 'singlePayment')){
            return parent::getPayment($con);
        }
        return null;
    }

    /**
     * @return mixed
     */
    public function getOrderLogComment()
    {
        return $this->orderLogComment;
    }

    /**
     * @param mixed $orderLogComment
     * @return Order
     */
    public function setOrderLogComment($orderLogComment)
    {
        $this->orderLogComment = $orderLogComment;
        return $this;
    }

    public function postSave(\PropelPDO $con = null)
    {
        parent::postSave($con); // TODO: Change the autogenerated stub
        $this->orderLogComment = null;
    }

    /**
     * 取得訂單定價(產品 + 運費)
     * @return int
     */
    public function getOrderAmount()
    {
        $amount = 0;
        /** @var OrderItem $product */
        foreach ($this->getProducts() as $product) {
            $amount += (int)$product->getQuantity() * (int)$product->getUnitPrice();
        }
        /** @var OrderItem $shipment */
        foreach ($this->getShipments() as $shipment) {
            $amount += (int)$shipment->getQuantity() * (int)$shipment->getUnitPrice();
        }
        return $amount;
    }

    /**
     * 取得訂單應付價格 (產品 + 運費 + 折扣)
     * @return int
     */
    public function getOrderAmountForPay()
    {
        $amountForPay = 0;
        /** @var OrderItem $product */
        foreach ($this->getProducts() as $product) {
            $amountForPay += (int)$product->getUnitPrice() * (int)$product->getQuantity();
        }

        /** @var OrderItem $shipment */
        foreach ($this->getShipments() as $shipment){
            $amountForPay += (int)$shipment->getUnitPrice() * (int)$shipment->getQuantity();
        }
        /** @var OrderItem $discount */
        foreach ($this->getDiscounts() as $discount) {
            $amountForPay += (int)$discount->getUnitPrice() * (int)$discount->getQuantity();
        }

        return $amountForPay;
    }

    /**
     * 取得不含運費的訂單總價 (計算免運折扣用)
     * @return int
     */
    public function getOrderNoShipmentAmount()
    {
        $amountForPay = 0;
        /** @var OrderItem $product */
        foreach ($this->getProducts() as $product) {
            $amountForPay += (int)$product->getUnitPrice() * (int)$product->getQuantity();
        }

        /** @var OrderItem $discount */
        foreach ($this->getDiscounts() as $discount) {
            $amountForPay += (int)$discount->getUnitPrice() * (int)$discount->getQuantity();
        }

        return $amountForPay;
    }

    /**
     * @return array
     */
    public function getProducts()
    {
        return $this->products;
    }

    /**
     * @param OrderItem $product
     */
    public function addProducts(OrderItem $product)
    {
        $this->products[] = $product;
    }

    /**
     * @return array
     */
    public function getShipments()
    {
        return $this->shipments;
    }

    /**
     * @param OrderItem $shipment
     */
    public function addShipments(OrderItem $shipment)
    {
        $this->shipments[] = $shipment;
    }

    /**
     * @return array
     */
    public function getDiscounts()
    {
        return $this->discounts;
    }

    /**
     * @param OrderItem $discount
     */
    public function addDiscounts(OrderItem $discount)
    {
        $this->discounts[] = $discount;
    }
    
    /**
     * 清掉 protect 的項目
     */
    public function clearItems()
    {
        $this->products = array();
        $this->shipments = array();
        $this->discounts = array();
    }

    public function clearDiscount()
    {
        $this->discounts = array();
    }

    /**
     * 將產品 運費 折扣 綁回訂單
     */
    public function setProtectItems()
    {
        foreach ($this->getProducts() as $product) {
            $product->setOrder($this);
        }

        foreach ($this->getShipments() as $shipment) {
            $shipment->setOrder($this);
        }

        foreach ($this->getDiscounts() as $discount) {
            $discount->setOrder($this);
        }
    }
}
