<?php
/**
 * Created by PhpStorm.
 * User: bubble
 * Date: 2016/7/5
 * Time: 下午 5:59
 */

namespace Widget\OrderBundle\EventListener\ControllerConfig;

use Backend\BaseBundle\Service\CustomMailerService;
use Backend\BaseBundle\SiteConfig\SiteConfigBuilder;
use JMS\DiExtraBundle\Annotation as DI;
use JMS\DiExtraBundle\Annotation\Inject;
use JMS\DiExtraBundle\Annotation\InjectParams;
use Widget\MemberBundle\Model\MemberQuery;
use Widget\OrderBundle\Event\ReceiveCreateOrderEvent;
use Widget\OrderBundle\Model\OrderQuery;
use Widget\OrderBundle\Model\OrderStatusQuery;
use Widget\PaymentBundle\Event\ReceiveOrderEvent;
use Widget\PaymentBundle\Event\ReceivePaymentnumberEvent;

/**
 * @DI\Service()
 */
class OrderSendMailListener
{
    /** @var  CustomMailerService */
    protected $customMailerService;

    /** @var  SiteConfigBuilder */
    protected $siteConfigBuilder;

    /**
     * @InjectParams({
     *    "customMailerService" = @Inject("site_custom_mailer"),
     * })
     */
    public function injectCustomMailerService(CustomMailerService $customMailerService)
    {
        $this->customMailerService = $customMailerService;
    }

    /**
     * @InjectParams({
     *    "siteConfigBuilder" = @Inject("backend_base.site_config_builder"),
     * })
     */
    public function injectSiteConfigBuilder(SiteConfigBuilder $siteConfigBuilder)
    {
        $this->siteConfigBuilder = $siteConfigBuilder;
    }

    protected function twigBuilder(array $templates)
    {
        return new \Twig_Environment(new \Twig_Loader_Array($templates));
    }

    /**
     * 付款成功寄信通知
     * @DI\Observe(ReceiveOrderEvent::EVENT_NAME)
     */
    public function onReceiveOrderStatusEvent(ReceiveOrderEvent $event)
    {
        if(!$event->getStatus()){
            $event->stopPropagation();
            return;
        }

        $order = OrderQuery::create()->findPk($event->getOrdernumber());
        $orderStatus = $order->getOrderStatus();
        $buyer = $order->getOrderInfo();
        $member = MemberQuery::create()
            ->findOneById($order->getMemberId());

        $mailerConfig = $this->siteConfigBuilder->build($order->getSiteId())->get('widget_order.mail')['PaymentSuccess'];
        $mailerTwig = $this->twigBuilder(array(
            'payment_success_mail_subject' => $mailerConfig['payment_success_mail_subject'],
            'payment_success_mail' => $mailerConfig['payment_success_mail'],
        ));
        $message = $this->customMailerService->newMessage($order->getSiteId())
            ->setSubject($mailerTwig->render('payment_success_mail_subject', array(
                'ordernumber' => $order->getId(),
                'orderStatus' => $orderStatus->getName(),
            )))
            ->setTo(
                array(
                    $member->getEmail(),
                    $buyer->getBuyerEmail()
                ),
                array(
                    $member->getName(),
                    $buyer->getBuyerName()
                )
            )
            ->setBody($mailerTwig->render('payment_success_mail', array(
                'ordernumber' => $order->getId(),
                'amountForPay' => $order->getAmountForPay(),
                'amount' => $order->getAmount(),
                'paidAt' => $order->getPayment()->getPaymentPaidAt('%Y-%m-%d %T'),
                'paymentType' => $order->getPayment()->getPaymentType(),
            )))
            ->setCharset('utf-8')
            ->setContentType('text/html')
        ;
        $this->sendMessage($order, $message);
    }

    /**
     * 繳款通知寄信
     * @DI\Observe(ReceivePaymentnumberEvent::EVENT_NAME)
     */
    public function onReceivePaymentnumberEvent(ReceivePaymentnumberEvent $event)
    {
        $order = OrderQuery::create()->findPk($event->getOrdernumber());
        $orderStatus = OrderStatusQuery::create()
            ->findOneById($order->getOrderStatusId());
        if(!$order->getOrderStatus() || !$order->getOrderStatus()->getStatus()){
            $event->stopPropagation();
            return;
        }
        $buyer = $order->getOrderInfo();
        $member = MemberQuery::create()
            ->findOneById($order->getMemberId());

        $mailerConfig = $this->siteConfigBuilder->build($order->getSiteId())->get('widget_order.mail')['Payment'];
        $mailerTwig = $this->twigBuilder(array(
            'payment_mail_subject' => $mailerConfig['payment_mail_subject'],
            'payment_mail' => $mailerConfig['payment_mail'],
        ));
        $message = $this->customMailerService->newMessage($order->getSiteId())
            ->setSubject($mailerTwig->render('payment_mail_subject', array(
                'ordernumber' => $order->getId(),
                'orderStatus' => $orderStatus->getName(),
            )))
            ->setTo(
                array(
                    $member->getEmail(),
                    $buyer->getBuyerEmail()
                ),
                array(
                    $member->getName(),
                    $buyer->getBuyerName()
                )
            )
            ->setBody($mailerTwig->render('payment_mail', array(
                'ordernumber' => $order->getId(),
                'paymentnumber' =>$event->getPaymentnumber(),
                'amountForPay' => $order->getAmountForPay(),
                'amount' => $order->getAmount(),
                'createdAt' => $order->getCreatedAt(),
                'expiredAt' => $event->getExpiredAt(),
                'paymentType' => $order->getPayment()->getPaymentType(),
            )))
            ->setCharset('utf-8')
            ->setContentType('text/html')
        ;
        $this->sendMessage($order, $message);
    }

    /**
     * 建立訂單寄信通知
     * @DI\Observe(ReceiveCreateOrderEvent::EVENT_NAME)
     */
    public function onReceiveCreateOrderEvent(ReceiveCreateOrderEvent $event)
    {
        $order = OrderQuery::create()->findPk($event->getOrder()->getId());
        $orderStatus = OrderStatusQuery::create()
            ->findOneById($order->getOrderStatusId());
        if(!$order->getOrderStatus() || !$order->getOrderStatus()->getStatus()){
            $event->stopPropagation();
            return;
        }
        $member = MemberQuery::create()
            ->findOneById($order->getMemberId());
        $buyer = $order->getOrderInfo();

        $mailerConfig = $this->siteConfigBuilder->build($order->getSiteId())->get('widget_order.mail')['statusChange'];
        $mailerTwig = $this->twigBuilder(array(
            'status_change_mail_subject' => $mailerConfig['status_change_mail_subject'],
            'status_change_mail' => $mailerConfig['status_change_mail'],
        ));
        $message = $this->customMailerService->newMessage($order->getSiteId())
            ->setSubject($mailerTwig->render('status_change_mail_subject', array(
                'ordernumber' => $order->getId(),
                'orderStatus' => $orderStatus->getName(),
            )))
            ->setTo(
                array(
                    $member->getEmail(),
                    $buyer->getBuyerEmail()
                ),
                array(
                    $member->getName(),
                    $buyer->getBuyerName()
                ))
            ->setContentType('text/html')
            ->setBody($mailerTwig->render('status_change_mail', array(
                'ordernumber' => $order->getId(),
                'orderStatus' => $orderStatus->getName(),
                'amountForPay' => $order->getAmountForPay(),
                'amount' => $order->getAmount(),
                'createdAt' => $order->getCreatedAt(),
            )))
        ;
        $message->getBody();
        $this->sendMessage($order, $message);
    }

    /**
     * @param $order
     * @param $message
     */
    protected function sendMessage($order, $message)
    {
        try {
            return $this->customMailerService->get($order->getSiteId())->send($message);
        } catch (\Exception $e) {
            return false;
        }
    }
}