<?php
namespace Widget\OrderBundle\EventListener\ControllerConfig;

use Backend\BaseBundle\Event\Controller\ConfigEvent;
use Backend\BaseBundle\Event\Controller\MenuEvent;
use Backend\BaseBundle\Event\Controller\TypeRolesEvent;
use Backend\BaseBundle\Form\Type\BaseFormType;
use Backend\BaseBundle\Form\Type\DisplayType;
use Backend\BaseBundle\Form\Type\RelationType;
use Backend\BaseBundle\Form\Type\SubmitBackType;
use JMS\DiExtraBundle\Annotation as DI;
use JMS\DiExtraBundle\Annotation\InjectParams;
use Symfony\Component\EventDispatcher\EventDispatcherInterface;
use Symfony\Component\Form\FormBuilderInterface;
use Symfony\Component\Routing\Router;
use Symfony\Component\Form\Extension\Core\Type as BaseType;
use Widget\OrderBundle\Event\RequestPaymentConfigEvent;
use Widget\OrderBundle\Model\OrderInfo;
use Widget\OrderBundle\Model\OrderStatus;
use Widget\PaymentBundle\Model\Payment;


/**
 * @DI\Service()
 */
class OrderListener
{
    const CONTROLLER_NAME = 'WidgetOrderBundle:Backend/Order';

    /** @var  Router */
    protected $router;

    /** @var  EventDispatcherInterface */
    protected  $eventDispatcher;


    /**
     * 注入router服務
     * @InjectParams()
     */
    public function injectRouter(Router $router)
    {
        $this->router = $router;
    }

    /**
     * @InjectParams()
     */
    public function injectEventDispatcher(EventDispatcherInterface $eventDispatcher)
    {
        $this->eventDispatcher = $eventDispatcher;
    }

    /**
     * 注入選單服務
     * @DI\Observe(MenuEvent::EVENT_MENU)
     */
    public function onMenu(MenuEvent $event)
    {
        $user = $event->getUser();
        // 驗證是否有權限
        if(!$user->hasRoleOrSuperAdmin('ROLE_ORDER')) {
            return;
        }

        $isOrderActive = preg_match('/^widget_order_backend_order_/i', $event->getRequestRouteName());
        $slug = $user->getSiteSlug();
        $event->addMenu(self::CONTROLLER_NAME, array(
            'menu.order' => array(
                'class' => 'fa fa-shopping-cart',
                'active' => $isOrderActive,
                'target' => array(
                    'menu.order_list' => array(
                        'class' => 'fa fa-shopping-cart',
                        'active' => $isOrderActive,
                        'target' => $this->router->generate('widget_order_backend_order_index', array('slug' => $slug)),
                    ),
                ),
            ),
        ));
    }

    /**
     * 讓帳號管理介面加入權限設定
     * @DI\Observe(TypeRolesEvent::EVENT_TYPE_ROLES)
     */
    public function onTypeRoles(TypeRolesEvent $event)
    {
        $event->addTypeRoles('ROLE_ORDER', array(
            'ROLE_ORDER_READ',
            'ROLE_ORDER_WRITE',
        ));
    }

    /**
     * @DI\Observe(ConfigEvent::EVENT_CONFIG)
     * 將訂單資訊與付款資訊分開
     * 有裝付款模組的話才會有付款資訊欄位
     */
    public function onConfig(ConfigEvent $event)
    {
        if($event->getControllerName() != self::CONTROLLER_NAME){
            return;
        }

        $requestPaymentConfigEvent = new RequestPaymentConfigEvent();

        $this->eventDispatcher->dispatch(RequestPaymentConfigEvent::EVENT_NAME, $requestPaymentConfigEvent);

        $paymentConfig = array();

        if($requestPaymentConfigEvent->getFormConfig()){
            $paymentConfig = array(
                array(
                    'Payment',  //column
                    new BaseFormType(function(FormBuilderInterface $builder) use($requestPaymentConfigEvent){
                        foreach($requestPaymentConfigEvent->getFormConfig() as $config){
                            $builder->add($config[0], $config[1], $config[2]);
                        }
                    }, 'Payment'),
                    array(
                        'data_class' => Payment::class,
                        'attr' => array(
                            'data-subform' => 'embaded',
                        ),
                    ),
                ),
            );
        }
        $event->setConfig(array(
            'route' => array(
                'index' => true,
                'sort' => true,
                'view' => true,
                'edit' => true,
                'search' => true,
            ),
            'index' => array_merge(
                array(
                    'index.order.id' => 'Id',
                    'index.order.info.buyer_name' => 'OrderInfo:BuyerName',
                    'index.order.amount_for_pay'=>'AmountForPay',
                ),
                $requestPaymentConfigEvent->getIndexConfig(),
                array(
                    'index.order.groups.status'=>'OrderStatus:Name',
                    'index.order.created_at'=>'CreatedAt',
                )
            ),
            'security' => array(
                'new' => 'ROLE_ORDER',
                'view' => 'ROLE_ORDER',
                'edit' => 'ROLE_ORDER_WRITE',
            ),
            'search' => array(
                array(
                    'Id',
                    BaseType\TextType::class,
                    array(
                        'label' => 'index.order.id',
                        'required' => false,
                        'disabled' => true,
                    ),
                ),
            ),
            'action' => array(
                'action.view' => array(
                    'class' => 'fa fa-pencil-square-o',
                    'action' => 'view',
                    'type' => 'ajax',
                ),
                'action.edit' => array(
                    'class' => 'fa fa-pencil-square-o',
                    'action' => 'edit'
                ),
            ),
            'view' => array_merge(
                array(
                    'Id' => array(
                        'label' => 'index.order.id',
                    ),
                    'Amount' => array(
                        'label' => 'index.order.amount',
                    ),
                    'AmountForPay' => array(
                        'label' => 'index.order.amount_for_pay',
                    ),
                    'OrderNote' => array(
                        'label' => 'index.order.order_note',
                    ),
                    'OrderItems' => array(
                        'label' => 'index.order.order_items',
                        'template' => 'WidgetOrderBundle:Backend:_orderitems.html.twig',
                    ),
                    'CreatedAt' => array(
                        'label' => 'index.order.created_at',
                    ),
                    'UpdatedAt' => array(
                        'label' => 'index.order.updated_at',
                    ),
                ),
                $requestPaymentConfigEvent->getViewConfig(),
                array(
                    'OrderInfo' => array(
                        'label' => 'index.order.info.order_info',
                        'template' => 'WidgetOrderBundle:Backend:_orderinfo.html.twig',
                    ),
                    'OrderLogs' => array(
                        'label' => 'index.order.order_logs.views',
                        'template' => 'WidgetOrderBundle:Backend:_orderlogs.html.twig',
                    ),
                )
            ),
            'form' => array_merge(
                array(
                    array(
                        'Id',
                        BaseType\TextType::class,
                        array(
                            'label' => 'index.order.id',
                            'required' => false,
                            'disabled' => true,
                        ),
                    ),
                    array(
                        'OrderLogs',
                        DisplayType::class,
                        array(
                            'label' => 'index.order.order_logs.views',
                            'template' => 'WidgetOrderBundle:Backend:_orderlogs.html.twig',
                        ),
                    ),
                    //關聯到order_status的資料表
                    array(
                        'OrderStatus',
                        RelationType::class,  //關聯欄位
                        array(
                            'label' => 'index.order.groups.status',
                            'class' => OrderStatus::class,
                            'multiple' => false,
                            'required' => false,
                        ),
                    ),
                    array(
                        'OrderLogComment',
                        BaseType\TextareaType::class,
                        array(
                            'label' => 'index.order.order_logs.write',
                            'required' => false,
                        )
                    ),
                    array(
                        'OrderSendMail',
                        DisplayType::class,
                        array(
                            'label' => 'index.order.send_mail',
                            'template' => 'WidgetOrderBundle:Backend:_sendmail.html.twig',
                            'mapped' => false,
                            'required' => false,
                        ),
                    ),
                ),
                $paymentConfig,
                array(
                    array(
                        'Amount',
                        BaseType\NumberType::class,
                        array(
                            'label' => 'index.order.amount',
                            'required' => false,
                            'disabled' => true,
                        ),
                    ),
                    array(
                        'AmountForPay',
                        BaseType\NumberType::class,
                        array(
                            'label' => 'index.order.amount_for_pay',
                            'required' => false,
                            'disabled' => true,
                        ),
                    ),
                    array(
                        'OrderNote',
                        DisplayType::class,
                        array(
                            'label' => 'index.order.order_note',
                            'template' => 'WidgetOrderBundle:Backend:_ordernote.html.twig',
                        ),
                    ),
                    array(
                        'OrderItems',
                        DisplayType::class,
                        array(
                            'label' => 'index.order.order_items',
                            'template' => 'WidgetOrderBundle:Backend:_orderitems.html.twig',
                        ),
                    ),
                    array(
                        'OrderInfo',  //column
                        new BaseFormType(function(FormBuilderInterface $builder){
                            $builder
                                ->add('BuyerName', BaseType\TextType::class, array( 'label' => 'index.order.info.buyer_name' ))
                                ->add('BuyerPhone', BaseType\TextType::class, array( 'label' => 'index.order.info.buyer_phone' ))
                                ->add('BuyerEmail', BaseType\EmailType::class, array( 'label' => 'index.order.info.buyer_email' ))
                                ->add('BuyerZip', BaseType\TextType::class, array( 'label' => 'index.order.info.buyer_zip' ))
                                ->add('BuyerAddress', BaseType\TextType::class, array( 'label' => 'index.order.info.buyer_address' ))
                                ->add('ReceiverName', BaseType\TextType::class, array( 'label' => 'index.order.info.receiver_name' ))
                                ->add('ReceiverPhone', BaseType\TextType::class, array( 'label' => 'index.order.info.receiver_phone' ))
                                ->add('ReceiverEmail', BaseType\EmailType::class, array( 'label' => 'index.order.info.receiver_email' ))
                                ->add('ReceiverZip', BaseType\TextType::class, array( 'label' => 'index.order.info.receiver_zip' ))
                                ->add('ReceiverAddress', BaseType\TextType::class, array( 'label' => 'index.order.info.receiver_address' ));
                        }, 'OrderInfo'),
                        array(
                            'label' => 'index.order.info.order_info',
                            'data_class' => OrderInfo::class,
                            'attr' => array(
                                'data-subform' => 'panel',
                            ),
                        ),
                    ),
                    array(
                        'submit',
                        SubmitBackType::class,
                        array(
                            'with_back' => true,
                            'with_reset' => true,
                            'back_route' => 'widget_order_backend_order_index',
                        ),
                    )
                )
            ),
        ));
    }
}