<?php
namespace Widget\OrderBundle\Tests\Service;

use Backend\BaseBundle\Tests\Fixture\BaseTestCase;
use Symfony\Component\EventDispatcher\EventDispatcher;
use Widget\OrderBundle\Event\ProductAddEvent;
use Widget\OrderBundle\Model\Order;
use Widget\OrderBundle\Model\OrderItem;
use Widget\OrderBundle\Service\OrderBuilder;

class OrderBuilderTest extends BaseTestCase
{
    /**
     * 1.先產生一個假的資料
     * 2.實際去呼叫create這個function
     * 3.爾後，驗證是否有去呼叫order、site_id_member_id這幾個function
     * 4.並比對實際呼叫出來是否與預設資料相符
     */
    public function test_create()
    {
        //arrange
        $siteId = '123456';
        $memberId = '456789';
        $orderbudilder = new OrderBuilder();

        //act
        $orderbudilder->create($siteId, $memberId);

        //assert
        $order = $this->getObjectAttribute($orderbudilder, 'order');
        $this->assertEquals($siteId, $this->getObjectAttribute($order, 'site_id'));
        $this->assertEquals($memberId, $this->getObjectAttribute($order, 'member_id'));
    }

    /**
     * 1.dispatchEvent取得產品名稱、價格(mock)
     * 2.把取得產品名稱、價格，寫回訂單item裡面(mock)
     */
    public function test_add()
    {
        //arrange
        $siteId = '12345';
        $id = '12345';
        $count = 10;
        $order = new Order();
        $order->setSiteId($siteId);

        $productAddEvent = new ProductAddEvent($id, $count, $siteId);
        $orderBuilder = $this->getMockBuilder(OrderBuilder::class)
            ->setMethods(array('queryProduct', 'addOrderItem'))
            ->disableOriginalConstructor()
            ->getMock();
        $orderBuilder
            ->expects($this->once())
            ->method('queryProduct')
            ->willReturnCallback(function($idForTest, $countForTest) use($id, $count, $productAddEvent){
                $this->assertEquals($id, $idForTest);
                $this->assertEquals($count, $countForTest);
                return $productAddEvent;
            });
        $orderBuilder
            ->expects($this->once())
            ->method('addOrderItem')
            ->willReturnCallback(function(Order $orderForTest, ProductAddEvent $event) use($order, $productAddEvent){
                $this->assertEquals($productAddEvent, $event);
                $this->assertEquals($order, $orderForTest);
            });
        $this->setObjectAttribute($orderBuilder, 'order', $order);

        //act
        $this->callObjectMethod($orderBuilder, 'add', $id, $count);

        //assert
    }

    public function test_add_has_error()
    {
        //arrange
        $siteId = '12345';
        $id = '12345';
        $count = 10;
        $order = new Order();
        $order->setSiteId($siteId);

        $productAddEvent = new ProductAddEvent($id, $count, $siteId);
        $productAddEvent->setError('error');

        $orderBuilder = $this->getMockBuilder(OrderBuilder::class)
            ->disableOriginalConstructor()
            ->setMethods(array('queryProduct', 'addOrderItem', 'addError'))
            ->getMock();

        $orderBuilder
            ->expects($this->once())
            ->method('queryProduct')
            ->willReturnCallback(function($idForTest, $countForTest) use($id, $count, $productAddEvent){
                $this->assertEquals($id, $idForTest);
                $this->assertEquals($count, $countForTest);
                return $productAddEvent;
            });

        $orderBuilder
            ->expects($this->once())
            ->method('addError')
            ->willReturnCallback(function($idforTest, $error) use($id){
                $this->assertEquals($id, $idforTest);
                $this->assertEquals('error', $error);
            });

        $orderBuilder
            ->expects($this->never())
            ->method('addOrderItem')
            ->willReturn(null);
        $this->setObjectAttribute($orderBuilder, 'order', $order);

        //act
        $orderBuilder->add($id, $count);

        //assert
    }

    public function test_addError()
    {
        //arrange
        $errors = array(
            5 => 'error1',
            1 => 'error2',
            2 => 'error3',
        );

        $orderBuilder = $this->getMockBuilder(OrderBuilder::class)
            ->disableOriginalConstructor()
            ->getMock();

        //act
        foreach ($errors as $id => $error) {
            $this->callObjectMethod($orderBuilder, 'addError', $id, $error);
        }

        //assert
        $this->assertEquals($errors, $this->getObjectAttribute($orderBuilder, 'errors'));
    }

    public function test_getErrors()
    {
        //arrange
        $errors = array(
            5 => 'error1',
            1 => 'error2',
            2 => 'error3',
        );

        $orderBuilder = $this->getMockBuilder(OrderBuilder::class)
            ->setMethods(null)
            ->disableOriginalConstructor()
            ->getMock();

        //act
        $this->setObjectAttribute($orderBuilder, 'errors', $errors);

        //assert
        $this->assertEquals($errors, $orderBuilder->getErrors());
    }

    public function test_queryProduct()
    {
        //arrange
        $siteId = '12345';
        $order = new Order();
        $order->setSiteId($siteId);
        $id = '123456';
        $count = 10;
        $productAddEvent = null;
        $dispatcher = $this->getMockBuilder(EventDispatcher::class)
            ->setMethods(array('dispatch'))
            ->disableOriginalConstructor()
            ->getMock();
        $orderBuilder = new OrderBuilder();
        $this->setObjectAttribute($orderBuilder, 'eventDispatcher', $dispatcher);
        $this->setObjectAttribute($orderBuilder, 'order', $order);
        $dispatcher
            ->expects($this->once())
            ->method('dispatch')
            ->willReturnCallback(function($eventName, ProductAddEvent $event) use(&$productAddEvent, $id, $count){
                $this->assertEquals($id, $event->getId());
                $this->assertEquals($count, $event->getCount());
                $this->assertEquals(ProductAddEvent::EVENT_NAME, $eventName);
                $productAddEvent = $event;
            });
        //act
        $result = $this->callObjectMethod($orderBuilder, 'queryProduct', $id, $count);

        //assert
        $this->assertEquals($productAddEvent, $result);
    }

    public function test_addOrderItem()
    {
        //arrange
        $siteId = '12345';
        $id = '123456';
        $count = 10;
        $productName = 'abcd';
        $productPrice = 12345;

        $order = new Order();
        $order->setSiteId($siteId);

        $productAddEvent = new ProductAddEvent($id, $count, $siteId);
        $productAddEvent
            ->setProductName($productName)
            ->setProductPrice($productPrice);
        $orderBudiler = new OrderBuilder();

        //act
        $this->callObjectMethod($orderBudiler, 'addOrderItem', $order, $productAddEvent);
        
        //assert
        $this->assertCount(1, $order->getProducts());
        /** @var OrderItem $orderItem */
        $orderItem = $order->getProducts()[0];
        $this->assertEquals($id, $orderItem->getProductId());
        $this->assertEquals($count, $orderItem->getQuantity());
        $this->assertEquals($productName, $orderItem->getProductName());
        $this->assertEquals($productPrice, $orderItem->getUnitPrice());
    }
}