<?php
namespace Widget\OrderBundle\Controller\API;

use Backend\BaseBundle\Controller\API\BaseController;
use Backend\BaseBundle\Exception\ErrorResponseException;
use Backend\BaseBundle\Model\Site;
use Backend\BaseBundle\Service\ConstraintViolationListConvertor;
use JMS\DiExtraBundle\Annotation as DI;
use Sensio\Bundle\FrameworkExtraBundle\Configuration\ParamConverter;
use Sensio\Bundle\FrameworkExtraBundle\Configuration\Route;
use Sensio\Bundle\FrameworkExtraBundle\Configuration\Method;
use Symfony\Component\EventDispatcher\EventDispatcherInterface;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\Validator\Constraints as Assert;
use Symfony\Component\Validator\Constraints\Callback;
use Symfony\Component\Validator\Context\ExecutionContextInterface;
use Symfony\Component\Validator\Validator\ValidatorInterface;
use Widget\MemberBundle\Token\MemberAuthToken;
use Widget\OrderBundle\Event\DiscountChainEvent;
use Widget\OrderBundle\Event\DiscountModulesEvent;
use Widget\OrderBundle\Event\ReceiveCreateOrderEvent;
use Widget\OrderBundle\Model\Order;
use Widget\OrderBundle\Model\OrderInfo;
use Widget\OrderBundle\Model\OrderQuery;
use Widget\OrderBundle\Model\OrderStatusQuery;
use Widget\OrderBundle\Service\DiscountBuilder;
use Widget\OrderBundle\Service\OrderBuilder;


/**
 * @Route("/order")
 */
class OrderController extends BaseController
{

    /**
     * @DI\Inject("validator")
     * @var ValidatorInterface
     */
    protected $validator;

    /**
     * @DI\Inject("api.validator.error.convertor")
     * @var ConstraintViolationListConvertor
     */
    protected $errorConvertor;

    /**
     * @DI\Inject("widget.order_bundle.discount_builder")
     * @var DiscountBuilder
     */
    protected $discountBuilder;

    /**
     * @DI\Inject("event_dispatcher")
     * @var EventDispatcherInterface
     */
    protected $eventDispatcher;

    /**
     * @DI\Inject("widget.order_bundle.order_builder")
     * @var OrderBuilder
     */
    protected $orderBuilder;
    
    /**
     * 試算訂單金額
     * @Route("/query")
     * @Method({"POST"})
     */
    public function queryAction(Request $request, Site $site)
    {
        $author = $request->attributes->get('_authorizedToken');
        if(!($author instanceof MemberAuthToken) || $author->getAuthData() == null){
            return new Response('', Response::HTTP_FORBIDDEN);
        }
        $parameters = (array) json_decode($request->getContent(), true);
        $result = $this->valid($parameters['order']);
        if($result){
            return $result;
        }

        // 準備 discount 會用到的參數
        $discountParameters = array();
        if (isset($parameters['discount'])) {
            $discountParameters = $parameters['discount'];
        }

        $con = \Propel::getConnection();
        $con->beginTransaction();
        try {
            // 建立訂單
            $order = $this->createOrder($site->getId(), $author->getAuthData()['uid'], $parameters['order'], $con);
            // shipment
            $this->runDiscount($order, $con);

            // 跑折扣
            $event = new DiscountModulesEvent($order, $con, $discountParameters);
            $this->eventDispatcher->dispatch(DiscountModulesEvent::EVENT_NAME, $event);
            
            $order->setProtectItems();
            // 算金額
            $order->syncAmount($con);
            $order->clearItems();
        } catch (ErrorResponseException $e) {
            return $e->makeJsonResponse(Response::HTTP_BAD_REQUEST);
        } finally {
            $con->rollBack();
        }

        return $this->createJsonSerializeResponse($order, array('query'));
    }

    /**
     * 查詢某筆訂單
     * @Route("/query/{id}")
     * @ParamConverter("site", options={"exclude": {"id"}})
     * @Method({"GET"})
     */
    public function queryOrderAction(Request $request, Order $order, Site $site)
    {
        $author = $request->attributes->get('_authorizedToken');

        if(!($author instanceof MemberAuthToken) || ($authData = $author->getAuthData()) == null){
            return new Response('', Response::HTTP_FORBIDDEN);
        }

        if($authData['uid'] != $order->getMemberId() || $authData['sid'] != $site->getId()){
            return new Response('', Response::HTTP_FORBIDDEN);
        }
        if($order->getSiteId() != $site->getId()){
            return new Response('', Response::HTTP_FORBIDDEN);
        }

        return $this->createJsonSerializeResponse($order, array('detail'));
    }

    /**
     * 查詢所有訂單
     * @Route("/query")
     * @Method({"GET"})
     */
    public function queryAllOrderAction(Request $request, Site $site)
    {
        $author = $request->attributes->get('_authorizedToken');

        if(!($author instanceof MemberAuthToken) || ($authData = $author->getAuthData()) == null){
            return new Response('', Response::HTTP_FORBIDDEN);
        }
        $page = $request->query->getInt('page', 1);

        /** @var \PropelModelPager $pager */
        $pager = OrderQuery::create()
            ->orderById(\Criteria::DESC)
            ->filterByMemberId($authData['uid'])
            ->paginate($page, 10);
        return $this->createJsonSerializeResponse($pager, array('list'));
    }

    /**
     * 建立訂單
     * @Route("/create")
     * @Method({"POST"})
     */
    public function createAction(Request $request, Site $site)
    {
        $author = $request->attributes->get('_authorizedToken');
        if(!($author instanceof MemberAuthToken) || $author->getAuthData() == null){
            return new Response('', Response::HTTP_FORBIDDEN);
        }
        $parameters = (array) json_decode($request->getContent(), true);

        $result = $this->valid($parameters['order']);
        if($result){
           return $result;
        }

        // 準備 discount 會用到的參數
        $discountParameters = array();
        if (isset($parameters['discount'])) {
            $discountParameters = $parameters['discount'];
        }

        $con = \Propel::getConnection();
        $con->query("SET TRANSACTION ISOLATION LEVEL SERIALIZABLE;");
        for($i = 0; $i<3; $i++) {
            try {
                $con->beginTransaction();
                $order = $this->createOrder($site->getId(), $author->getAuthData()['uid'], $parameters['order'], $con);
                // shipment
                $this->runDiscount($order, $con);

                // 跑折扣
                $event = new DiscountModulesEvent($order, $con, $discountParameters);
                $this->eventDispatcher->dispatch(DiscountModulesEvent::EVENT_NAME, $event);

                $order->setProtectItems();
                $order->syncAmount($con);
                $order->clearItems();

                if($parameters['order_note']){
                    $order->setOrderNote($parameters['order_note']);
                }
                if($order->getAmount() == 0){
                    $con->rollBack();
                    return new Response('', Response::HTTP_BAD_REQUEST);
                }


                if($orderStatus = OrderStatusQuery::create()->findOneByDefaultStatus(true)){
                    $order->setOrderStatus($orderStatus);
                }

                $order->save($con);
                $con->commit();
                $this->createOrderInfo($site->getId(), $order->getId(), $parameters['info']);
                break;
            } catch (\PropelException $e) {
                $con->rollBack();
                if($i==2){
                    return new Response('', Response::HTTP_FORBIDDEN);
                }
            }
            catch (ErrorResponseException $e){
                $con->rollBack();
                return $e->makeJsonResponse(Response::HTTP_BAD_REQUEST);
            }
        }
        $time = new \DateTime();
        $time->setTimestamp(time() + 300);
        $token = (string)$this->get('widget_order.payment.token')->createToken(
            $order->getId(),
            $order->getAmountForPay(),
            'payment',
            $time
        );

        return $this->createJsonSerializeResponse(array(
            'order' => $order,
            'token' => $token,
        ), array('list'));
    }

    /**
     * 事後付款 API
     * @Route("/payafter")
     * @Method("POST")
     */
    public function payAfterAction(Request $request)
    {
        $requestData = json_decode($request->getContent(), true);
        $paymentToken = $this->get('widget_order.payment.token');
        $payload = $paymentToken->verifyToken($requestData['token']);

        if (!$payload) {
            throw $this->createNotFoundException();
        }

        if(!($order = OrderQuery::create()->findPk($payload->ordernumber))){
            throw $this->createNotFoundException();
        }

        $receiveCreateOrder = new ReceiveCreateOrderEvent($order);
        $this->eventDispatcher->dispatch(ReceiveCreateOrderEvent::EVENT_NAME, $receiveCreateOrder);

        return $this->createJsonResponse(array('status' => 'OK'));
    }

    /**
     * 抽離出來做驗證格式部份
     * @return null|Response 如果有錯誤則回傳response
     */
    protected function valid($parameters)
    {
        $validator = $this->get('validator'); //驗證格式
        $constraints = array(
            new Assert\Count(array(
                'min' => 1,
                'max' => 10,
            )),
            new Callback(function($rows, ExecutionContextInterface $context){
                $tmpArray = array();
                foreach ($rows as $row){
                    $tmpArray[] = $row['id'];
                }
                if(count($rows) != count(array_unique($tmpArray))){
                    $context->addViolation('validator.order.type.duplicate');
                }
            }),
            new Assert\All(array(
                new Assert\NotBlank(),
                new Assert\Collection(array(
                    'fields' => array(
                        'id' => array(
                            new Assert\NotBlank(),
                            new Assert\Regex(array(
                                'pattern' => '/^\\d+$/',
                            )),
                        ),
                        'count' => array(
                            new Assert\NotBlank(),
                            new Assert\Range(array(
                                'min'=>1,
                            )),
                        ),
                    ),
                )),
            ))
        );
        $errors = $validator->validate($parameters, $constraints); //把資料驗證(資料, 驗證格式)

        //判斷回傳是否有錯誤 等於0沒錯
        if(count($errors) > 0){
            return $this->createJsonResponse($this->get('api.validator.error.convertor')->toArray($errors), Response::HTTP_BAD_REQUEST);
        }
    }

    /**
     * @param Site $site
     * @param $author
     * @param $parameters
     * @param $con
     * @return \Widget\OrderBundle\Model\Order
     */
    protected function createOrder($siteId, $memberId, $parameters, \PropelPDO $con = null)
    {
        $orderBuilder = $this->orderBuilder
            ->create($siteId, $memberId, $con);

        foreach ($parameters as $parameter) {
            $orderBuilder->add($parameter['id'], $parameter['count']);
        }

        if (count($orderBuilder->getErrors()) >= 1) {
            throw new ErrorResponseException($orderBuilder->getErrors());
        }

        $order = $orderBuilder->getOrder();

        $order->syncAmount($con);
        return $order;
    }
    
    protected function runDiscount(Order $order, \PropelPDO $con = null)
    {
        $event = new DiscountChainEvent($this->discountBuilder, $order->getSiteId());
        $this->eventDispatcher->dispatch(DiscountChainEvent::EVENT_NAME, $event);
        $this->discountBuilder->run($order, $con);
    }

    protected function createOrderInfo($siteId, $orderId, $parameters)
    {
        $buyer = $parameters['buyer'];
        $receiver = $parameters['receiver'];
        $payAfter = isset($parameters['pay_after']) && $parameters['pay_after'] != '' ? $parameters['pay_after'] : null;

        $result = $this->validOrderInfo($buyer);
        if($result){
            return $result;
        }

        $result = $this->validOrderInfo($receiver);
        if($result){
            return $result;
        }

        $orderInfo = new OrderInfo();
        $orderInfo->setId($orderId);
        $orderInfo->setSiteId($siteId);
        $orderInfo->setBuyerName($buyer['name']);
        $orderInfo->setBuyerEmail($buyer['email']);
        $orderInfo->setBuyerPhone($buyer['phone']);
        $orderInfo->setBuyerZip($buyer['zip']);
        $orderInfo->setBuyerAddress($buyer['address']);
        $orderInfo->setReceiverName($receiver['name']);
        $orderInfo->setReceiverEmail($receiver['email']);
        $orderInfo->setReceiverPhone($receiver['phone']);
        $orderInfo->setReceiverZip($receiver['zip']);
        $orderInfo->setReceiverAddress($receiver['address']);
        $orderInfo->setPayAfter($payAfter);
        $orderInfo->save();
    }

    protected function validOrderInfo($info)
    {
        $constraint = new Assert\Collection(array(
            'missingFieldsMessage' => 'error.missing_field',
            'fields' => array(
                'name' => new Assert\NotBlank(),
                'email' => array(
                    new Assert\NotBlank(),
                    new Assert\Email(),
                ),
                'phone' => new Assert\NotBlank(),
                'zip' => new Assert\NotBlank(),
                'address' => new Assert\NotBlank(),
            ),
        ));

        $errors = $this->validator->validate($info, $constraint);

        if(count($errors) > 0){
            return $this->createJsonResponse($this->errorConvertor->toArray($errors), Response::HTTP_BAD_REQUEST);
        }
    }


}
