<?php
namespace Widget\MemberBundle\Token;

use Lcobucci\JWT\Builder;
use Lcobucci\JWT\Parser;
use Lcobucci\JWT\Signer\Hmac\Sha256;
use Lcobucci\JWT\Token;
use Lcobucci\JWT\ValidationData;
use Widget\MemberBundle\Model\Member;


class MemberAuthToken
{
    const SECRET_PREFIX = 'widget_member.member_auth_token.';

    /** @var  Token */
    protected $token;
    /** @var  Member */
    protected $member;
    protected $secret;
    protected $payload;
    protected $issuer;
    protected $audience;


    public function __construct($secret, $issuer, $audience)
    {
        $this->secret = self::SECRET_PREFIX.$secret;
        $this->issuer = $issuer;
        $this->audience = $audience;
    }

    public function setJWTToken($jwt)
    {
        try {
            $token = (new Parser())->parse($jwt);
            if ($token->verify(new Sha256(), $this->secret)) {
                $data = new ValidationData();
                $data->setCurrentTime(time());
                $data->setIssuer($this->issuer);
                $data->setAudience($this->audience);
                if(!$token->validate($data)){
                    return;
                }
                $this->token = $token;
                $this->payload = json_decode(json_encode($token->getClaims()), true);
            }
        }
        catch(\Exception $e){
            return;
        }
    }

    public function setMember(Member $member)
    {
        $this->payload['member'] = array(
            'uid' => $member->getId(),
            'email' => $member->getEmail(),
        );
    }

    public function getPayload()
    {
        return $this->payload;
    }

    public function getAuthData()
    {
        return isset($this->payload['member'])?$this->payload['member']:null;
    }

    public function getAll()
    {
        if(!isset($this->payload['data'])){
            return null;
        }
        return $this->payload['data'];
    }

    public function get($key)
    {
        $data = $this->getAll();
        if(!isset($data[$key])){
            return null;
        }
        return $data[$key];
    }

    public function set($key, $data)
    {
        if(!isset($this->payload['data'])){
            $this->payload['data'] = array();
        }
        $this->payload['data'][$key] = $data;
    }

    /**
     * @return Token|null
     */
    public function getJWTToken()
    {
        if($this->token === null && $this->payload){
            $this->signToken();
        }
        return $this->token;
    }

    protected function signToken()
    {
        $signer = new Sha256();
        $builder = new Builder();
        $builder
            ->setIssuedAt(time())
            ->setNotBefore(time())
            ->setIssuer($this->issuer)
            ->setAudience($this->audience)
            ->setExpiration(time()+3600)
        ;
        foreach ($this->payload as $key => $data) {
            $builder->set($key, $data);
        }

        $this->token = $builder
            ->sign($signer, $this->secret)
            ->getToken();
    }
}
