<?php
namespace Widget\MemberBundle\Tests\Security;


use Backend\BaseBundle\Tests\Fixture\BaseTestCase;
use Widget\MemberBundle\Security\MemberSoldiumPassword;

class MemberSoldiumPasswordTest extends BaseTestCase
{
    public function test_create_md5()
    {
        //arrange
        $password = 'test';
        $encoder = new MemberSoldiumPassword(MemberSoldiumPassword::MD5);

        //act
        $result = $encoder->create($password);

        //assert
        $this->assertStringStartsWith('$1$', $result);
        $this->assertTrue(password_verify($password, $result));
    }

    public function test_create_sha256()
    {
        //arrange
        $password = 'test';
        $encoder = new MemberSoldiumPassword(MemberSoldiumPassword::SHA256);

        //act
        $result = $encoder->create($password);

        //assert
        $this->assertStringStartsWith('$5$', $result);
        $this->assertTrue(password_verify($password, $result));
    }

    public function test_create_sha512()
    {
        //arrange
        $password = 'test';
        $encoder = new MemberSoldiumPassword(MemberSoldiumPassword::SHA512);

        //act
        $result = $encoder->create($password);

        //assert
        $this->assertStringStartsWith('$6$', $result);
        $this->assertTrue(password_verify($password, $result));
    }

    public function test_create_blowfish()
    {
        //arrange
        $password = 'test';
        $encoder = new MemberSoldiumPassword(MemberSoldiumPassword::BLOWFISH);

        //act
        $result = $encoder->create($password);

        //assert
        $this->assertStringStartsWith('$2a$', $result);
        $this->assertTrue(password_verify($password, $result));
    }

    public function test_create_scrypt()
    {
        if(!function_exists('sodium_crypto_pwhash_scryptsalsa208sha256_str_verify')) {
            $this->markTestSkipped('scrypt not exists');
        }

        if(PHP_VERSION_ID < 70200){
            $this->markTestSkipped('skip scrypt test if php less than 7.2.0');
        }
        
        //arrange
        $password = 'test';
        $encoder = new MemberSoldiumPassword(MemberSoldiumPassword::SODIUM_SCRYPT);

        //act
        $result = $encoder->create($password);

        //assert
        $this->assertStringStartsWith('$7$C6', $result);
        $this->assertTrue(sodium_crypto_pwhash_str_verify($result, $password));
    }

    public function test_create_argon2()
    {
        //arrange
        $password = 'test';
        $encoder = new MemberSoldiumPassword(MemberSoldiumPassword::SODIUM_ARGON2);

        //act
        $result = $encoder->create($password);

        //assert
        $this->assertStringStartsWith('$argon2i', $result);
        $this->assertTrue(sodium_crypto_pwhash_str_verify($result, $password));
    }

    public function test_verify_md5_no_salt()
    {
        //arrange
        $plainPassword = 'test';
        $hash = md5($plainPassword);
        $encoder = new MemberSoldiumPassword();

        //act
        $result = $encoder->verify($plainPassword, $hash);

        //assert
        $this->assertTrue($result);
    }

    public function test_verify_bcrypt_md5()
    {
        //arrange
        $plainPassword = 'rasmuslerdorf';
        $hash = '$1$rasmusle$rISCgZzpwk3UhDidwXvin0';
        $encoder = new MemberSoldiumPassword();

        //act
        $result = $encoder->verify($plainPassword, $hash);

        //assert
        $this->assertTrue($result);
    }

    public function test_verify_bcrypt_blowfish()
    {
        //arrange
        $plainPassword = 'rasmuslerdorf';
        $hash = '$2a$07$usesomesillystringfore2uDLvp1Ii2e./U9C8sBjqp8I90dH6hi';
        $encoder = new MemberSoldiumPassword();

        //act
        $result = $encoder->verify($plainPassword, $hash);

        //assert
        $this->assertTrue($result);
    }

    public function test_verify_bcrypt_sha256()
    {
        //arrange
        $plainPassword = 'rasmuslerdorf';
        $hash = '$5$rounds=5000$usesomesillystri$KqJWpanXZHKq2BOB43TSaYhEWsQ1Lr5QNyPCDH/Tp.6';
        $encoder = new MemberSoldiumPassword();

        //act
        $result = $encoder->verify($plainPassword, $hash);

        //assert
        $this->assertTrue($result);
    }

    public function test_verify_bcrypt_sha512()
    {
        //arrange
        $plainPassword = 'rasmuslerdorf';
        $hash = '$6$rounds=5000$usesomesillystri$D4IrlXatmP7rx3P3InaxBeoomnAihCKRVQP22JZ6EY47Wc6BkroIuUUBOov1i.S5KPgErtP/EN5mcO.ChWQW21';
        $encoder = new MemberSoldiumPassword();

        //act
        $result = $encoder->verify($plainPassword, $hash);

        //assert
        $this->assertTrue($result);
    }

    public function test_verify_soldium_argon2()
    {
        //arrange
        $plainPassword = '12853714';
        $hash = '$argon2i$v=19$m=32768,t=4,p=1$px6iy91RbGKvokpDBtx96g$z+zQScO1w6eDpXBwbsVWbwzL06Dnkhz9K35Q5pGmq8E';
        $encoder = new MemberSoldiumPassword();
        //act
        $result = $encoder->verify($plainPassword, $hash);

        //assert
        $this->assertTrue($result);
    }

    public function test_verify_soldium_scrypt()
    {
        if(!function_exists('sodium_crypto_pwhash_scryptsalsa208sha256_str_verify')) {
            $this->markTestSkipped('scrypt not exists');
        }

        if(PHP_VERSION_ID < 70200){
            $this->markTestSkipped('skip scrypt test if php less than 7.2.0');
        }

        //arrange
        $plainPassword = 'test';
        $hash = '$7$C6..../....WTYVNHFJVUm7jtaq52VbD2PSkQ4fa1ovrQkSYB6/LO.$S7MZxfph020et1/niVMt2Z8/a8trVQF/ZYUsPLwRI21';
        $encoder = new MemberSoldiumPassword();
        //act
        $result = $encoder->verify($plainPassword, $hash);

        //assert
        $this->assertTrue($result);
    }
}