<?php
namespace Widget\MemberBundle\Tests\Controller\API;

use Backend\BaseBundle\Service\ReCaptchaVerifyService;
use Backend\BaseBundle\Service\SendMailService;
use Backend\BaseBundle\Tests\Fixture\BaseWebTestCase;
use Symfony\Component\HttpFoundation\Response;
use Widget\MemberBundle\Model;
use Widget\MemberBundle\Model\MemberQuery;
use Widget\MemberBundle\Service\OAuthService;
use Widget\MemberBundle\Service\RecaptchaEnableChecker;
use Widget\MemberBundle\Token\MemberAuthToken;

/**
 * @group functional
 */
class MainControllerTest extends BaseWebTestCase
{
    public function test_registerAction()
    {
        //arrange
        $data = array(
            'name' => 'Ricky Su',
            'email' => 'ricky@dgfactor.com',
            'password' => '12345678',
        );

        //act
        $this->client->request('POST', $this->generateUrl('widget_member_api_main_register'), array(), array(), array(), json_encode($data));

        $response = json_decode($result = $this->client->getResponse()->getContent(), true);
        $mailCollector = $this->client->getProfile()->getCollector('swiftmailer');

        $to = $mailCollector->getMessages('site')[0]->getTo();
        $mailSubject = $mailCollector->getMessages('site')[0]->getSubject();
        $mailContent = $mailCollector->getMessages('site')[0]->getBody();
        $group = Model\GroupQuery::create()->findPk(1);

        //assert
        $this->assertGreaterThan(0, $response['id']);
        $this->assertEquals(1, $mailCollector->getMessageCount());
        $member = Model\MemberQuery::create()->findPk($response['id']);

        $this->assertEquals(array('ricky@dgfactor.com' => 'Ricky Su'), $to);
        $this->assertEquals("hi {$member->getName()} this is your register mail!", $mailSubject);
        $this->assertEquals("register mail: {$member->getName()}-{$member->getEmail()}-{$member->getId()}-{$member->getConfirmToken()}-{$member->getTokenExpiredAt('Y-m-d H:i:s')}", $mailContent);
        // 確定有寫入預設群組
        $this->assertEquals($group, $member->getGroups()->getFirst());
        $member->delete();
    }

    public function test_registerAction_bad_format()
    {
        //arrange
        $data = array(
            'name' => 'Shanhu Chang',
            'email' => 'shanhu',
            'password' => '12345678',
        );

        //act
        $this->client->request('POST', $this->generateUrl('widget_member_api_main_register'), array(), array(), array(), json_encode($data));
        $response = $this->client->getResponse();

        //assert
        $this->assertEquals(Response::HTTP_BAD_REQUEST, $response->getStatusCode());
    }

    public function test_registerAction_member_exist()
    {
        //arrange
        $data = array(
            'name' => 'Shanhu Chang',
            'email' => 'shanhu_exist@example.com',
            'password' => '12345678',
        );

        //act
        $this->client->request('POST', $this->generateUrl('widget_member_api_main_register'), array(), array(), array(), json_encode($data));
        $response = $this->client->getResponse();

        //assert
        $this->assertEquals(Response::HTTP_FORBIDDEN, $response->getStatusCode());
    }

    public function test_resendVerifyMailAction()
    {
        //arrange
        $member = new Model\Member();
        $member
            ->setConfirmToken(null)
            ->setTokenExpiredAt(null)
            ->setEnabled(false)
            ->setName('Ricky Su')
            ->setEmail('ricky@dgfactor.com')
            ->setPlainPassword('12345')
        ;
        $this->client->getContainer()->get('member_manager')->updateMember($member);

        $data = array(
            'email' => 'ricky@dgfactor.com',
        );

        //act
        $this->client->request('PUT', $this->generateUrl('widget_member_api_main_resendverifymail'), array(), array(), array(), json_encode($data));

        $response = json_decode($this->client->getResponse()->getContent(), true);
        $mailCollector = $this->client->getProfile()->getCollector('swiftmailer');

        $to = $mailCollector->getMessages('site')[0]->getTo();
        $mailSubject = $mailCollector->getMessages('site')[0]->getSubject();
        $mailContent = $mailCollector->getMessages('site')[0]->getBody();

        //assert
        $member->reload();
        $this->assertEquals(1, $mailCollector->getMessageCount());
        $this->assertEquals(array('ricky@dgfactor.com' => 'Ricky Su'), $to);
        $this->assertEquals("hi {$member->getName()} this is your register mail!", $mailSubject);
        $this->assertEquals("register mail: {$member->getName()}-{$member->getEmail()}-{$member->getId()}-{$member->getConfirmToken()}-{$member->getTokenExpiredAt('Y-m-d H:i:s')}", $mailContent);
        $member->delete();
    }

    public function test_resendVerifyMailAction_bad_format()
    {
        //arrange
        $member = new Model\Member();
        $member
            ->setConfirmToken(null)
            ->setTokenExpiredAt(null)
            ->setEnabled(false)
            ->setName('Ricky Su')
            ->setEmail('ricky@dgfactor.com')
            ->setPlainPassword('12345')
        ;
        $this->client->getContainer()->get('member_manager')->updateMember($member);

        $data = array(
            'email' => 'ricky',
        );

        //act
        $this->client->request('PUT', $this->generateUrl('widget_member_api_main_resendverifymail'), array(), array(), array(), json_encode($data));

        $response = $this->client->getResponse();
        $result = json_decode($response->getContent(), true);

        //assert
        $this->assertEquals(Response::HTTP_FORBIDDEN, $response->getStatusCode());
        $member->delete();
    }

    public function test_resendVerifyMailAction_not_exist()
    {
        //arrange
        $member = new Model\Member();
        $member
            ->setConfirmToken(null)
            ->setTokenExpiredAt(null)
            ->setEnabled(false)
            ->setName('Ricky Su')
            ->setEmail('ricky@dgfactor.com')
            ->setPlainPassword('12345')
        ;
        $this->client->getContainer()->get('member_manager')->updateMember($member);

        $data = array(
            'email' => 'ricky@dgfactor.com.tw',
        );

        //act
        $this->client->request('PUT', $this->generateUrl('widget_member_api_main_resendverifymail'), array(), array(), array(), json_encode($data));
        $response = $this->client->getResponse();
        $result = json_decode($response->getContent(), true);
        $member->delete();

        //assert
        $this->assertEquals(Response::HTTP_FORBIDDEN, $response->getStatusCode());
    }

    public function test_resendVerifyMailAction_been_enabled()
    {
        //arrange
        $member = new Model\Member();
        $member
            ->setConfirmToken(null)
            ->setTokenExpiredAt(null)
            ->setEnabled(true)
            ->setName('Ricky Su')
            ->setEmail('ricky@dgfactor.com')
            ->setPlainPassword('12345')
        ;
        $this->client->getContainer()->get('member_manager')->updateMember($member);

        $data = array(
            'email' => 'ricky@dgfactor.com',
        );

        //act
        $this->client->request('PUT', $this->generateUrl('widget_member_api_main_resendverifymail'), array(), array(), array(), json_encode($data));
        $response = $this->client->getResponse();
        $result = json_decode($response->getContent(), true);
        $member->delete();

        //assert
        $this->assertEquals(Response::HTTP_CONFLICT, $response->getStatusCode());
    }

    public function test_enableAction_bad_token()
    {
        //arrange
        $token = base_convert(sha1(uniqid(mt_rand(), true)), 16, 36);
        $member = new Model\Member();
        $member
            ->setConfirmToken($token)
            ->setTokenExpiredAt(time()+86400)
            ->setName('ricky')
            ->setEmail('ricky@dgfactor.com')
            ->setPlainPassword('12345')
        ;
        $this->client->getContainer()->get('member_manager')->updateMember($member);
        $data = array(
            'token' => 'bad_token',
        );

        //act
        $this->client->request('PUT', $this->generateUrl('widget_member_api_main_enable', array('id' => $member->getId())), array(), array(), array(), json_encode($data));
        $response = $this->client->getResponse();
        $member->delete();

        //assert
        $this->assertEquals(Response::HTTP_UNAUTHORIZED, $response->getStatusCode());
    }

    public function test_enableAction_bad_member()
    {
        //arrange
        $token = base_convert(sha1(uniqid(mt_rand(), true)), 16, 36);
        $member = new Model\Member();
        $member
            ->setConfirmToken($token)
            ->setTokenExpiredAt(time()+86400)
            ->setName('ricky')
            ->setEmail('ricky@dgfactor.com')
            ->setPlainPassword('12345')
        ;
        $this->client->getContainer()->get('member_manager')->updateMember($member);
        $data = array(
            'token' => 'bad_token',
        );

        //act
        $this->client->request('PUT', $this->generateUrl('widget_member_api_main_enable', array('id' => 'bad_id')), array(), array(), array(), json_encode($data));
        $response = $this->client->getResponse();
        $member->delete();

        //assert
        $this->assertTrue($response->isNotFound());
    }

    public function test_enableAction_member_enabled()
    {
        //arrange
        $token = base_convert(sha1(uniqid(mt_rand(), true)), 16, 36);
        $member = new Model\Member();
        $member
            ->setEnabled(true)
            ->setName('ricky')
            ->setEmail('ricky@dgfactor.com')
            ->setPlainPassword('12345')
        ;
        $this->client->getContainer()->get('member_manager')->updateMember($member);
        $data = array(
            'token' => 'bad_token',
        );

        //act
        $this->client->request('PUT', $this->generateUrl('widget_member_api_main_enable', array('id' => $member->getId())), array(), array(), array(), json_encode($data));
        $response = $this->client->getResponse();
        $member->delete();

        //assert
        $this->assertEquals(Response::HTTP_UNAUTHORIZED, $response->getStatusCode());
    }

    public function test_enableAction_ok()
    {
        //arrange
        $token = base_convert(sha1(uniqid(mt_rand(), true)), 16, 36);
        $member = new Model\Member();
        $member
            ->setConfirmToken($token)
            ->setTokenExpiredAt(time()+86400)
            ->setName('ricky')
            ->setEmail('ricky@dgfactor.com')
            ->setPlainPassword('12345')
            ;
        $this->client->getContainer()->get('member_manager')->updateMember($member);
        $data = array(
            'token' => $token,
        );

        //act
        $this->client->request('PUT', $this->generateUrl('widget_member_api_main_enable', array('id' => $member->getId())), array(), array(), array(), json_encode($data));
        $response = $this->client->getResponse();
        $result = json_decode($this->client->getResponse()->getContent(), true);
        $member->delete();

        //assert
        $this->assertTrue($response->isOk());
    }

    public function test_forgetpasswordAction_recaptcha_disabled()
    {
        //arrange
        $recaptchaResponse = 'captcha response';
        $member = new Model\Member();
        $member
            ->setConfirmToken(null)
            ->setTokenExpiredAt(null)
            ->setEnabled(true)
            ->setName('Ricky Su')
            ->setEmail('ricky@dgfactor.com')
            ->setPlainPassword('12345')
        ;
        $this->client->getContainer()->get('member_manager')->updateMember($member);
        $reCaptchaVerifyService = $this->getMockBuilder(ReCaptchaVerifyService::class)
            ->setMethods(array('verify'))
            ->getMock();
        $reCaptchaVerifyService
            ->expects($this->never())
            ->method('verify')
            ->with($recaptchaResponse)
            ->willReturn(false);
        $recaptchaEnableChecker = $this->getMockBuilder(RecaptchaEnableChecker::class)
            ->setMethods(array('isEnable'))
            ->getMock();
        $recaptchaEnableChecker
            ->expects($this->once())
            ->method('isEnable')
            ->with('forgetpassword')
            ->willReturn(false);

        $this->client->getContainer()->set('recaptcha.enable.checker', $recaptchaEnableChecker);
        $this->client->getContainer()->set('backend_base.recaptcha_verify', $reCaptchaVerifyService);
        $data = array(
            'email' => 'ricky@dgfactor.com',
        );

        //act
        $this->client->request('PUT', $this->generateUrl('widget_member_api_main_forgetpassword', array()), array(), array(), array(), json_encode($data));

        $response = json_decode($this->client->getResponse()->getContent(), true);
        $mailCollector = $this->client->getProfile()->getCollector('swiftmailer');

        $to = $mailCollector->getMessages('site')[0]->getTo();
        $mailSubject = $mailCollector->getMessages('site')[0]->getSubject();
        $mailContent = $mailCollector->getMessages('site')[0]->getBody();

        //assert
        $member->reload();
        $this->assertEquals(1, $mailCollector->getMessageCount());
        $this->assertEquals(array('ricky@dgfactor.com' => 'Ricky Su'), $to);
        $this->assertEquals("hi {$member->getName()} this is your forget password mail!", $mailSubject);
        $this->assertEquals("forgetpassword mail: {$member->getName()}-{$member->getEmail()}-{$member->getId()}-{$member->getConfirmToken()}-{$member->getTokenExpiredAt('Y-m-d H:i:s')}", $mailContent);
        $member->delete();
    }

    public function test_forgetpasswordAction_recaptcha_enabled_no_captcha()
    {
        //arrange
        $recaptchaResponse = 'captcha response';
        $member = new Model\Member();
        $member
            ->setConfirmToken(null)
            ->setTokenExpiredAt(null)
            ->setEnabled(true)
            ->setName('Ricky Su')
            ->setEmail('ricky@dgfactor.com')
            ->setPlainPassword('12345')
        ;
        $this->client->getContainer()->get('member_manager')->updateMember($member);
        $reCaptchaVerifyService = $this->getMockBuilder(ReCaptchaVerifyService::class)
            ->setMethods(array('verify'))
            ->getMock();
        $reCaptchaVerifyService
            ->expects($this->never())
            ->method('verify')
            ->with($recaptchaResponse)
            ->willReturn(false);
        $recaptchaEnableChecker = $this->getMockBuilder(RecaptchaEnableChecker::class)
            ->setMethods(array('isEnable'))
            ->getMock();
        $recaptchaEnableChecker
            ->expects($this->once())
            ->method('isEnable')
            ->with('forgetpassword')
            ->willReturn(true);

        $this->client->getContainer()->set('recaptcha.enable.checker', $recaptchaEnableChecker);
        $this->client->getContainer()->set('backend_base.recaptcha_verify', $reCaptchaVerifyService);
        $data = array(
            'email' => 'ricky@dgfactor.com',
        );

        //act
        $this->client->request('PUT', $this->generateUrl('widget_member_api_main_forgetpassword', array()), array(), array(), array(), json_encode($data));

        $response = $this->client->getResponse();
        $member->delete();

        //assert
        $this->assertFalse($response->isOk());
        $this->assertEquals(Response::HTTP_BAD_REQUEST, $response->getStatusCode());
    }

    public function test_forgetpasswordAction_recaptcha_enabled_verify_fail()
    {
        //arrange
        $recaptchaResponse = 'captcha response';
        $member = new Model\Member();
        $member
            ->setConfirmToken(null)
            ->setTokenExpiredAt(null)
            ->setEnabled(true)
            ->setName('Ricky Su')
            ->setEmail('ricky@dgfactor.com')
            ->setPlainPassword('12345')
        ;
        $this->client->getContainer()->get('member_manager')->updateMember($member);
        $reCaptchaVerifyService = $this->getMockBuilder(ReCaptchaVerifyService::class)
            ->setMethods(array('verify'))
            ->getMock();
        $reCaptchaVerifyService
            ->expects($this->once())
            ->method('verify')
            ->with($recaptchaResponse)
            ->willReturn(false);
        $recaptchaEnableChecker = $this->getMockBuilder(RecaptchaEnableChecker::class)
            ->setMethods(array('isEnable'))
            ->getMock();
        $recaptchaEnableChecker
            ->expects($this->once())
            ->method('isEnable')
            ->with('forgetpassword')
            ->willReturn(true);

        $this->client->getContainer()->set('recaptcha.enable.checker', $recaptchaEnableChecker);
        $this->client->getContainer()->set('backend_base.recaptcha_verify', $reCaptchaVerifyService);
        $data = array(
            'email' => 'ricky@dgfactor.com',
            'recaptcha' => $recaptchaResponse,
        );

        //act
        $this->client->request('PUT', $this->generateUrl('widget_member_api_main_forgetpassword', array()), array(), array(), array(), json_encode($data));

        $response = $this->client->getResponse();
        $member->delete();

        //assert
        $this->assertFalse($response->isOk());
        $this->assertEquals(Response::HTTP_BAD_REQUEST, $response->getStatusCode());
    }

    public function test_forgetpasswordAction_recaptcha_enabled_verify_ok()
    {
        //arrange
        $recaptchaResponse = 'captcha response';
        $member = new Model\Member();
        $member
            ->setConfirmToken(null)
            ->setTokenExpiredAt(null)
            ->setEnabled(true)
            ->setName('Ricky Su')
            ->setEmail('ricky@dgfactor.com')
            ->setPlainPassword('12345')
        ;
        $this->client->getContainer()->get('member_manager')->updateMember($member);
        $reCaptchaVerifyService = $this->getMockBuilder(ReCaptchaVerifyService::class)
            ->setMethods(array('verify'))
            ->getMock();
        $reCaptchaVerifyService
            ->expects($this->once())
            ->method('verify')
            ->with($recaptchaResponse)
            ->willReturn(true);
        $recaptchaEnableChecker = $this->getMockBuilder(RecaptchaEnableChecker::class)
            ->setMethods(array('isEnable'))
            ->getMock();
        $recaptchaEnableChecker
            ->expects($this->once())
            ->method('isEnable')
            ->with('forgetpassword')
            ->willReturn(true);

        $this->client->getContainer()->set('recaptcha.enable.checker', $recaptchaEnableChecker);
        $this->client->getContainer()->set('backend_base.recaptcha_verify', $reCaptchaVerifyService);
        $data = array(
            'email' => 'ricky@dgfactor.com',
            'recaptcha' => $recaptchaResponse,
        );

        //act
        $this->client->request('PUT', $this->generateUrl('widget_member_api_main_forgetpassword', array()), array(), array(), array(), json_encode($data));

        $response = $this->client->getResponse();
        $member->delete();

        //assert
        $this->assertTrue($response->isOk());
    }

    public function test_forgetpasswordAction()
    {
        //arrange
        $member = new Model\Member();
        $member
            ->setConfirmToken(null)
            ->setTokenExpiredAt(null)
            ->setEnabled(true)
            ->setName('Ricky Su')
            ->setEmail('ricky@dgfactor.com')
            ->setPlainPassword('12345')
        ;
        $this->client->getContainer()->get('member_manager')->updateMember($member);

        $data = array(
            'email' => 'ricky@dgfactor.com',
        );

        //act
        $this->client->request('PUT', $this->generateUrl('widget_member_api_main_forgetpassword', array()), array(), array(), array(), json_encode($data));

        $response = json_decode($this->client->getResponse()->getContent(), true);
        $mailCollector = $this->client->getProfile()->getCollector('swiftmailer');

        $to = $mailCollector->getMessages('site')[0]->getTo();
        $mailSubject = $mailCollector->getMessages('site')[0]->getSubject();
        $mailContent = $mailCollector->getMessages('site')[0]->getBody();

        //assert
        $member->reload();
        $this->assertEquals(1, $mailCollector->getMessageCount());
        $this->assertEquals(array('ricky@dgfactor.com' => 'Ricky Su'), $to);
        $this->assertEquals("hi {$member->getName()} this is your forget password mail!", $mailSubject);
        $this->assertEquals("forgetpassword mail: {$member->getName()}-{$member->getEmail()}-{$member->getId()}-{$member->getConfirmToken()}-{$member->getTokenExpiredAt('Y-m-d H:i:s')}", $mailContent);
        $member->delete();
    }

    public function test_forgetpasswordAction_bad_format()
    {
        //arrange
        $member = new Model\Member();
        $member
            ->setConfirmToken(null)
            ->setTokenExpiredAt(null)
            ->setEnabled(true)
            ->setName('Ricky Su')
            ->setEmail('ricky@dgfactor.com')
            ->setPlainPassword('12345')
        ;
        $this->client->getContainer()->get('member_manager')->updateMember($member);

        $data = array(
            'email' => 'ricky',
        );

        //act
        $this->client->request('PUT', $this->generateUrl('widget_member_api_main_forgetpassword', array()), array(), array(), array(), json_encode($data));

        $response = $this->client->getResponse();
        $member->delete();

        //assert
        $this->assertEquals(Response::HTTP_FORBIDDEN, $response->getStatusCode());
    }

    public function test_forgetpasswordAction_not_exist()
    {
        //arrange
        $member = new Model\Member();
        $member
            ->setConfirmToken(null)
            ->setTokenExpiredAt(null)
            ->setEnabled(true)
            ->setName('Ricky Su')
            ->setEmail('ricky@dgfactor.com')
            ->setPlainPassword('12345')
        ;
        $this->client->getContainer()->get('member_manager')->updateMember($member);

        $data = array(
            'email' => 'ricky@dgfactor.com.tw',
        );

        //act
        $this->client->request('PUT', $this->generateUrl('widget_member_api_main_forgetpassword', array()), array(), array(), array(), json_encode($data));
        $response = $this->client->getResponse();
        $member->delete();

        //assert
        $this->assertEquals(Response::HTTP_FORBIDDEN, $response->getStatusCode());
    }

    public function test_loginAction()
    {
        //arrange
        $data = array(
            'email' => 'ricky@dgfactor.com',
            'password' => '12345',
        );

        $member = new Model\Member();
        $member
            ->setConfirmToken(null)
            ->setTokenExpiredAt(null)
            ->setEnabled(true)
            ->setName('Ricky Su')
            ->setEmail('ricky@dgfactor.com')
            ->setPlainPassword('12345')
        ;
        $this->client->getContainer()->get('member_manager')->updateMember($member);

        //act
        $this->client->request(
            'PUT',
            $this->generateUrl('widget_member_api_main_login'),
            array(),
            array(),
            array(
                'HTTP_ORIGIN' => 'http://localhost:8080',
            ),
            json_encode($data)
        );
        $response = $this->client->getResponse();
        $result = json_decode($response->getContent(), true);
        $member->delete();

        //assert
        $this->assertNotEmpty($result['token']);
    }

    public function test_loginAction_member_not_exist()
    {
        //arrange
        $data = array(
            'email' => 'shanhu@dgfactor.com',
            'password' => '12345',
        );

        //act
        $this->client->request(
            'PUT',
            $this->generateUrl('widget_member_api_main_login'),
            array(),
            array(),
            array(),
            json_encode($data)
        );
        $response = $this->client->getResponse();

        //assert
        $this->assertEquals(Response::HTTP_NOT_FOUND, $response->getStatusCode());
    }

    public function test_loginAction_email_bad_format()
    {
        //arrange
        $data = array(
            'email' => 'ricky',
            'password' => '12345',
        );

        $member = new Model\Member();
        $member
            ->setConfirmToken(null)
            ->setTokenExpiredAt(null)
            ->setEnabled(true)
            ->setName('Ricky Su')
            ->setEmail('ricky@dgfactor.com')
            ->setPlainPassword('12345')
        ;
        $this->client->getContainer()->get('member_manager')->updateMember($member);

        //act
        $this->client->request(
            'PUT',
            $this->generateUrl('widget_member_api_main_login'),
            array(),
            array(),
            array(),
            json_encode($data)
        );
        $response = $this->client->getResponse();
        $member->delete();

        //assert
        $this->assertEquals(Response::HTTP_BAD_REQUEST, $response->getStatusCode());
    }

    public function test_getMemberInfoAction_no_auth()
    {
        // arrange
        // act
        $this->client->request(
            'GET',
            $this->generateUrl('widget_member_api_main_getmemberinfo')
        );
        $response = $this->client->getResponse();

        // assert
        $this->assertEquals(Response::HTTP_UNAUTHORIZED, $response->getStatusCode());
    }

    public function test_getMemberInfoAction_bad_user()
    {
        // arrange
        $origin = 'http://localhost';
        $secret = $this->client->getContainer()->getParameter('secret');
        $member = new Model\Member();
        $member->setId(32768);    // 沒這會員ID

        $authToken = new MemberAuthToken($secret, $origin, $origin);
        $authToken->setMember($member);
        $token = $authToken->getJWTToken();
        // act
        $this->client->request(
            'GET',
            $this->generateUrl('widget_member_api_main_getmemberinfo'),
            array(),
            array(),
            array(
                'HTTP_ORIGIN' => $origin,
                'HTTP_AUTHORIZATION' => "Bearer $token",
            )
        );
        $response = $this->client->getResponse();
        $resultData = $response->getContent();

        // assert
        $this->assertEquals(Response::HTTP_FORBIDDEN, $response->getStatusCode());
    }

    public function test_getMemberInfoAction()
    {
        // arrange
        $origin = 'http://localhost';
        $secret = $this->client->getContainer()->getParameter('secret');
        $member = Model\MemberQuery::create()->findPk(1000);
        $memberInfo = $member->getMemberInfo();
        $expectArray = array(
            'name' => $memberInfo->getName(),
            'phone' => $memberInfo->getPhone(),
            'phone2' => $memberInfo->getPhone2(),
            'email' => $memberInfo->getEmail(),
            'zip' => $memberInfo->getZip(),
            'address' => $memberInfo->getAddress(),
            'photos' => $memberInfo->getPhotos(),
        );
        $expectData = json_encode($expectArray);

        $authToken = new MemberAuthToken($secret, $origin, $origin);
        $authToken->setMember($member);
        $token = $authToken->getJWTToken();

        // act
        $this->client->request(
            'GET',
            $this->generateUrl('widget_member_api_main_getmemberinfo'),
            array(),
            array(),
            array(
                'HTTP_ORIGIN' => $origin,
                'HTTP_AUTHORIZATION' => "Bearer $token",
            )
        );
        $response = $this->client->getResponse();
        $resultData = $response->getContent();

        // assert
        $this->assertTrue($response->isOk());
        $this->assertEquals($expectData, $resultData);
    }

    public function test_passwordLoginAction_bad_token()
    {
        //arrange
        $audience = 'http://localhost:8080';
        $issuer = 'http://localhost';
        $secret = $this->client->getContainer()->getParameter('secret');
        $data = array(
            'email' => 'ricky@dgfactor.com',
            'password' => '1234567',
        );

        $member = new Model\Member();
        $member
            ->setConfirmToken(null)
            ->setTokenExpiredAt(null)
            ->setEnabled(true)
            ->setName('Ricky Su')
            ->setEmail('ricky@dgfactor.com')
            ->setPlainPassword('12345')
        ;
        $this->client->getContainer()->get('member_manager')->updateMember($member);
        $memberAuthToken = new MemberAuthToken($secret, $issuer, $audience);
        $memberAuthToken->setMember($member);

        //act
        $this->client->request(
            'PUT',
            $this->generateUrl('widget_member_api_main_passwordlogin'),
            array(),
            array(),
            array(
                //'Authorization' => "Bearer {$memberAuthToken->getJWTToken()}",
            ),
            json_encode($data)
        );
        $response = $this->client->getResponse();
        $member->delete();

        //assert
        $this->assertEquals(Response::HTTP_UNAUTHORIZED, $response->getStatusCode());
    }

    public function test_passwordLoginAction_member_not_exist()
    {
        //arrange
        $audience = 'http://localhost:8080';
        $issuer = 'http://localhost';
        $secret = $this->client->getContainer()->getParameter('secret');
        $data = array(
            'email' => 'ricky@dgfactor.com',
            'password' => '1234567',
        );

        $member = new Model\Member();
        $member
            ->setConfirmToken(null)
            ->setTokenExpiredAt(null)
            ->setEnabled(true)
            ->setName('Ricky Su')
            ->setEmail('ricky@dgfactor.com')
            ->setPlainPassword('12345')
        ;
        $this->client->getContainer()->get('member_manager')->updateMember($member);
        $memberAuthToken = new MemberAuthToken($secret, $issuer, $audience);
        $memberAuthToken->setMember($member);
        $member->delete();

        //act
        $this->client->request(
            'PUT',
            $this->generateUrl('widget_member_api_main_passwordlogin'),
            array(),
            array(),
            array(
                'HTTP_AUTHORIZATION' => "Bearer {$memberAuthToken->getJWTToken()}",
            ),
            json_encode($data)
        );
        $response = $this->client->getResponse();

        //assert
        $this->assertEquals(Response::HTTP_CONFLICT, $response->getStatusCode());
    }

    public function test_passwordLoginAction_member_not_enabled()
    {
        //arrange
        $audience = 'http://localhost:8080';
        $issuer = 'http://localhost';
        $secret = $this->client->getContainer()->getParameter('secret');
        $data = array(
            'email' => 'ricky@dgfactor.com',
            'password' => '1234567',
        );

        $member = new Model\Member();
        $member
            ->setConfirmToken(null)
            ->setTokenExpiredAt(null)
            ->setEnabled(false)
            ->setName('Ricky Su')
            ->setEmail('ricky@dgfactor.com')
            ->setPlainPassword('12345')
        ;
        $this->client->getContainer()->get('member_manager')->updateMember($member);
        $memberAuthToken = new MemberAuthToken($secret, $issuer, $audience);
        $memberAuthToken->setMember($member);

        //act
        $this->client->request(
            'PUT',
            $this->generateUrl('widget_member_api_main_passwordlogin'),
            array(),
            array(),
            array(
                'HTTP_AUTHORIZATION' => "Bearer {$memberAuthToken->getJWTToken()}",
            ),
            json_encode($data)
        );
        $response = $this->client->getResponse();
        $member->delete();

        //assert
        $this->assertEquals(Response::HTTP_CONFLICT, $response->getStatusCode());
    }

    public function test_passwordLoginAction_member_locked()
    {
        //arrange
        $audience = 'http://localhost:8080';
        $issuer = 'http://localhost';
        $secret = $this->client->getContainer()->getParameter('secret');
        $data = array(
            'email' => 'ricky@dgfactor.com',
            'password' => '1234567',
        );

        $member = new Model\Member();
        $member
            ->setConfirmToken(null)
            ->setTokenExpiredAt(null)
            ->setEnabled(true)
            ->setLocked(true)
            ->setName('Ricky Su')
            ->setEmail('ricky@dgfactor.com')
            ->setPlainPassword('12345')
        ;
        $this->client->getContainer()->get('member_manager')->updateMember($member);
        $memberAuthToken = new MemberAuthToken($secret, $issuer, $audience);
        $memberAuthToken->setMember($member);

        //act
        $this->client->request(
            'PUT',
            $this->generateUrl('widget_member_api_main_passwordlogin'),
            array(),
            array(),
            array(
                'HTTP_AUTHORIZATION' => "Bearer {$memberAuthToken->getJWTToken()}",
            ),
            json_encode($data)
        );
        $response = $this->client->getResponse();
        $member->delete();

        //assert
        $this->assertEquals(Response::HTTP_CONFLICT, $response->getStatusCode());
    }

    public function test_passwordLoginAction_ok()
    {
        //arrange
        $audience = 'http://localhost:8080';
        $issuer = 'http://localhost';
        $secret = $this->client->getContainer()->getParameter('secret');
        $data = array(
            'email' => 'ricky@dgfactor.com',
            'password' => '1234567',
        );

        $member = new Model\Member();
        $member
            ->setConfirmToken(null)
            ->setTokenExpiredAt(null)
            ->setEnabled(true)
            ->setName('Ricky Su')
            ->setEmail('ricky@dgfactor.com')
            ->setPlainPassword('12345')
        ;
        $memberManager = $this->client->getContainer()->get('member_manager');
        $memberManager->updateMember($member);
        $memberAuthToken = new MemberAuthToken($secret, $issuer, $audience);
        $memberAuthToken->setMember($member);

        //act
        $this->client->request(
            'PUT',
            $this->generateUrl('widget_member_api_main_passwordlogin'),
            array(),
            array(),
            array(
                'HTTP_AUTHORIZATION' => "Bearer {$memberAuthToken->getJWTToken()}",
            ),
            json_encode($data)
        );
        $response = $this->client->getResponse();
        $member->reload();

        //assert
        $this->assertTrue($response->isOk());
        $this->assertTrue($memberManager->verifyMember($member, '1234567'));
        $member->delete();
    }

    public function test_passwordAction_bad_site()
    {
        //arrange
        $data = array(
            'email' => 'ricky@dgfactor.com',
            'password' => '1234567',
        );

        $member = new Model\Member();
        $member
            ->setConfirmToken(null)
            ->setTokenExpiredAt(null)
            ->setEnabled(true)
            ->setName('Ricky Su')
            ->setEmail('ricky@dgfactor.com')
            ->setPlainPassword('12345')
        ;
        $memberManager = $this->client->getContainer()->get('member_manager');
        $memberManager->updateMember($member);

        //act
        $this->client->request(
            'PUT',
            $this->generateUrl('widget_member_api_main_password', array('id' => $member->getId())),
            array(),
            array(),
            array(),
            json_encode($data)
        );
        $response = $this->client->getResponse();
        $member->delete();

        //assert
        $this->assertEquals(Response::HTTP_UNAUTHORIZED, $response->getStatusCode());
    }

    public function test_passwordAction_member_is_locked()
    {
        //arrange
        $data = array(
            'email' => 'ricky@dgfactor.com',
            'password' => '1234567',
        );

        $member = new Model\Member();
        $member
            ->setConfirmToken(null)
            ->setTokenExpiredAt(null)
            ->setEnabled(true)
            ->setLocked(true)
            ->setName('Ricky Su')
            ->setEmail('ricky@dgfactor.com')
            ->setPlainPassword('12345')
        ;
        $memberManager = $this->client->getContainer()->get('member_manager');
        $memberManager->updateMember($member);

        //act
        $this->client->request(
            'PUT',
            $this->generateUrl('widget_member_api_main_password', array('id' => $member->getId())),
            array(),
            array(),
            array(),
            json_encode($data)
        );
        $response = $this->client->getResponse();
        $member->delete();

        //assert
        $this->assertEquals(Response::HTTP_CONFLICT, $response->getStatusCode());
    }

    public function test_passwordAction_member_no_token()
    {
        //arrange
        $data = array(
            'email' => 'ricky@dgfactor.com',
            'password' => '1234567',
        );

        $member = new Model\Member();
        $member
            ->setConfirmToken(null)
            ->setTokenExpiredAt(null)
            ->setEnabled(true)
            ->setName('Ricky Su')
            ->setEmail('ricky@dgfactor.com')
            ->setPlainPassword('12345')
        ;
        $memberManager = $this->client->getContainer()->get('member_manager');
        $memberManager->updateMember($member);

        //act
        $this->client->request(
            'PUT',
            $this->generateUrl('widget_member_api_main_password', array('id' => $member->getId())),
            array(),
            array(),
            array(),
            json_encode($data)
        );
        $response = $this->client->getResponse();
        $member->delete();

        //assert
        $this->assertEquals(Response::HTTP_UNAUTHORIZED, $response->getStatusCode());
    }

    public function test_passwordAction_member_bad_token()
    {
        //arrange
        $data = array(
            'email' => 'ricky@dgfactor.com',
            'password' => '1234567',
        );

        $member = new Model\Member();
        $member
            ->setConfirmToken(null)
            ->setTokenExpiredAt(null)
            ->setEnabled(true)
            ->setName('Ricky Su')
            ->setEmail('ricky@dgfactor.com')
            ->setPlainPassword('12345')
        ;
        $memberManager = $this->client->getContainer()->get('member_manager');
        $memberManager->updateMember($member);
        $data['token'] = 'bad_token';

        //act
        $this->client->request(
            'PUT',
            $this->generateUrl('widget_member_api_main_password', array('id' => $member->getId())),
            array(),
            array(),
            array(),
            json_encode($data)
        );
        $response = $this->client->getResponse();
        $member->delete();

        //assert
        $this->assertEquals(Response::HTTP_UNAUTHORIZED, $response->getStatusCode());
    }

    public function test_passwordAction_member_token_expire()
    {
        //arrange
        $data = array(
            'email' => 'ricky@dgfactor.com',
            'password' => '1234567',
        );

        $member = new Model\Member();
        $member
            ->setConfirmToken(null)
            ->setTokenExpiredAt(null)
            ->setEnabled(true)
            ->setName('Ricky Su')
            ->setEmail('ricky@dgfactor.com')
            ->setPlainPassword('12345')
        ;
        $member->regenerateConfirmToken(-10);
        $memberManager = $this->client->getContainer()->get('member_manager');
        $memberManager->updateMember($member);
        $data['token'] = $member->getConfirmToken();

        //act
        $this->client->request(
            'PUT',
            $this->generateUrl('widget_member_api_main_password', array('id' => $member->getId())),
            array(),
            array(),
            array(),
            json_encode($data)
        );
        $response = $this->client->getResponse();
        $member->delete();

        //assert
        $this->assertEquals(Response::HTTP_UNAUTHORIZED, $response->getStatusCode());
    }

    public function test_passwordAction_ok()
    {
        //arrange
        $data = array(
            'email' => 'ricky@dgfactor.com',
            'password' => '1234567',
        );

        $member = new Model\Member();
        $member
            ->setConfirmToken(null)
            ->setTokenExpiredAt(null)
            ->setName('Ricky Su')
            ->setEmail('ricky@dgfactor.com')
            ->setPlainPassword('12345')
        ;
        $member->regenerateConfirmToken();
        $memberManager = $this->client->getContainer()->get('member_manager');
        $memberManager->updateMember($member);
        $data['token'] = $member->getConfirmToken();

        //act
        $this->client->request(
            'PUT',
            $this->generateUrl('widget_member_api_main_password', array('id' => $member->getId())),
            array(),
            array(),
            array(),
            json_encode($data)
        );
        $response = $this->client->getResponse();
        $member->reload();

        //assert
        $this->assertTrue($response->isOk());
        $this->assertTrue($memberManager->verifyMember($member, '1234567'));
        $this->assertTrue($member->getEnabled());
        $member->delete();
    }

    public function test_editAction_empty_data()
    {
        //arrange
        $username = 'bubble1';
        $audience = 'http://localhost:8080';
        $issuer = 'http://localhost';
        $secret = $this->client->getContainer()->getParameter('secret');
        $data = array();
        $member = Model\MemberQuery::create()->findOneByName($username);

        $memberAuthToken = new MemberAuthToken($secret, $issuer, $audience);
        $memberAuthToken->setMember($member);

        //act
        $this->client->request(
            'PUT',
            $this->generateUrl('widget_member_api_main_edit'),
            array(),
            array(),
            array(
                'HTTP_AUTHORIZATION' => "Bearer {$memberAuthToken->getJWTToken()}",
            ),
            json_encode($data)
        );
        $response = $this->client->getResponse();

        //assert
        $this->assertEquals(Response::HTTP_BAD_REQUEST, $response->getStatusCode());
    }

    public function test_editAction_ok()
    {
        //arrange
        $username = 'bubble1';
        $audience = 'http://localhost:8080';
        $issuer = 'http://localhost';
        $secret = $this->client->getContainer()->getParameter('secret');
        $data = array(
            'name' => 'Shanhu Chang',
            'email' => 'shanhu@dgfactor.com.tw',
            'phone' => '0911-111-111',
            'phone2' => '0911-111-111',
            'zip' => '220',
            'address' => '新北市板橋區中山路一段1號',
        );
        $member = Model\MemberQuery::create()->findOneByName($username);

        $memberAuthToken = new MemberAuthToken($secret, $issuer, $audience);
        $memberAuthToken->setMember($member);

        //act
        $this->client->request(
            'PUT',
            $this->generateUrl('widget_member_api_main_edit'),
            array(),
            array(),
            array(
                'HTTP_AUTHORIZATION' => "Bearer {$memberAuthToken->getJWTToken()}",
            ),
            json_encode($data)
        );
        $response = $this->client->getResponse();
        $memberInfo = $member->getMemberInfo();
        $memberInfo->reload(true);

        $result = array_intersect_key($memberInfo->toArray(\BasePeer::TYPE_FIELDNAME), $data);

        //assert
        $this->assertTrue($response->isOk());
        $this->assertEquals($data, $result);
    }

    public function test_editAction_create()
    {
        //arrange
        $username = 'bubble_noinfo';
        $audience = 'http://localhost:8080';
        $issuer = 'http://localhost';
        $secret = $this->client->getContainer()->getParameter('secret');
        $data = array(
            'name' => 'Shanhu Chang',
            'email' => 'shanhu@dgfactor.com.tw',
            'phone' => '0911-111-111',
            'phone2' => '0911-111-111',
            'zip' => '220',
            'address' => '新北市板橋區中山路一段1號',
        );
        $member = Model\MemberQuery::create()->findOneByName($username);

        $memberAuthToken = new MemberAuthToken($secret, $issuer, $audience);
        $memberAuthToken->setMember($member);

        //act
        $this->client->request(
            'PUT',
            $this->generateUrl('widget_member_api_main_edit'),
            array(),
            array(),
            array(
                'HTTP_AUTHORIZATION' => "Bearer {$memberAuthToken->getJWTToken()}",
            ),
            json_encode($data)
        );
        $response = $this->client->getResponse();
        $memberInfo = Model\MemberInfoQuery::create()
            ->findPk($member->getId());

        //assert
        $this->assertNotNull($memberInfo);
        $result = array_intersect_key($memberInfo->toArray(\BasePeer::TYPE_FIELDNAME), $data);
        $this->assertTrue($response->isOk());
        $this->assertEquals($data, $result);
    }

    public function test_editAction_empty_name()
    {
        //arrange
        $username = 'bubble1';
        $audience = 'http://localhost:8080';
        $issuer = 'http://localhost';
        $secret = $this->client->getContainer()->getParameter('secret');
        $data = array(
            'name' => '',
            'email' => 'shanhu@dgfactor.com.tw',
            'phone' => '0911-111-111',
            'phone2' => '0911-111-111',
            'zip' => '220',
            'address' => '新北市板橋區中山路一段1號',
        );
        $member = Model\MemberQuery::create()->findOneByName($username);

        $memberAuthToken = new MemberAuthToken($secret, $issuer, $audience);
        $memberAuthToken->setMember($member);

        //act
        $this->client->request(
            'PUT',
            $this->generateUrl('widget_member_api_main_edit'),
            array(),
            array(),
            array(
                'HTTP_AUTHORIZATION' => "Bearer {$memberAuthToken->getJWTToken()}",
            ),
            json_encode($data)
        );
        $response = $this->client->getResponse();

        //assert
        $this->assertEquals(Response::HTTP_BAD_REQUEST, $response->getStatusCode());
    }

    public function test_editAction_email_bad_format()
    {
        //arrange
        $username = 'bubble1';
        $audience = 'http://localhost:8080';
        $issuer = 'http://localhost';
        $secret = $this->client->getContainer()->getParameter('secret');
        $data = array(
            'name' => 'Shanhu Chang',
            'email' => 'shanhu',
            'phone' => '0911-111-111',
            'phone2' => '0911-111-111',
            'zip' => '220',
            'address' => '新北市板橋區中山路一段1號',
        );
        $member = Model\MemberQuery::create()->findOneByName($username);

        $memberAuthToken = new MemberAuthToken($secret, $issuer, $audience);
        $memberAuthToken->setMember($member);

        //act
        $this->client->request(
            'PUT',
            $this->generateUrl('widget_member_api_main_edit'),
            array(),
            array(),
            array(
                'HTTP_AUTHORIZATION' => "Bearer {$memberAuthToken->getJWTToken()}",
            ),
            json_encode($data)
        );
        $response = $this->client->getResponse();

        //assert
        $this->assertEquals(Response::HTTP_BAD_REQUEST, $response->getStatusCode());
    }

    public function test_Oauth2LoginAction_member_not_exists()
    {
        //arrange
        $type = 'google';

        $data = array(
            'access_token' => '12345',
        );

        $responseArray = array(
            'name' => 'Shanhu Chang',
            'email' => 'shanhu@dgfactor.com.tw'
        );

        $service = $this->getMockBuilder(OAuthService::class)
            ->disableOriginalConstructor()
            ->setMethods(array('queryUserInfo'))
            ->getMock();
        $service
            ->expects($this->once())
            ->method('queryUserInfo')
            ->willReturnCallback(function($endpoint, $accessToken) use($type, $data, $responseArray){
                $this->assertEquals($type, $endpoint);
                $this->assertEquals($accessToken, $data['access_token']);
                return $responseArray;
            });

        $this->client->getContainer()->set('widget_member.oauth', $service);

        //act
        $this->client->request(
            'PUT',
            $this->generateUrl('widget_member_api_main_oauth2login', array('type' => $type)),
            array(),
            array(),
            array(
                'HTTP_ORIGIN' => 'http://localhost:8080',
            ),
            json_encode($data)
        );
        $response = $this->client->getResponse();
        $result = json_decode($response->getContent(), true);
        $member = Model\MemberQuery::create()
            ->findOneByEmail($responseArray['email']);

        //assert
        $this->assertTrue($response->isOk());
        $this->assertNotEmpty($result['token']);
        $this->assertNotNull($member);
        $member->delete();
    }

    public function test_Oauth2LoginAction_member_exists()
    {
        //arrange
        $type = 'google';

        $data = array(
            'access_token' => '12345',
        );

        $responseArray = array(
            'name' => 'Shanhu Chang',
            'email' => 'shanhu@dgfactor.com.tw'
        );

        $member = new Model\Member();
        $member
            ->setConfirmToken(null)
            ->setTokenExpiredAt(null)
            ->setEnabled(false)
            ->setName($responseArray['name'])
            ->setEmail($responseArray['email'])
            ->setPlainPassword('12345')
        ;
        $this->client->getContainer()->get('member_manager')->updateMember($member);

        $service = $this->getMockBuilder(OAuthService::class)
            ->disableOriginalConstructor()
            ->setMethods(array('queryUserInfo'))
            ->getMock();
        $service
            ->expects($this->once())
            ->method('queryUserInfo')
            ->willReturnCallback(function($endpoint, $accessToken) use($type, $data, $responseArray){
                $this->assertEquals($type, $endpoint);
                $this->assertEquals($accessToken, $data['access_token']);
                return $responseArray;
            });

        $this->client->getContainer()->set('widget_member.oauth', $service);

        //act
        $this->client->request(
            'PUT',
            $this->generateUrl('widget_member_api_main_oauth2login', array('type' => $type)),
            array(),
            array(),
            array(
                'HTTP_ORIGIN' => 'http://localhost:8080',
            ),
            json_encode($data)
        );
        $response = $this->client->getResponse();
        $result = json_decode($response->getContent(), true);
        $member = Model\MemberQuery::create()
            ->findOneByEmail($responseArray['email']);
        $member->delete();
        
        //assert
        $this->assertTrue($response->isOk());
        $this->assertNotEmpty($result['token']);
        $this->assertNotNull($member);
        $this->assertTrue($member->getEnabled());
    }

    public function test_renewTokenAction()
    {
        // arrange
        $username = 'bubble1';
        $audience = 'http://localhost:8080';
        $issuer = 'http://localhost';
        $secret = $this->client->getContainer()->getParameter('secret');

        $member = Model\MemberQuery::create()->findOneByName($username);

        $memberAuthToken = new MemberAuthToken($secret, $issuer, $audience);
        $memberAuthToken->setMember($member);

        // act
        $this->client->request(
            'PUT',
            $this->generateUrl('widget_member_api_main_renewtoken'),
            array(),
            array(),
            array(
                'HTTP_ORIGIN' => $audience,
                'HTTP_AUTHORIZATION' => "Bearer {$memberAuthToken->getJWTToken()}",
            )
        );
        $response = $this->client->getResponse();
        $result = json_decode($response->getContent(), true);

        // assert
        $this->assertTrue($response->isOk());
        $this->assertNotEmpty($result['token']);
    }

    public function test_OauthConfigAction()
    {
        // arrange
        $expectData = array(
            array(
                'type' => 'facebook',
                'client_id' => 'test_facebook_client_id',
            ),
            array(
                'type' => 'google',
                'client_id' => 'test_google_client_id',
            ),
        );
        $expectJson = json_encode($expectData);

        // act
        $this->client->request(
            'GET',
            $this->generateUrl('widget_member_api_main_oauthconfig')
        );
        $response = $this->client->getResponse();

        // assert
        $this->assertEquals($expectJson, $response->getContent());
    }

    public function test_otherMemberLoginAction_bad_param()
    {
        // arrange

        // act
        $this->client->request(
            'PUT',
            $this->generateUrl('widget_member_api_main_othermemberlogin')
        );

        $response = $this->client->getResponse();

        // assert
        $this->assertEquals(Response::HTTP_FORBIDDEN, $response->getStatusCode());
    }

    public function test_otherMemberLoginAction_recaptcha_verify_fail()
    {
        // arrange
        $issuer = 'http://localhost';
        $origin = 'http://localhost';
        $member = MemberQuery::create()->findOneByName('bubble0');
        $token = (string)$this->client->getContainer()->get('widget_member.token.signer')->sign($issuer, $origin, $member);
        $parameters = array(
            'email' => $member->getEmail(),
            'recaptcha' => 'fhsdjfhsdjklfhsd'
        );
        $recaptchaService = $this->getMockBuilder(ReCaptchaVerifyService::class)
            ->disableOriginalConstructor()
            ->setMethods(array('verify'))
            ->getMock();

        $recaptchaService
            ->expects($this->once())
            ->method('verify')
            ->with($parameters['recaptcha'])
            ->willReturn(false);

        $recaptchaEnableChecker = $this->getMockBuilder(RecaptchaEnableChecker::class)
            ->disableOriginalConstructor()
            ->setMethods(array('isEnable'))
            ->getMock();

        $recaptchaEnableChecker
            ->expects($this->once())
            ->method('isEnable')
            ->with($this->equalTo('otherMemberLogin'))
            ->willReturn(true);

        $this->client->getContainer()->set('backend_base.recaptcha_verify', $recaptchaService);
        $this->client->getContainer()->set('recaptcha.enable.checker', $recaptchaEnableChecker);
        // act
        $this->client->request(
            'PUT',
            $this->generateUrl('widget_member_api_main_othermemberlogin'),
            array(),
            array(),
            array(
                'HTTP_ORIGIN' => $origin,
                'HTTP_AUTHORIZATION' => "Bearer $token",
            ),
            json_encode($parameters)
        );

        $response = $this->client->getResponse();

        // assert
        $this->assertFalse($response->isOk());
        $this->assertEquals(Response::HTTP_BAD_REQUEST, $response->getStatusCode());
    }

    public function test_otherMemberLoginAction_recaptcha_check_disabled()
    {
        // arrange
        $issuer = 'http://localhost';
        $origin = 'http://localhost';
        $member = MemberQuery::create()->findOneByName('bubble0');
        $token = (string)$this->client->getContainer()->get('widget_member.token.signer')->sign($issuer, $origin, $member);
        $parameters = array(
            'email' => $member->getEmail(),
        );
        $recaptchaService = $this->getMockBuilder(ReCaptchaVerifyService::class)
            ->disableOriginalConstructor()
            ->setMethods(array('verify'))
            ->getMock();

        $recaptchaService
            ->expects($this->never())
            ->method('verify')
            ->willReturn(true);

        $recaptchaEnableChecker = $this->getMockBuilder(RecaptchaEnableChecker::class)
            ->disableOriginalConstructor()
            ->setMethods(array('isEnable'))
            ->getMock();

        $recaptchaEnableChecker
            ->expects($this->once())
            ->method('isEnable')
            ->with($this->equalTo('otherMemberLogin'))
            ->willReturn(false);

        $this->client->getContainer()->set('backend_base.recaptcha_verify', $recaptchaService);
        $this->client->getContainer()->set('recaptcha.enable.checker', $recaptchaEnableChecker);

        $sendMailService = $this->getMockBuilder(SendMailService::class)
            ->disableOriginalConstructor()
            ->setMethods(array('send'))
            ->getMock();

        $sendMailService
            ->expects($this->once())
            ->method('send')
            ->willReturn(true);

        $this->client->getContainer()->set('backend_base.send_mail', $sendMailService);
        // act
        $this->client->request(
            'PUT',
            $this->generateUrl('widget_member_api_main_othermemberlogin'),
            array(),
            array(),
            array(
                'HTTP_ORIGIN' => $origin,
                'HTTP_AUTHORIZATION' => "Bearer $token",
            ),
            json_encode($parameters)
        );

        $response = $this->client->getResponse();
        $member->reload(true);

        // assert
        $this->assertTrue($response->isOk());
        $this->assertNotNull($member->getConfirmToken());
        // reset
        $member->setConfirmToken(null)->setTokenExpiredAt(null)->save();
    }

    public function test_otherMemberLoginAction()
    {
        // arrange
        $issuer = 'http://localhost';
        $origin = 'http://localhost';
        $member = MemberQuery::create()->findOneByName('bubble0');
        $token = (string)$this->client->getContainer()->get('widget_member.token.signer')->sign($issuer, $origin, $member);
        $parameters = array(
            'email' => $member->getEmail(),
            'recaptcha' => 'fhsdjfhsdjklfhsd'
        );
        $recaptchaService = $this->getMockBuilder(ReCaptchaVerifyService::class)
            ->disableOriginalConstructor()
            ->setMethods(array('verify'))
            ->getMock();

        $recaptchaService
            ->expects($this->once())
            ->method('verify')
            ->with($parameters['recaptcha'])
            ->willReturn(true);

        $recaptchaEnableChecker = $this->getMockBuilder(RecaptchaEnableChecker::class)
            ->disableOriginalConstructor()
            ->setMethods(array('isEnable'))
            ->getMock();

        $recaptchaEnableChecker
            ->expects($this->once())
            ->method('isEnable')
            ->with($this->equalTo('otherMemberLogin'))
            ->willReturn(true);

        $this->client->getContainer()->set('backend_base.recaptcha_verify', $recaptchaService);
        $this->client->getContainer()->set('recaptcha.enable.checker', $recaptchaEnableChecker);

        $sendMailService = $this->getMockBuilder(SendMailService::class)
            ->disableOriginalConstructor()
            ->setMethods(array('send'))
            ->getMock();

        $sendMailService
            ->expects($this->once())
            ->method('send')
            ->willReturn(true);

        $this->client->getContainer()->set('backend_base.send_mail', $sendMailService);
        // act
        $this->client->request(
            'PUT',
            $this->generateUrl('widget_member_api_main_othermemberlogin'),
            array(),
            array(),
            array(
                'HTTP_ORIGIN' => $origin,
                'HTTP_AUTHORIZATION' => "Bearer $token",
            ),
            json_encode($parameters)
        );

        $response = $this->client->getResponse();
        $member->reload(true);

        // assert
        $this->assertTrue($response->isOk());
        $this->assertNotNull($member->getConfirmToken());
        // reset
        $member->setConfirmToken(null)->setTokenExpiredAt(null)->save();

    }

    public function test_otherMemberLoginVerifyAction_bad_token_confirmToken_empty()
    {
        // arrange
        $issuer = 'http://localhost';
        $origin = 'http://localhost';
        $member = MemberQuery::create()->findOneByName('bubble0');
        $token = (string)$this->client->getContainer()->get('widget_member.token.signer')->sign($issuer, $origin, $member);

        $parameters = array(
            'token' => $member->getConfirmToken()
        );

        // act
        $this->client->request(
            'PUT',
            $this->generateUrl('widget_member_api_main_othermemberloginverify', array('id' => $member->getId())),
            array(),
            array(),
            array(
                'HTTP_ORIGIN' => $origin,
                'HTTP_AUTHORIZATION' => "Bearer $token",
            ),
            json_encode($parameters)
        );

        $response = $this->client->getResponse();

        // assert
        $this->assertEquals(Response::HTTP_UNAUTHORIZED, $response->getStatusCode());

        // reset
        $member->setConfirmToken(null)->setTokenExpiredAt(null)->save();
    }

    public function test_otherMemberLoginVerifyAction_bad_token_confirmToken_not_equal()
    {
        // arrange
        $issuer = 'http://localhost';
        $origin = 'http://localhost';
        $member = MemberQuery::create()->findOneByName('bubble0');
        $member->regenerateConfirmToken(30*60)->save();
        $token = (string)$this->client->getContainer()->get('widget_member.token.signer')->sign($issuer, $origin, $member);

        $parameters = array(
            'token' => "not valid token"
        );

        // act
        $this->client->request(
            'PUT',
            $this->generateUrl('widget_member_api_main_othermemberloginverify', array('id' => $member->getId())),
            array(),
            array(),
            array(
                'HTTP_ORIGIN' => $origin,
                'HTTP_AUTHORIZATION' => "Bearer $token",
            ),
            json_encode($parameters)
        );

        $response = $this->client->getResponse();

        // assert
        $this->assertEquals(Response::HTTP_UNAUTHORIZED, $response->getStatusCode());

        // reset
        $member->setConfirmToken(null)->setTokenExpiredAt(null)->save();
    }

    public function test_otherMemberLoginVerifyAction_bad_token_confirmToken_expired()
    {
        // arrange
        $issuer = 'http://localhost';
        $origin = 'http://localhost';
        $member = MemberQuery::create()->findOneByName('bubble0');
        $member->regenerateConfirmToken(-3600)->save();
        $token = (string)$this->client->getContainer()->get('widget_member.token.signer')->sign($issuer, $origin, $member);

        $parameters = array(
            'token' => $member->getConfirmToken()
        );

        // act
        $this->client->request(
            'PUT',
            $this->generateUrl('widget_member_api_main_othermemberloginverify', array('id' => $member->getId())),
            array(),
            array(),
            array(
                'HTTP_ORIGIN' => $origin,
                'HTTP_AUTHORIZATION' => "Bearer $token",
            ),
            json_encode($parameters)
        );

        $response = $this->client->getResponse();

        // assert
        $this->assertEquals(Response::HTTP_UNAUTHORIZED, $response->getStatusCode());

        // reset
        $member->setConfirmToken(null)->setTokenExpiredAt(null)->save();
    }

    public function test_otherMemberLoginVerifyAction_ok()
    {
        // arrange
        $issuer = 'http://localhost';
        $origin = 'http://localhost';
        $member = MemberQuery::create()->findOneByName('bubble0');
        $token = (string)$this->client->getContainer()->get('widget_member.token.signer')->sign($issuer, $origin, $member);
        $member->regenerateConfirmToken(30*60)->save();
        $parameters = array(
            'token' => $member->getConfirmToken()
        );
        
        // act 
        $this->client->request(
            'PUT',
            $this->generateUrl('widget_member_api_main_othermemberloginverify', array('id' => $member->getId())),
            array(),
            array(),
            array(
                'HTTP_ORIGIN' => $origin,
                'HTTP_AUTHORIZATION' => "Bearer $token",
            ),
            json_encode($parameters)
        );

        $response = $this->client->getResponse();
        $responseArray = json_decode($response->getContent(), true);

        // assert
        $this->assertTrue($response->isOk());
        $this->assertArrayHasKey('token', $responseArray);

        // reset
        $member->setConfirmToken(null)->setTokenExpiredAt(null)->save();
    }
}