<?php
namespace Widget\MemberBundle\Security;


class MemberSoldiumPassword
{
    const MD5 = 3;
    const BLOWFISH = 4;
    const SHA256 = 5;
    const SHA512 = 6;
    const SODIUM_ARGON2 = 7;
    const SODIUM_SCRYPT = 8;

    protected $type;

    public function __construct($type = self::BLOWFISH)
    {
        $this->type = $type;
    }

    /**
     * isSodiumAlgo
     *
     * @param int $type
     *
     * @return  bool
     */
    protected function isSodiumAlgo($type)
    {
        return $type === static::SODIUM_ARGON2 || $type === static::SODIUM_SCRYPT;
    }

    /**
     * isSodiumHash
     *
     * @param string $hash
     *
     * @return  bool
     */
    protected function isSodiumHash($hash)
    {
        return (strpos($hash, '$argon2i') === 0 || strpos($hash, '$7$C6') === 0);
    }

    public function create($password, $salt = null)
    {
        if(!$this->isSodiumAlgo($this->type)){
            return $this->generateCryptPassword($this->type, $password, $salt);
        }

        return $this->generateSodiumPassword($this->type, $password);
    }

    public function verify($password, $hash)
    {
        if($this->isMD5NoSalt($hash)){
            return md5($password) === $hash;
        }

        if ($this->isSodiumHash($hash))
        {
            return $this->verifySodium($password, $hash);
        }

        return password_verify($password, $hash);
    }

    /**
     * @param $hash
     * @return boolean
     */
    public function isMD5NoSalt($hash)
    {
        return (boolean) preg_match('/^[0-9a-f]{32}$/i', $hash, $match);
    }

    protected function generateCryptPassword($type, $password, $salt)
    {
        if($salt === null){
            $salt = $this->generateSalt();
        }

        switch ($type)
        {
            case static::MD5:
                return crypt($password, '$1$' . $salt . '$');
            case static::SHA256:
                return crypt($password, '$5$rounds=1000$' . $salt . '$');
            case static::SHA512:
                return crypt($password, '$6$rounds=1000$' . $salt . '$');
            case static::BLOWFISH:
            default:
                return crypt($password, '$2a$07$' . base64_encode($salt) . '$');
        }
    }

    protected function generateSalt($length = 16)
    {
        return sha1(random_bytes($length));
    }

    protected function generateSodiumPassword($type, $password)
    {
        switch ($type)
        {
            case static::SODIUM_SCRYPT:
                return sodium_crypto_pwhash_scryptsalsa208sha256_str(
                    $password,
                    SODIUM_CRYPTO_PWHASH_SCRYPTSALSA208SHA256_OPSLIMIT_INTERACTIVE,
                    SODIUM_CRYPTO_PWHASH_SCRYPTSALSA208SHA256_MEMLIMIT_INTERACTIVE
                );
            case static::SODIUM_ARGON2:
            default:
                return sodium_crypto_pwhash_str(
                    $password,
                    SODIUM_CRYPTO_PWHASH_OPSLIMIT_INTERACTIVE,
                    SODIUM_CRYPTO_PWHASH_MEMLIMIT_INTERACTIVE
                );
        }
    }

    protected function verifySodium($password, $hash)
    {
        if (strpos($hash, '$argon2i') === 0)
        {
            $result = sodium_crypto_pwhash_str_verify($hash, $password);
            sodium_memzero($password);
            return $result;
        }
        elseif (strpos($hash, '$7$C6') === 0)
        {
            $result = sodium_crypto_pwhash_scryptsalsa208sha256_str_verify($hash, $password);
            sodium_memzero($password);
            return $result;
        }

        return false;
    }

    public function getType()
    {
        return $this->type;
    }
}
