<?php

namespace Widget\MemberBundle\Controller\BackendAPI;

use Backend\BaseBundle\Controller\BackendAPI\BaseBackendAPIController;
use Backend\BaseBundle\Form\Type\APIFormTypeItem;
use Sensio\Bundle\FrameworkExtraBundle\Configuration\Method;
use Sensio\Bundle\FrameworkExtraBundle\Configuration\Route;
use Sensio\Bundle\FrameworkExtraBundle\Configuration\Security;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\Validator\Constraints\Callback;
use Symfony\Component\Validator\Constraints\NotBlank;
use Symfony\Component\Validator\Constraints\Regex;
use Symfony\Component\Validator\Context\ExecutionContextInterface;
use Widget\MemberBundle\Model\GroupQuery;
use Widget\MemberBundle\Model\Member;
use Widget\MemberBundle\Model\MemberPeer;
use Widget\MemberBundle\Model\MemberQuery;

/**
 * @Route("/member")
 * @Security("has_role_or_superadmin('ROLE_MEMBER')")
 */
class MemberController extends BaseBackendAPIController
{

    /**
     * 欄位表單設定
     * 主要是USER送過來的欄位
     * @return APIFormTypeItem[]
     */
    protected function getFormConfig()
    {
        return array(
            (new APIFormTypeItem('name'))->setOptions(array(
                'constraints' => array(
                    new NotBlank(array(
                        'message' => 'error.email.missing_field',
                    )),
                )
            )),
            (new APIFormTypeItem('email'))->setOptions(array(
                'constraints' => array(
                    new NotBlank(array(
                        'message' => 'error.email.missing_field',
                    )),
                    new Regex(array(
                        'pattern' => '/^\w+((-\w+)|(\.\w+))*\@[A-Za-z0-9]+((\.|-)[A-Za-z0-9]+)*\.[A-Za-z]+$/',
                        'message' => 'error.email.format_incorrect_field'
                    )),
                    new Callback(function($value, ExecutionContextInterface $context){
                        $object = $context->getRoot()->getData();
                        $member = MemberQuery::create()
                            ->filterByEmail($value)
                            ->findOne();
                        if($member && $member->getId() != $object->getId()){
                            $context->addViolation('error.email.duplicate');
                        }
                    }),
                ),
            )),
            new APIFormTypeItem('plain_password'),
            new APIFormTypeItem('enabled'),
            new APIFormTypeItem('locked'),
            new APIFormTypeItem('confirm_token'),
            new APIFormTypeItem('token_expired_at'),
            new APIFormTypeItem('roles'),
            new APIFormTypeItem('last_login'),
        );
    }

    /**
     * 新增
     * @Route("s")
     * @Method({"POST"})
     * @Security("has_role_or_superadmin('ROLE_MEMBER_WRITE')")
     */
    public function createAction(Request $request)
    {
        $requestData = json_decode($request->getContent(), true); //把資料用JSON解碼

        $member = new Member();
        $form = $this->bindObject($member, $request->getContent());

        if(!($form->isValid())) {
            return $this->createFormErrorJsonResponse($form->getErrors(true));
        }
        $this->get('member_manager')->updateMember($member);

        return $this->createJsonSerializeResponse($member, array(
            'detail'
        ));
    }

    /**
     * 搜尋
     * @Route("s")
     * @Method({"GET"})
     * @Security("has_role_or_superadmin('ROLE_MEMBER_READ')")
     */
    public function searchAction(Request $request)
    {
        $query = MemberQuery::create()->leftJoinWithMemberInfo();
        return $this->doSearch($request->query->all(), $query->distinct(), MemberPeer::class);
    }

    /**
     * 讀取
     * @Route("/{id}")
     * @Method({"GET"})
     * @Security("has_role_or_superadmin('ROLE_MEMBER_READ')")
     */
    public function readAction(Member $member)
    {
        return $this->createJsonSerializeResponse($member, array('detail'));
    }

    /**
     * 讀取
     * @Route("s/all")
     * @Method({"GET"})
     * @Security("has_role_or_superadmin('ROLE_SUPERADMIN')")
     */
    public function listAction(Request $request)
    {
        $members = MemberQuery::create()->leftJoinWithMemberInfo()->find();
        if (!$members){
            return new Response('', Response::HTTP_BAD_REQUEST);
        }
        return $this->createJsonSerializeResponse($members, array('list'));
    }

    /**
     * 更新
     * @Route("/{id}")
     * @Method({"PUT"})
     * @Security("has_role_or_superadmin('ROLE_MEMBER_WRITE')")
     */
    public function updateAction(Request $request, Member $member)
    {
        $form = $this->bindObject($member, $request->getContent());

        if(!($form->isValid())) {
            return $this->createFormErrorJsonResponse($form->getErrors(true));
        }

        $this->get('member_manager')->updateMember($member);
        $member->save();
        return $this->createJsonSerializeResponse($member, array(
            'detail'
        ));
    }

    /**
     * 刪除
     * @Route("/{id}")
     * @Method({"DELETE"})
     * @Security("has_role_or_superadmin('ROLE_MEMBER_WRITE')")
     */
    public function deleteAction(Member $member)
    {
        $this->deleteObject($member);
        return $this->createJsonResponse();
    }

    /**
     * 批次管理
     * @Route("s")
     * @Method({"PUT"})
     * @Security("has_role_or_superadmin('ROLE_MEMBER_WRITE')")
     */
    public function batchAction(Request $request)
    {
        return parent::batchAction($request);
    }

    /**
     * 批次切換狀態處理
     * @param $ids
     * @param $con
     * @param $column
     */
    protected function batchSwitch($ids, $column, \PropelPDO $con)
    {
        $members = MemberQuery::create()->findPks($ids, $con);
        foreach ($members as $member) {
            $value = $member->getByName($column, \BasePeer::TYPE_FIELDNAME);
            $member->setByName($column, !$value, \BasePeer::TYPE_FIELDNAME);
            $member->save($con);
        }
    }

    /**
     * 批次設定值
     * @param $ids
     * @param $column
     * @param $value
     * @param \PropelPDO $con
     * @return \Symfony\Component\HttpFoundation\JsonResponse|void
     */
    protected function batchValue($ids, $column, $value, \PropelPDO $con)
    {
        $members = MemberQuery::create()->findPks($ids, $con);
        foreach ($members as $member) {
            $member->setByName($column, $value, \BasePeer::TYPE_FIELDNAME);
            $member->save($con);
        }
    }

    /**
     * 批次切換狀態處理
     * @param $ids
     * @param $con
     * @param $column
     */
    protected function batchDelete($ids, \PropelPDO $con)
    {
        MemberQuery::create()->findPks($ids, $con)->delete($con);
    }

    /**
     * 設定會員 群組 關聯
     * @Route("/{id}/membergroup")
     * @Method({"POST"})
     */
    public function relateToGroup(Request $request, Member $member)
    {
        $requestParams = json_decode($request->getContent(), true);
        if (!isset($requestParams['groups']) || count($requestParams['groups']) == 0) {
            $emptyGroup = GroupQuery::create()->findPks(array());
            $member->setGroups($emptyGroup)->save();
            return $this->createJsonSerializeResponse($member, array('list'));
        }

        $groupIds = array();
        if (!is_array($requestParams['groups'])) {
            $groupIds[] = $requestParams['groups'];
        } else {
            foreach ($requestParams['groups'] as $group) {
                $groupIds[] = $group['id'];
            }
        }

        $groups = GroupQuery::create()->findPks($groupIds);
        $member->setGroups($groups)->save();
        return $this->createJsonSerializeResponse($member, array('list'));
    }

    protected function setMemberPassword($requestData, Member $member)
    {
        $member->setPlainPassword($requestData['password']);
        $this->get('member_manager')->updateMember($member);
    }
}