<?php
namespace Widget\MemberBundle\Tests\Security;

use Backend\BaseBundle\Tests\Fixture\BaseTestCase;
use Symfony\Component\Security\Core\Encoder\EncoderFactory;
use Symfony\Component\Security\Core\Encoder\PlaintextPasswordEncoder;
use Symfony\Component\Security\Core\Encoder\UserPasswordEncoder;
use Widget\MemberBundle\Model;
use Widget\MemberBundle\Security\MemberManager;
use Widget\MemberBundle\Service\DefaultGroupService;

/**
 * @group unit
 */
class MemberMangerTest extends BaseTestCase
{

    public function test_injectService()
    {
        //arrange
        $encoderFactory = $this->getMockBuilder(EncoderFactory::class)
            ->disableOriginalConstructor()
            ->setMethods()
            ->getMock();
        $manager = new MemberManager();

        //act
        $manager->injectService($encoderFactory);

        //assert
        $this->assertEquals($encoderFactory, $this->getObjectAttribute($manager, 'encoderFactory'));
    }

    public function test_getEncoder()
    {
        //arrange
        $encoder = $this->getMockBuilder(UserPasswordEncoder::class)
            ->disableOriginalConstructor()
            ->setMethods()
            ->getMock();
        $encoderFactory = $this->getMockBuilder(EncoderFactory::class)
            ->setMethods(array('getEncoder'))
            ->disableOriginalConstructor()
            ->getMock();
        $encoderFactory
            ->expects($this->once())
            ->method('getEncoder')
            ->willReturn($encoder);

        $manager = new MemberManager();
        $this->setObjectAttribute($manager, 'encoderFactory', $encoderFactory);

        //act
        $result = $this->callObjectMethod($manager, 'getEncoder', new Model\Member());

        //assert
        $this->assertEquals($encoder, $result);
    }

    public function test_updateMember_plain_password_is_null()
    {
        //arrange
        $member = $this->getMockBuilder(Model\Member::class)
            ->setMethods(array('regenerateSalt', 'setPassword', 'save'))
            ->disableOriginalConstructor()
            ->getMock();
        $member->expects($this->never())->method('regenerateSalt');
        $member->expects($this->never())->method('setPassword');
        $member->expects($this->once())->method('save');

        $manager = $this->getMockBuilder(MemberManager::class)
            ->setMethods(array('getEncoder'))
            ->disableOriginalConstructor()
            ->getMock();

        $manager = $this->getMockBuilder(MemberManager::class)->setMethods(array('getEncoder'))->getMock();

        $manager->expects($this->never())->method('getEncoder');

        //act
        $manager->updateMember($member);
    }

    public function test_updateMember_plain_password_is_not_null()
    {
        //arrange
        $encoder = $this->getMockBuilder(PlaintextPasswordEncoder::class)
            ->setMethods(array('encodePassword'))
            ->disableOriginalConstructor()
            ->getMock();
        $encoder
            ->expects($this->once())
            ->method('encodePassword')
            ->willReturnCallback(function($password, $salt){
                $this->assertEquals('some_password', $password);
                $this->assertEquals('some_salt', $salt);
                return 'encoded_password';
            });
        $member = $this->getMockBuilder(Model\Member::class)
            ->setMethods(array('regenerateSalt', 'setPassword', 'save'))
            ->disableOriginalConstructor()
            ->getMock();
        $this->setObjectAttribute($member, 'plainPassword', 'some_password');
        $this->setObjectAttribute($member, 'salt', 'some_salt');

        $member->expects($this->once())->method('regenerateSalt');
        $member->expects($this->once())->method('setPassword')->willReturnCallback(function($encodedPassword){
            $this->assertEquals('encoded_password', $encodedPassword);
        });
        $member->expects($this->once())->method('save');


        $manager = $this->getMockBuilder(MemberManager::class)
            ->setMethods(array('getEncoder'))
            ->disableOriginalConstructor()
            ->getMock();

        $manager = $this->getMockBuilder(MemberManager::class)->setMethods(array('getEncoder'))->getMock();

        $manager->expects($this->once())->method('getEncoder')->willReturn($encoder);

        //act
        $manager->updateMember($member);
    }

    public function test_verifyMember_member_is_not_enabled()
    {
        //arrange
        $member = new Model\Member();
        $this->setObjectAttribute($member, 'enabled', false);
        $this->setObjectAttribute($member, 'locked', false);
        $manager = new MemberManager();

        //act
        $result = $manager->verifyMember($member, 'test_password');

        //assert
        $this->assertFalse($result);
    }

    public function test_verifyMember_member_is_locked()
    {
        //arrange
        $member = new Model\Member();
        $this->setObjectAttribute($member, 'enabled', true);
        $this->setObjectAttribute($member, 'locked', true);
        $manager = new MemberManager();

        //act
        $result = $manager->verifyMember($member, 'test_password');

        //assert
        $this->assertFalse($result);
    }

    public function test_verifyMember_member_is_valid()
    {
        //arrange
        $encoder = $this->getMockBuilder(PlaintextPasswordEncoder::class)
            ->setMethods(array('isPasswordValid'))
            ->disableOriginalConstructor()
            ->getMock();
        $encoder
            ->expects($this->once())
            ->method('isPasswordValid')
            ->willReturnCallback(function($password, $inputPassword, $salt){
                $this->assertEquals('test_password', $password);
                $this->assertEquals('input_password', $inputPassword);
                $this->assertEquals('test_salt', $salt);
                return true;
            });
        $member = new Model\Member();
        $this->setObjectAttribute($member, 'password', 'test_password');
        $this->setObjectAttribute($member, 'salt', 'test_salt');
        $this->setObjectAttribute($member, 'enabled', true);
        $this->setObjectAttribute($member, 'locked', false);

        $manager = $this->getMockBuilder(MemberManager::class)
            ->setMethods(array('getEncoder'))
            ->disableOriginalConstructor()
            ->getMock();

        $manager = $this->getMockBuilder(MemberManager::class)->setMethods(array('getEncoder'))->getMock();

        $manager
            ->expects($this->once())
            ->method('getEncoder')
            ->willReturn($encoder)
            ;
        //act
        $result = $manager->verifyMember($member, 'input_password');

        //assert
        $this->assertTrue($result);
    }

    public function test_verifyMember_member_is_not_valid()
    {
        //arrange
        $encoder = $this->getMockBuilder(PlaintextPasswordEncoder::class)
            ->setMethods(array('isPasswordValid'))
            ->disableOriginalConstructor()
            ->getMock();
        $encoder
            ->expects($this->once())
            ->method('isPasswordValid')
            ->willReturnCallback(function($password, $inputPassword, $salt){
                $this->assertEquals('test_password', $password);
                $this->assertEquals('input_password', $inputPassword);
                $this->assertEquals('test_salt', $salt);
                return false;
            });
        $member = new Model\Member();
        $this->setObjectAttribute($member, 'password', 'test_password');
        $this->setObjectAttribute($member, 'salt', 'test_salt');
        $this->setObjectAttribute($member, 'enabled', true);
        $this->setObjectAttribute($member, 'locked', false);

        $manager = $this->getMockBuilder(MemberManager::class)
            ->setMethods(array('getEncoder'))
            ->disableOriginalConstructor()
            ->getMock();

        $manager = $this->getMockBuilder(MemberManager::class)->setMethods(array('getEncoder'))->getMock();

        $manager
            ->expects($this->once())
            ->method('getEncoder')
            ->willReturn($encoder)
        ;
        //act
        $result = $manager->verifyMember($member, 'input_password');

        //assert
        $this->assertFalse($result);
    }

    public function test_createMember()
    {
        //arrange
        $member = new Model\Member();
        $con = $this->getMockBuilder(\PropelPDO::class)
            ->disableOriginalConstructor()
            ->getMock();

        $memberManager = $this->getMockBuilder(MemberManager::class)
            ->setMethods(array('updateMember'))
            ->getMock();

        $memberManager
            ->expects($this->once())
            ->method('updateMember')
            ->with($member, $con)
            ;

        $defaultGroupService = $this->getMockBuilder(DefaultGroupService::class)
            ->setMethods(array('setMemberDefaultGroup'))
            ->getMock();
        $defaultGroupService
            ->expects($this->once())
            ->method('setMemberDefaultGroup')
            ->with($member, $con)
            ;

        $this->setObjectAttribute($memberManager, 'defaultGroupService', $defaultGroupService);

        //act
        $memberManager->createMember($member, $con);

        //assert
    }

    public function test_injectDefaultGroupService()
    {
        //arrange
        $defaultGroupService = $this->getMockBuilder(DefaultGroupService::class)
            ->getMock();

        $memberManager = new MemberManager();

        //act
        $memberManager->injectDefaultGroupService($defaultGroupService);

        //assert
        $this->assertEquals($defaultGroupService, $this->getObjectAttribute($memberManager, 'defaultGroupService'));
    }
}