<?php
namespace Widget\MemberBundle\Tests\EventListener;

use Backend\BaseBundle\Tests\Fixture\BaseWebTestCase;
use Lcobucci\JWT\Builder;
use Lcobucci\JWT\Signer\Hmac\Sha256;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpKernel\Event\GetResponseEvent;
use Symfony\Component\HttpKernel\HttpKernel;
use Widget\MemberBundle\EventListener\AuthTokenListener;
use Widget\MemberBundle\Token\MemberAuthToken;

/**
 * @group units
 */
class AuthTokenListenerTest extends BaseWebTestCase
{
    protected $secret = 'abcdefg';
    protected $token;

    public function setUp()
    {
        parent::setUp();

        $signer = new Sha256();
        $builder = new Builder();
        $this->token = $builder
            ->setIssuedAt(time())
            ->setNotBefore(time())
            ->setExpiration(time()+3600)
            ->setAudience('http://example.com:8000')
            ->setIssuer('http://foo.bar:8080')
            ->sign($signer, MemberAuthToken::SECRET_PREFIX.$this->secret)
            ->getToken();
    }

    public function test_injectSecret()
    {
        //arrange
        $listener = new AuthTokenListener();

        //act
        $listener->injectSecret($this->secret);

        //assert
        $this->assertEquals($this->secret, $this->getObjectAttribute($listener, 'secret'));
    }

    public function test_parseToken_bad_jwt()
    {
        //arrange
        $jwt = 'bad_jwt';
        $request = new Request(
            array(),
            array(),
            array(),
            array(),
            array(),
            array(
                'SERVER_NAME' => 'foo.bar',
                'SERVER_PORT' => 8080,
                'HTTP_ORIGIN' => 'http://example.com:8000',
            )
        );
        $listener = new AuthTokenListener();
        $this->setObjectAttribute($listener, 'secret', $this->secret);

        //act
        $result = $this->callObjectMethod($listener, 'parseToken', $request, $jwt);

        //assert
        $this->assertInstanceOf(MemberAuthToken::class, $result);
        $this->assertNull($result->getJWTToken());
    }

    public function test_parseToken_jwt()
    {
        //arrange
        $jwt = (string) $this->token;
        $request = new Request(
            array(),
            array(),
            array(),
            array(),
            array(),
            array(
                'SERVER_NAME' => 'foo.bar',
                'SERVER_PORT' => 8080,
                'HTTP_ORIGIN' => 'http://example.com:8000',
            )
        );
        $listener = new AuthTokenListener();
        $this->setObjectAttribute($listener, 'secret', $this->secret);

        //act
        $result = $this->callObjectMethod($listener, 'parseToken', $request, $jwt);
        //assert
        $this->assertInstanceOf(MemberAuthToken::class, $result);
        $this->assertNotNull($result->getJWTToken());
        $this->assertEquals($jwt, (string)$result->getJWTToken());
    }

    public function test_onRequest_no_auth()
    {
        //arrange
        $jwt = (string) $this->token;
        $audience = 'http://example.com:8000';
        $issuer = 'http://foo.bar:8080';
        $request = new Request();
        $event = new GetResponseEvent(static::$kernel, $request, HttpKernel::MASTER_REQUEST);
        $memberAuthToken = $this->getMockBuilder(MemberAuthToken::class)
            ->setMethods(array('getJWTToken'))
            ->setConstructorArgs(array($this->secret, $issuer, $audience))
            ->getMock();
        $memberAuthToken
            ->expects($this->never())
            ->method('getJWTToken')
            ->willReturn($this->token);
        $listener = $this->getMockBuilder(AuthTokenListener::class)
            ->setMethods(array('parseToken'))
            ->getMock();
        $listener
            ->expects($this->never())
            ->method('parseToken')
            ->willReturnCallback(function($jwtForTest) use($jwt, $memberAuthToken){
                $this->assertEquals($jwt, $jwtForTest);
                return $memberAuthToken;
            });

        //act
        $listener->onRequest($event);
        $token = $request->attributes->get('_authorizedToken');

        //assert
        $this->assertNull($token);
    }

    public function test_onRequest_bad_Bearer()
    {
        //arrange
        $jwt = (string) $this->token;
        $audience = 'http://example.com:8000';
        $issuer = 'http://foo.bar:8080';
        $request = new Request();
        $request->headers->set('Authorization', "$jwt");
        $event = new GetResponseEvent(static::$kernel, $request, HttpKernel::MASTER_REQUEST);
        $memberAuthToken = $this->getMockBuilder(MemberAuthToken::class)
            ->setMethods(array('getJWTToken'))
            ->setConstructorArgs(array($this->secret, $issuer, $audience))
            ->getMock();
        $memberAuthToken
            ->expects($this->never())
            ->method('getJWTToken')
            ->willReturn($this->token);
        $listener = $this->getMockBuilder(AuthTokenListener::class)
            ->setMethods(array('parseToken'))
            ->getMock();
        $listener
            ->expects($this->never())
            ->method('parseToken')
            ->willReturnCallback(function($jwtForTest) use($jwt, $memberAuthToken){
                $this->assertEquals($jwt, $jwtForTest);
                return $memberAuthToken;
            });

        //act
        $listener->onRequest($event);
        $token = $request->attributes->get('_authorizedToken');

        //assert
        $this->assertNull($token);
    }

    public function test_onRequest_parseToken_fail()
    {
        //arrange
        $audience = 'http://example.com:8000';
        $issuer = 'http://foo.bar:8080';
        $jwt = (string) $this->token;
        $request = new Request();
        $request->headers->set('Authorization', "Bearer $jwt");
        $event = new GetResponseEvent(static::$kernel, $request, HttpKernel::MASTER_REQUEST);
        $memberAuthToken = $this->getMockBuilder(MemberAuthToken::class)
            ->setMethods(array('getJWTToken'))
            ->setConstructorArgs(array($this->secret, $issuer, $audience))
            ->getMock();
        $memberAuthToken
            ->expects($this->never())
            ->method('getJWTToken')
            ->willReturn($this->token);
        $listener = $this->getMockBuilder(AuthTokenListener::class)
            ->setMethods(array('parseToken'))
            ->getMock();
        $listener
            ->expects($this->once())
            ->method('parseToken')
            ->willReturnCallback(function(Request $request, $jwtForTest) use($jwt){
                $this->assertEquals($jwt, $jwtForTest);
                return null;
            });

        //act
        $listener->onRequest($event);
        $token = $request->attributes->get('_authorizedToken');

        //assert
        $this->assertNull($token);
    }

    public function test_onRequest_getJWTToken_fail()
    {
        //arrange
        $audience = 'http://example.com:8000';
        $issuer = 'http://foo.bar:8080';
        $jwt = (string) $this->token;
        $request = new Request();
        $request->headers->set('Authorization', "Bearer $jwt");
        $event = new GetResponseEvent(static::$kernel, $request, HttpKernel::MASTER_REQUEST);
        $memberAuthToken = $this->getMockBuilder(MemberAuthToken::class)
            ->setMethods(array('getJWTToken'))
            ->setConstructorArgs(array($this->secret, $issuer, $audience))
            ->getMock();
        $memberAuthToken
            ->expects($this->once())
            ->method('getJWTToken')
            ->willReturn(null);
        $listener = $this->getMockBuilder(AuthTokenListener::class)
            ->setMethods(array('parseToken'))
            ->getMock();
        $listener
            ->expects($this->once())
            ->method('parseToken')
            ->willReturnCallback(function(Request $request, $jwtForTest) use($jwt, $memberAuthToken){
                $this->assertEquals($jwt, $jwtForTest);
                return $memberAuthToken;
            });

        //act
        $listener->onRequest($event);
        $token = $request->attributes->get('_authorizedToken');

        //assert
        $this->assertNull($token);
    }

    public function test_onRequest()
    {
        //arrange
        $audience = 'http://example.com:8000';
        $issuer = 'http://foo.bar:8080';
        $jwt = (string) $this->token;
        $request = new Request();
        $request->headers->set('Authorization', "Bearer $jwt");
        $event = new GetResponseEvent(static::$kernel, $request, HttpKernel::MASTER_REQUEST);
        $memberAuthToken = $this->getMockBuilder(MemberAuthToken::class)
            ->setMethods(array('getJWTToken'))
            ->setConstructorArgs(array($this->secret, $issuer, $audience))
            ->getMock();

        $memberAuthToken
            ->expects($this->once())
            ->method('getJWTToken')
            ->willReturn($this->token);
        $listener = $this->getMockBuilder(AuthTokenListener::class)
            ->setMethods(array('parseToken'))
            ->getMock();
        $listener
            ->expects($this->once())
            ->method('parseToken')
            ->willReturnCallback(function(Request $request, $jwtForTest) use($jwt, $memberAuthToken){
                $this->assertEquals($jwt, $jwtForTest);
                return $memberAuthToken;
            });

        //act
        $listener->onRequest($event);
        $token = $request->attributes->get('_authorizedToken');

        //assert
        $this->assertInstanceOf(MemberAuthToken::class, $token);
    }

}