<?php

namespace Widget\MemberBundle\Controller\BackendAPI;

use Backend\BaseBundle\Controller\BackendAPI\BaseBackendAPIController;
use Backend\BaseBundle\Form\Type\APIFormTypeItem;
use Sensio\Bundle\FrameworkExtraBundle\Configuration\Method;
use Sensio\Bundle\FrameworkExtraBundle\Configuration\Route;
use Sensio\Bundle\FrameworkExtraBundle\Configuration\Security;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\Validator\Constraints\NotBlank;
use Widget\DiscountBundle\Model\DiscountModule;
use Widget\DiscountBundle\Model\DiscountModuleQuery;
use Widget\MemberBundle\Model\Group;
use Widget\MemberBundle\Model\GroupPeer;
use Widget\MemberBundle\Model\GroupQuery;
use Widget\MemberBundle\Model\MemberGroupQuery;
use Widget\MemberBundle\Model\MemberQuery;

/**
 * @Route("/group")
 * @Security("has_role_or_superadmin('ROLE_MEMBER')")
 */
class GroupController extends BaseBackendAPIController
{

    /**
     * 欄位表單設定
     * 主要是USER送過來的欄位
     * @return APIFormTypeItem[]
     */
    protected function getFormConfig()
    {
        return array(
            (new APIFormTypeItem('name'))->setOptions(array(
                'constraints' => array(
                    new NotBlank(array(
                        'message' => 'error.missing_field',
                    )),
                )
            )),
        );
    }

    /**
     * 新增
     * @Route("s")
     * @Method({"POST"})
     * @Security("has_role_or_superadmin('ROLE_MEMBER_WRITE')")
     */
    public function createAction(Request $request)
    {
        return $this->doProcessForm(new Group(), $request->getContent());
    }

    /**
     * 搜尋
     * @Route("s")
     * @Method({"GET"})
     * @Security("has_role_or_superadmin('ROLE_MEMBER_READ')")
     */
    public function searchAction(Request $request)
    {
        return $this->doSearch($request->query->all(), GroupQuery::create()->distinct(), GroupPeer::class);
    }

    /**
     * 讀取
     * @Route("/{id}")
     * @Method({"GET"})
     * @Security("has_role_or_superadmin('ROLE_MEMBER_READ')")
     */
    public function readAction(Group $group)
    {
        return $this->createJsonSerializeResponse($group, array('detail'));
    }

    /**
     * 讀取
     * @Route("s/all")
     * @Method({"GET"})
     * @Security("has_role_or_superadmin('ROLE_MEMBER_READ')")
     */
    public function listAction(Request $request)
    {
        $group = GroupQuery::create()->find();
        if(!$group){
            return new Response('', Response::HTTP_BAD_REQUEST);
        }
        return $this->createJsonSerializeResponse($group, array('list'));
    }

    /**
     * 更新
     * @Route("/{id}")
     * @Method({"PUT"})
     * @Security("has_role_or_superadmin('ROLE_MEMBER_WRITE')")
     */
    public function updateAction(Request $request, Group $group)
    {
        return $this->doProcessForm($group, $request->getContent());
    }

    /**
     * 刪除
     * @Route("/{id}")
     * @Method({"DELETE"})
     * @Security("has_role_or_superadmin('ROLE_MEMBER_WRITE')")
     */
    public function deleteAction(Group $group)
    {
        $this->deleteObject($group);
        return $this->createJsonResponse();
    }

    /**
     * 批次管理
     * @Route("s")
     * @Method({"PUT"})
     * @Security("has_role_or_superadmin('ROLE_MEMBER_WRITE')")
     */
    public function batchAction(Request $request)
    {
        return parent::batchAction($request);
    }

    /**
     * 批次切換狀態處理
     * @param $ids
     * @param $con
     * @param $column
     */
    protected function batchSwitch($ids, $column, \PropelPDO $con)
    {
        $groups = GroupQuery::create()->findPks($ids, $con);
        foreach ($groups as $group) {
            $value = $group->getByName($column, \BasePeer::TYPE_FIELDNAME);
            $group->setByName($column, !$value, \BasePeer::TYPE_FIELDNAME);
            $group->save($con);
        }
    }

    /**
     * 批次切換狀態處理
     * @param $ids
     * @param $con
     * @param $column
     */
    protected function batchDelete($ids, \PropelPDO $con)
    {
        GroupQuery::create()->findPks($ids, $con)->delete($con);
    }

    /**
     * 設定 群組 會員 關聯
     * @Route("/{id}/membergroup")
     * @Method({"POST"})
     */
    public function relateToGroup(Request $request, Group $group)
    {
        $requestParams = json_decode($request->getContent(), true);
        if (!isset($requestParams['members']) || count($requestParams['members']) == 0) {
            $emptyMember = MemberQuery::create()->findPks(array());
            $group->setMembers($emptyMember)->save();
            return $this->createJsonSerializeResponse($group, array('list'));
        }

        $memberIds = array();
        foreach ($requestParams['members'] as $member){
            $memberIds[] = $member['id'];
        }
        $members = MemberQuery::create()->findPks($memberIds);
        $group->setMembers($members)->save();
        return $this->createJsonSerializeResponse($group, array('list'));
    }

    /**
     * @Route("/check/discountbundle")
     * @Method({"GET"})
     * @Security("has_role_or_superadmin('ROLE_MEMBER_WRITE')")
     */
    public function checkDiscountAction()
    {
        $bundles = $this->container->getParameter('kernel.bundles');
        $enableDiscountBundle = false;
        foreach ($bundles as $key => $bundle) {
            if ($key == 'WidgetDiscountBundle') {
                $enableDiscountBundle = true;
            }
        }
        return $this->createJsonResponse($enableDiscountBundle);
    }
}