<?php
namespace Widget\MemberBundle\Controller\API;

use Backend\BaseBundle\Controller\API\BaseController;
use Backend\BaseBundle\Model\SiteConfigQuery;
use Backend\BaseBundle\Service\ReCaptchaVerifyService;
use Backend\BaseBundle\Service\SendMailService;
use Backend\BaseBundle\SiteConfig\SiteConfigBuilder;
use JMS\DiExtraBundle\Annotation as DI;
use Symfony\Component\HttpFoundation\Response;
use Widget\MemberBundle\Model;
use Sensio\Bundle\FrameworkExtraBundle\Configuration\Route;
use Sensio\Bundle\FrameworkExtraBundle\Configuration\Method;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\Validator\Constraints as Assert;
use Widget\MemberBundle\Model\Member;
use Widget\MemberBundle\Model\MemberQuery;
use Widget\MemberBundle\Service\RecaptchaEnableChecker;

/**
 * @Route("/member")
 */
class MainController extends BaseController
{
    /**
     * @var SiteConfigBuilder
     * @DI\Inject("backend_base.site_config_builder")
     */
    protected $siteConfigBuilder;

    /**
     * 會員註冊
     * @Route("/register")
     * @Method({"POST"})
     * error[400] => 'bad format'
     * error[403] => 'member exist'
     */
    public function registerAction(Request $request)
    {
        $requestData = json_decode($request->getContent(), true); //把資料用JSON解碼
        $validator = $this->get('validator'); //驗證格式
        $constraint = new Assert\Collection(array(
            'missingFieldsMessage' => 'error.missing_field',
            'fields' => array(
                'name' => new Assert\NotBlank(),
                'email' => array(
                    new Assert\NotBlank(array(
                        'message' => 'error.account.missing_field',
                    )),
                    new Assert\Email(array(
                        'message' => 'error.account.format_incorrect_field',
                    )),
                ),
                'password' => new Assert\NotBlank(array(
                    'message' => 'error.password.missing_field',
                )),
            ),
        )); //驗證資料的驗證格式規範

        $errors = $validator->validate($requestData, $constraint); //把資料驗證(資料, 驗證格式)

        #判斷回傳是否有錯誤 等於0沒錯
        if(count($errors) > 0){
            return $this->createJsonSerializeResponse($this->get('api.validator.error.convertor')->toArray($errors), array(), Response::HTTP_BAD_REQUEST);
        }

        if(Model\MemberQuery::create()->findOneByEmail($requestData['email'])){
            return $this->createHttpExceptionResponse(Response::HTTP_FORBIDDEN);
        }

        $member = new Model\Member();
        $member
            ->setName($requestData['name'])
            ->setEmail($requestData['email'])
            ->setPlainPassword($requestData['password'])
            ->regenerateConfirmToken()
            ;

        $this->get('member_manager')->createMember($member);

        $this->sendMail($member, 'register');
        return $this->createJsonResponse(array(
            'id' => $member->getId(),
        ));
    }

    /**
     * 重寄驗證信
     * @Route("/resendverifymail")
     * @Method({"PUT"})
     * error[400] => 'bad recaptcha'
     * error[401] => 'member been locked'
     * error[403] => 'member not exist'
     * error[409] => 'member been enabled'
     */
    public function resendVerifyMailAction(Request $request)
    {
        $requestData = json_decode($request->getContent(), true); //把資料用JSON解碼

        $recaptchaEnableChecker = $this->get('recaptcha.enable.checker');

        if ($recaptchaEnableChecker->isEnable('resendverifymail')){
            if (!isset($requestData['recaptcha']) || !$this->get('backend_base.recaptcha_verify')->verify($requestData['recaptcha'])){
                return $this->createHttpExceptionResponse(Response::HTTP_BAD_REQUEST);
            }
        }

        if(!isset($requestData['email']) || !($member = Model\MemberQuery::create()->findOneByEmail($requestData['email']))){
            return $this->createHttpExceptionResponse(Response::HTTP_FORBIDDEN);
        }

        if($member->getLocked()){
            return $this->createHttpExceptionResponse(Response::HTTP_UNAUTHORIZED);
        }
        
        if($member->getEnabled()){
            return $this->createHttpExceptionResponse(Response::HTTP_CONFLICT);
        }

        $member
            ->regenerateConfirmToken()
            ->save();

        $this->sendMail($member, 'register');

        return $this->createJsonResponse();
    }

    /**
     * 會員啟用
     * @Route("/{id}/enable")
     * @Method({"PUT"})
     * error[401] => 'member been enabled'
     */
    public function enableAction(Model\Member $member, Request $request)
    {
        $requestData = json_decode($request->getContent(), true);

        $token = isset($requestData['token'])?$requestData['token']:'';

        if($member->getEnabled() || $member->getConfirmToken() == null || $member->getConfirmToken() != $token || $member->getTokenExpiredAt('U') < time()){
            return $this->createHttpExceptionResponse(Response::HTTP_UNAUTHORIZED);
        }

        $member
            ->setEnabled(true)
            ->setConfirmToken(null)
            ->setTokenExpiredAt(null)
            ->save();

        return $this->createJsonResponse();
    }

    /**
     * 忘記密碼
     * @Route("/forgetpassword")
     * @Method({"PUT"})
     * error[400] => 'bad recaptcha'
     * error[401] => 'member been not enabled & locked'
     * error[403] => 'member not exist'
     */
    public function forgetpasswordAction(Request $request)
    {
        $requestData = json_decode($request->getContent(), true); //把資料用JSON解碼

        $recaptchaEnableChecker = $this->get('recaptcha.enable.checker');

        if ($recaptchaEnableChecker->isEnable('forgetpassword')){
            if (!isset($requestData['recaptcha']) || !$this->get('backend_base.recaptcha_verify')->verify($requestData['recaptcha'])){
                return $this->createHttpExceptionResponse(Response::HTTP_BAD_REQUEST);
            }
        }

        if(!isset($requestData['email']) || !($member = Model\MemberQuery::create()->findOneByEmail($requestData['email']))){
            return $this->createHttpExceptionResponse(Response::HTTP_FORBIDDEN);
        }

        /** 改為不需驗證是否啟用，只要點選忘記密碼修改密碼時，自動幫他啟用帳號 */
        if($member->getLocked()){
            return $this->createHttpExceptionResponse(Response::HTTP_UNAUTHORIZED);
        }

        if($member->getTokenExpiredAt('U') < time()) {
            $member
                ->regenerateConfirmToken(3600)
                ->save();
        }

        $this->sendMail($member, 'forgetpassword');

        return $this->createJsonResponse();
    }

    /**
     * 變更密碼 ( for 忘記密碼)
     * @Route("/{id}/password")
     * @Method({"PUT"})
     * error[401] => 'bad token'
     * error[409] => 'member been not enabled & locked'
     */
    public function passwordAction(Model\Member $member, Request $request)
    {
        if($member->getLocked()){
            return $this->createHttpExceptionResponse(Response::HTTP_CONFLICT);
        }

        $requestData = json_decode($request->getContent(), true);

        if(!(isset($requestData['token']) && $this->verifyToken($member, $requestData['token']))){
            return $this->createHttpExceptionResponse(Response::HTTP_UNAUTHORIZED);
        }

        $member
            ->setEnabled(true)
            ->setPlainPassword($requestData['password'])
            ->setConfirmToken(null)
            ->setTokenExpiredAt(null)
            ;

        $this->get('member_manager')->updateMember($member);
        return $this->createJsonResponse();
    }

    /**
     * 變更密碼 ( for 登入狀態)
     * @Route("/password")
     * @Method({"PUT"})
     * error[401] => 'bad token'
     * error[409] => 'member been not enabled & locked'
     */
    public function passwordLoginAction(Request $request)
    {
        $authorizedToken = $request->attributes->get('_authorizedToken');

        if(!$authorizedToken){
            return $this->createHttpExceptionResponse(Response::HTTP_UNAUTHORIZED);
        }

        $member = Model\MemberQuery::create()->findPk($authorizedToken->getAuthData()['uid']);

        if(!$member || !$member->getEnabled() || $member->getLocked()){
            return $this->createHttpExceptionResponse(Response::HTTP_CONFLICT);
        }

        $requestData = json_decode($request->getContent(), true);

        if(isset($requestData['oldpassword']) && !$this->get('member_manager')->verifyMember($member, isset($requestData['oldpassword'])?$requestData['oldpassword']:'')){
            return $this->createHttpExceptionResponse(Response::HTTP_FORBIDDEN);
        }

        $member
            ->setPlainPassword($requestData['password'])
            ->setConfirmToken(null)
            ->setTokenExpiredAt(null)
        ;

        $this->get('member_manager')->updateMember($member);
        return $this->createJsonResponse();
    }

    /**
     * 變更個人資料 ( for 登入狀態)
     * @Route("/edit")
     * @Method({"PUT"})
     * error[400] => 'bad format'
     * error[401] => 'bad token'
     */
    public function editAction(Request $request)
    {
        $authorizedToken = $request->attributes->get('_authorizedToken');

        if(!$authorizedToken){
            return $this->createHttpExceptionResponse(Response::HTTP_UNAUTHORIZED);
        }

        $memberInfo = Model\MemberInfoQuery::create()->filterById($authorizedToken->getAuthData()['uid'])->findOneOrCreate();

        $requestArray = json_decode($request->getContent(), true); //把資料用JSON解碼

        $requestData = array_merge(array(
            'name' => null,
            'email' => null,
            'phone' => null,
            'phone2' => null,
            'zip' => null,
            'address' => null,
            'photos' => null
        ), $requestArray); //預設欄位並調整
        
        $validator = $this->get('validator'); //驗證格式
        $constraint = new Assert\Collection(array(
            'missingFieldsMessage' => 'error.missing_field',
            'fields' => array(
                'name' => array(
                    new Assert\NotBlank(array(
                        'message' => 'error.name.missing_field',
                    ))
                ),
                'email' => array(
                    new Assert\Email(array(
                        'message' => 'error.email.format_incorrect_field',
                    )),
                ),
                'phone' => array(),
                'phone2' => array(),
                'zip' => array(),
                'address' => array(),
                'photos' => array(),
            ),
        )); //驗證資料的驗證格式規範

        $errors = $validator->validate($requestData, $constraint); //把資料驗證(資料, 驗證格式)

        #判斷回傳是否有錯誤 等於0沒錯
        if(count($errors) > 0){
            return $this->createJsonSerializeResponse($this->get('api.validator.error.convertor')->toArray($errors), array(), Response::HTTP_BAD_REQUEST);
        }

        $memberInfo
            ->setName($requestData['name'])
            ->setEmail($requestData['email'])
            ->setPhone($requestData['phone'])
            ->setPhone2($requestData['phone2'])
            ->setZip($requestData['zip'])
            ->setAddress($requestData['address'])
            ->setPhotos($requestData['photos'])
            ->save()
        ;
        
        return $this->createJsonResponse();
    }

    /**
     * 會員登入
     * @Route("/login")
     * @Method({"PUT"})
     * error[400] => 'bad format'
     * error[403] => 'member not exist & password empty'
     * error[406] => 'bad host'
     */
    public function loginAction(Request $request)
    {
        $requestData = json_decode($request->getContent(), true); //把資料用JSON解碼
        $validator = $this->get('validator'); //驗證格式
        $constraint = new Assert\Collection(array(
            'missingFieldsMessage' => 'error.missing_field',
            'fields' => array(
                'email' => array(
                    new Assert\NotBlank(array(
                        'message' => 'error.account.missing_field',
                    )),
                    new Assert\Email(array(
                        'message' => 'error.account.format_incorrect_field',
                    )),
                ),
                'password' => array(
                    new Assert\NotBlank(array(
                        'message' => 'error.password.missing_field',
                    )),
                ),
            ),
        )); //驗證資料的驗證格式規範

        $errors = $validator->validate($requestData, $constraint); //把資料驗證(資料, 驗證格式)

        #判斷回傳是否有錯誤 等於0沒錯
        if(count($errors) > 0){
            return $this->createJsonSerializeResponse($this->get('api.validator.error.convertor')->toArray($errors), array(), Response::HTTP_BAD_REQUEST);
        }

        if(!isset($requestData['email']) || !($member = Model\MemberQuery::create()->findOneByEmail($requestData['email']))){
            return $this->createHttpExceptionResponse(Response::HTTP_NOT_FOUND);
        }

        if(!$this->get('member_manager')->verifyMember($member, isset($requestData['password'])?$requestData['password']:'')){
            return $this->createHttpExceptionResponse(Response::HTTP_FORBIDDEN);
        }

        if(!($audience = $request->headers->get('origin', $request->headers->get('host', null)))){
            return $this->createHttpExceptionResponse(Response::HTTP_NOT_ACCEPTABLE);
        }

        $jwtToken = $this->createJWTToken($request, $audience, $member);

        return $this->createJsonResponse(array(
            'token' => (string) $jwtToken,
        ));
    }

    /**
     * 取得 member_info 個人資料
     * @Route("/info")
     * @Method("GET")
     * error[400] => 'bad format'
     * error[401] => 'bad token'
     */
    public function getMemberInfoAction(Request $request)
    {
        $authorizedToken = $request->attributes->get('_authorizedToken');

        if(!$authorizedToken){
            return $this->createHttpExceptionResponse(Response::HTTP_UNAUTHORIZED);
        }

        $member = Model\MemberQuery::create()->findPk($authorizedToken->getAuthData()['uid']);

        if(!$member || !$member->getEnabled() || $member->getLocked()){
            return $this->createHttpExceptionResponse(Response::HTTP_FORBIDDEN);
        }

        return $this->createJsonSerializeResponse($member->getMemberInfo(), array('detail'));
    }

    /**
     * Oauth2 API 登入
     * @Route("/{type}/oauth2")
     * @Method({"PUT"})
     * error[401] => 'bad token'
     * error[403] => 'email empty'
     * error[406] => 'bad host'
     */
    public function Oauth2LoginAction($type, Request $request)
    {
        $params = json_decode($request->getContent(), true);

        if(!isset($params['access_token'])){
            return $this->createHttpExceptionResponse(Response::HTTP_UNAUTHORIZED);
        }

        $accessToken = $params['access_token'];

        $responseData = $this->queryUserInfo($type, $accessToken);

        if(!isset($responseData['email'])){
            return $this->createHttpExceptionResponse(Response::HTTP_FORBIDDEN);
        }

        $member = $this->createMember($responseData);
        $member->setEnabled(true);

        if(!($audience = $request->headers->get('origin', $request->headers->get('host', null)))){
            return $this->createHttpExceptionResponse(Response::HTTP_CONFLICT);
        }

        $this->get('member_manager')->updateMember($member);

        $jwtToken = $this->createJWTToken($request, $audience, $member);

        return $this->createJsonResponse(array(
            'token' => (string) $jwtToken,
        ));
    }

    /**
     * 更新即將過期的 token
     * @Route("/renewtoken")
     * @Method({"PUT"})
     * error[401] => 'bad token'
     * error[403] => 'email empty'
     * error[406] => 'bad host'
     */
    public function renewTokenAction(Request $request)
    {
        $authorizedToken = $request->attributes->get('_authorizedToken');

        if(!$authorizedToken){
            return $this->createHttpExceptionResponse(Response::HTTP_UNAUTHORIZED);
        }

        $member = Model\MemberQuery::create()->findPk($authorizedToken->getAuthData()['uid']);

        if(!$member || $member->getLocked()){
            return $this->createHttpExceptionResponse(Response::HTTP_FORBIDDEN);
        }

        if(!($audience = $request->headers->get('origin', $request->headers->get('host', null)))){
            return $this->createHttpExceptionResponse(Response::HTTP_CONFLICT);
        }

        $jwtToken = $this->createJWTToken($request, $audience, $member);

        return $this->createJsonResponse(array(
            'token' => (string) $jwtToken,
        ));
    }

    /**
     * 第三方登入設定
     * @Route("/oauthconfig")
     */
    public function OauthConfigAction()
    {
        $siteConfig = $this->siteConfigBuilder->build();

        if(!($memberConfig = $siteConfig->get('widget_member'))){
            return $this->createHttpExceptionResponse(Response::HTTP_NOT_FOUND);
        }

        $response = array();
        foreach (array('facebook', 'google') as $type) {
            if(isset($memberConfig["{$type}_client_id"])) {
                $response[] = array(
                    'type' => $type,
                    'client_id' => $memberConfig["{$type}_client_id"],
                );
            }
        }

        return $this->createJsonResponse($response);
    }

    /**
     * 非會員登入驗證 寄送 email magic link
     * @Route("/othermeber")
     * @Method({"PUT"})
     */
    public function otherMemberLoginAction(Request $request)
    {
        $requestData = json_decode($request->getContent(), true);
        /** 驗證email */
        if(!isset($requestData['email']) || !($member = MemberQuery::create()->findOneByEmail($requestData['email']))){
            return $this->createHttpExceptionResponse(Response::HTTP_FORBIDDEN);
        }

        /** @var RecaptchaEnableChecker $recaptchaEnableChecker */
        $recaptchaEnableChecker = $this->get('recaptcha.enable.checker');
        if ($recaptchaEnableChecker->isEnable('otherMemberLogin')){
            /** @var ReCaptchaVerifyService $reCaptchaVerify */
            $reCaptchaVerify = $this->get('backend_base.recaptcha_verify');
            /** recaptcha是否正確 */
            if (!$reCaptchaVerify->verify($requestData['recaptchaResponse'])) {
                return $this->createHttpExceptionResponse(Response::HTTP_CONFLICT);
            }
        }

        /** 寄信 */
        $member->regenerateConfirmToken(30*60)->save();
        $this->doSendLoginLinkMail($member, 'other_member_login');
        return $this->createJsonResponse();
    }

    /**
     * 非會員登入驗證
     * @Route("/{id}/verify")
     * @Method({"PUT"})
     */
    public function otherMemberLoginVerifyAction(Request $request, Member $member)
    {
        $requestData = json_decode($request->getContent(), true);
        $token = isset($requestData['token'])?$requestData['token']:'';

        if($member->getConfirmToken() == null || $member->getConfirmToken() !== $token || $member->getTokenExpiredAt('U') < time()){
            return $this->createHttpExceptionResponse(Response::HTTP_UNAUTHORIZED);
        }

        if(!($audience = $request->headers->get('origin', $request->headers->get('host', null)))){
            return $this->createHttpExceptionResponse(Response::HTTP_CONFLICT);
        }

        $member->setConfirmToken(null)->setTokenExpiredAt(null)->save();

        $jwtToken = $this->createJWTToken($request, $audience, $member);

        return $this->createJsonResponse(array(
            'token' => (string) $jwtToken,
        ));
    }

    protected function doSendLoginLinkMail(Member $member, $templateName)
    {
        $group = array(
            'element' => 'other_member_login',
            'name'  => 'OtherMemberLogin'
        );
        $to = array(
            'email' => array(
                $member->getEmail(),
            ),
            'name' => array(
                $member->getName()
            )
        );
        $templateSubject = array(
            'name' => $member->getName(),
        );
        $templateBody = array(
            'name' => $member->getName(),
            'email' => $member->getEmail(),
            'id' => $member->getId(),
            'token' => $member->getConfirmToken(),
            'token_expire' => $member->getTokenExpiredAt(),
        );
        /** @var SendMailService $sendMailService */
        $sendMailService = $this->get('backend_base.send_mail');
        /** 寄信出去 */
        try {
            $sendMailService->send($group, $to, $templateName, $templateSubject, $templateBody);
        } catch (\Exception $e) {
            return;
        }
    }

    protected function verifyToken(Model\Member $member, $token)
    {
        return $member->getConfirmToken() == $token && $member->getTokenExpiredAt('U') > time();
    }

    protected function twigBuilder(array $templates)
    {
        return new \Twig_Environment(new \Twig_Loader_Array($templates));
    }

    protected function sendMail(Model\Member $member, $type)
    {
        try {
            $memberMailerConfig = $this
                ->get('backend_base.site_config_builder')
                ->build()
                ->get('widget_member');

            $mailerService = $this->get('site_custom_mailer');

            $mailerTwig = $this->twigBuilder(array(
                $type . '_mail_subject' => $memberMailerConfig[$type . '_mail_subject'] ? $memberMailerConfig[$type . '_mail_subject'] : '',
                $type . '_mail' => $memberMailerConfig[$type . '_mail'] ? $memberMailerConfig[$type . '_mail'] : '',
            ));

            $message = $mailerService->newMessage()
                ->setCharset('utf-8')
                ->setContentType('text/html')
                ->setSubject($mailerTwig->render($type . '_mail_subject', array(
                    'name' => $member->getName(),
                )))
                ->setTo($member->getEmail(), $member->getName())
                ->setBody($mailerTwig->render($type . '_mail', array(
                    'name' => $member->getName(),
                    'email' => $member->getEmail(),
                    'id' => $member->getId(),
                    'token' => $member->getConfirmToken(),
                    'token_expire' => $member->getTokenExpiredAt(),
                )));
            $mailerService->get()->send($message);
        }
        catch (\Exception $e){
            return;
        }
    }

    /**
     * @param $accessToken
     * @return mixed
     */
    protected function queryUserInfo($endpoint, $accessToken)
    {
        return $this->get('widget_member.oauth')->queryUserInfo($endpoint, $accessToken);
    }

    /**
     * @param $responseData
     * @return Model\Member
     */
    protected function createMember($responseData)
    {
        if (!($member = Model\MemberQuery::create()->findOneByEmail($responseData['email']))) {
            $member = new Model\Member();
            $member
                ->setEnabled(true)
                ->setName($responseData['name'])
                ->setEmail($responseData['email']);
            $this->get('member_manager')->updateMember($member);
            return $member;
        }
        return $member;
    }

    /**
     * @param Request $request
     * @param $audience
     * @param $member
     * @return \Lcobucci\JWT\Token|null
     */
    protected function createJWTToken(Request $request, $audience, $member)
    {
        $issuer = "{$request->getScheme()}://{$request->getHttpHost()}";
        $signer = $this->get('widget_member.token.signer');
        return $signer->sign($issuer, $audience, $member);
    }
}
