<?php
namespace Widget\InvoiceBundle\Tests\Service;


use Backend\BaseBundle\Tests\Fixture\BaseKernelTestCase;
use Widget\InvoiceBundle\Model\Invoice;
use Widget\InvoiceBundle\Model\InvoiceAllowance;
use Widget\InvoiceBundle\Model\InvoiceItem;
use Widget\InvoiceBundle\Model\InvoicePeer;
use Widget\InvoiceBundle\Service\ECPayInvoice;
use Widget\InvoiceBundle\Service\OrderTokenInvoiceCreateRequest;
use Widget\MemberBundle\Model\MemberQuery;
use Widget\OrderBundle\Model\Order;
use Widget\OrderBundle\Model\OrderItem;
use Widget\OrderBundle\Token\OrderTokenRequest;
use Widget\PaymentBundle\Model\PaymentPeer;

class ECPayInvoiceTest extends BaseKernelTestCase
{
    public function test_createCheckMac_real()
    {
        //arrange
        $originArray = $this->shuffleArray(array(
            'CarruerNum' => '',
            'CarruerType' => '',
            'CustomerAddr' => '%E5%8F%B0%E5%8C%97%E5%B8%82%E5%87%B1%E9%81%94%E6%A0%BC%E8%98%AD%E5%A4%A7%E9%81%931%E8%99%9F',
            'CustomerEmail' => 'ricky%40example.com.tw',
            'CustomerIdentifier' => '',
            'CustomerName' => 'Ricky',
            'Donation' => '1',
            'InvType' => '07',
            'ItemAmount' => '100|150|250',
            'ItemCount' => '1|1|1',
            'ItemPrice' => '100|150|250',
            'ItemTaxType' => '1|1|1',
            'LoveCode' => 'X13579',
            'MerchantID' => '2000132',
            'Print' => '0',
            'RelateNumber' => 'O12345',
            'SalesAmount' => '500',
            'TaxType' => '1',
            'TimeStamp' => '1498034076',
        ));
        $hashKey = 'ejCk326UnaZWKisg';
        $hashIv = 'q9jcZX8Ib9LM8wYk';
        $checkMac = '007388390EC1FB363BF8DF76CCE1C363';
        $service = $this->container->get('widget_invoice.invoice.ecpay');

        //act
        $result = $this->callObjectMethod($service, 'createCheckMac', $originArray, $hashKey, $hashIv);

        //assert
        $this->assertEquals($checkMac, $result);
    }

    public function test_createCheckMac()
    {
        //arrange
        $service = $this->container->get('widget_invoice.invoice.ecpay');
        $originArray = $this->shuffleArray(array(
            'CustomerAddr' => $this->callObjectMethod($service, 'uglyUrlencode', '臺北市南港區三重路19-2號56樓-2 (D棟)'),
            'CustomerEmail' => $this->callObjectMethod($service, 'uglyUrlencode', 'abc@ecpay.com.tw'),
            'CustomerName' => $this->callObjectMethod($service, 'uglyUrlencode', '綠界科技電子支付股份有限公司'),
            'Donation' => '2',
            'InvType' => '07',
            'ItemAmount' => '100',
            'ItemCount' => '1',
            'ItemPrice' => '100',
            'MerchantID' => '2000132',
            'Print' => '0',
            'RelateNumber' => '1234567890E',
            'SalesAmount' => '100',
            'TaxType' => '1',
            'TimeStamp' => '1489735068'
        ));
        $hashKey = 'ejCk326UnaZWKisg';
        $hashIv = 'q9jcZX8Ib9LM8wYk';
        $checkMac = '0BEE7CB7FBCA6784D0C2BCB1EAC4BE0C';

        //act
        $result = $this->callObjectMethod($service, 'createCheckMac', $originArray, $hashKey, $hashIv);

        //assert
        $this->assertEquals($checkMac, $result);
    }

    private function shuffleArray($array)
    {
        $result = array();
        $keys = array_keys($array);
        shuffle($keys);
        foreach ($keys as $key){
            $result[$key] = $array[$key];
        }
        return $result;
    }

    public function test_uglyUrlencode()
    {
        //arrange
        $origin = '-_.!~*() @#$%^&=abcdefghijklmnopqrstuvwxyz中文測試';
        $encoded = '-_.!%7e*()+%40%23%24%25%5e%26%3dabcdefghijklmnopqrstuvwxyz%e4%b8%ad%e6%96%87%e6%b8%ac%e8%a9%a6';
        $service = $this->container->get('widget_invoice.invoice.ecpay');

        //act
        $result = $this->callObjectMethod($service, 'uglyUrlencode', $origin);

        //assert
        $this->assertEquals($encoded, $result);
    }

    protected function createTestConfig()
    {
        return array(
            'merchantId' => '2000132',
            'apiBase' => 'https://einvoice-stage.ecpay.com.tw/',
            'hashKey' => 'ejCk326UnaZWKisg',
            'hashIv' => 'q9jcZX8Ib9LM8wYk',
        );
    }

    public function test_sendOne_fail()
    {
        //arrange
        $apiResponse = array(
            "InvoiceDate" => "2017-06-21 17:54:02",
            "InvoiceNumber" => "",
            "RandomNumber" => "4504",
            "RtnCode" => "1500046",
            "RtnMsg" => "開立發票失敗(自訂編號重覆，請重新設定)",
            "CheckMacValue" => "FB3933E2F27A8A17F22C55775D5A0E4D",
        );

        $checkMackValue = 'C13F338AC87EF95FC0F0A973026D2F5E';
        $postData = array(
            'CarruerNum' => '',
            'CarruerType' => '1',
            'CustomerID' => 'O12345',
            'CustomerAddr' => '%E5%8F%B0%E5%8C%97%E5%B8%82%E5%87%B1%E9%81%94%E6%A0%BC%E8%98%AD%E5%A4%A7%E9%81%931%E8%99%9F',
            'CustomerEmail' => 'ricky%40example.com.tw',
            'CustomerIdentifier' => '',
            'CustomerName' => 'Ricky',
            'Donation' => '1',
            'InvType' => '07',
            'InvoiceRemark' => '%E9%80%99%E6%98%AF%E6%B8%AC%E8%A9%A6%E7%99%BC%E7%A5%A8',
            'ItemAmount' => '100|300|750',
            'ItemCount' => '1|2|3',
            'ItemName' => '%E5%95%86%E5%93%81%E5%90%8D%E7%A8%B1%E4%B8%80%7C%E5%95%86%E5%93%81%E5%90%8D%E7%A8%B1%E4%BA%8C%7C%E5%95%86%E5%93%81%E5%90%8D%E7%A8%B1%E4%B8%89',
            'ItemPrice' => '100|150|250',
            'ItemRemark' => '%E5%95%86%E5%93%81%E5%90%8D%E7%A8%B1%E4%B8%80%7C%E5%95%86%E5%93%81%E5%90%8D%E7%A8%B1%E4%BA%8C%7C%E5%95%86%E5%93%81%E5%90%8D%E7%A8%B1%E4%B8%89',
            'ItemTaxType' => '1|1|1',
            'ItemWord' => '%E6%89%B9%7C%E9%9A%BB%7C%E5%80%8B',
            'LoveCode' => 'X13579',
            'MerchantID' => '2000132',
            'Print' => '0',
            'RelateNumber' => 'O12345',
            'SalesAmount' => '1150',
            'TaxType' => '1',
            'TimeStamp' => strtotime($apiResponse['InvoiceDate']),
        );
        $config = $this->createTestConfig();
        $invoiceService = $this->getMockBuilder(ECPayInvoice::class)
            ->setMethods(array('makeTime', 'doPost', 'createCheckMac', 'filterInvoiceUnCheckArray'))
            ->getMock();
        $invoiceService
            ->expects($this->once())
            ->method('filterInvoiceUnCheckArray')
            ->with($postData)
            ->willReturn($postData);
        $invoiceService
            ->expects($this->once())
            ->method('createCheckMac')
            ->with($postData, $config['hashKey'], $config['hashIv'])
            ->willReturn($checkMackValue);
        $invoiceService
            ->expects($this->atLeastOnce())
            ->method('makeTime')
            ->willReturn(strtotime($apiResponse['InvoiceDate']));
        $invoiceService
            ->expects($this->once())
            ->method('doPost')
            ->willReturnCallback(function($uri, $data) use($postData, $apiResponse, $checkMackValue){
                $postData['CheckMacValue'] = $checkMackValue;
                ksort($data);
                ksort($postData);
                $this->assertEquals('/Invoice/Issue', $uri);
                $this->assertEquals($postData, $data);
                return $apiResponse;
            })
            ;
        $this->setObjectAttribute($invoiceService, 'config', $config);
        $invoiceItem1 = new InvoiceItem();
        $invoiceItem1
            ->setDescription('商品名稱一')
            ->setQuantity(1)
            ->setUnitPrice(100)
            ->setUnit('批');
        $invoiceItem2 = new InvoiceItem();
        $invoiceItem2
            ->setDescription('商品名稱二')
            ->setQuantity(2)
            ->setUnitPrice(150)
            ->setUnit('隻');
        $invoiceItem3 = new InvoiceItem();
        $invoiceItem3
            ->setDescription('商品名稱三')
            ->setQuantity(3)
            ->setUnitPrice(250)
            ->setUnit('個');
        $invoice = $this->getMockBuilder(Invoice::class)
            ->setMethods(array('save'))
            ->getMock();
        $invoice
            ->expects($this->once())
            ->method('save')
            ;
        $invoice
            ->addInvoiceItem($invoiceItem1)
            ->addInvoiceItem($invoiceItem2)
            ->addInvoiceItem($invoiceItem3)
            ->setId('O12345')
            ->setPaymentPaid(true)
            ->setEmail('ricky@example.com.tw')
            ->setDonateMark(InvoicePeer::DONATE_MARK_DONATE)
            ->setLoveCode('13579')
            ->setApiType(ECPayInvoice::API_TYPE)
            ->setRemark('這是測試發票')
            ->setName('Ricky')
            ->setAddress('台北市凱達格蘭大道1號')
            ;
        $invoice->syncTotalPrice();

        //act
        $result = $invoiceService->sendOne($invoice);

        //assert
        $this->assertFalse($result);
        $this->assertEquals($apiResponse, $invoice->getApiResponse());
        $this->assertEmpty($invoice->getInvoiceNumber());
        $this->assertEquals($apiResponse['InvoiceDate'], $invoice->getProcessAt('%Y-%m-%d %T'));
    }

    public function test_sendOne_success()
    {
        //arrange
        $apiResponse = array(
            "InvoiceDate" => "2016-02-25 17:18:57",
            "InvoiceNumber" => "EV00004242",
            "RandomNumber" => "5528",
            "RtnCode" => "1",
            "RtnMsg" => "開立發票成功",
            "CheckMacValue" => "A6E8A40FF19717A4B169165B9C672FB4",
        );

        $checkMackValue = 'C13F338AC87EF95FC0F0A973026D2F5E';
        $postData = array(
            'CarruerNum' => '',
            'CarruerType' => '1',
            'CustomerID' => 'O12345',
            'CustomerAddr' => '%E5%8F%B0%E5%8C%97%E5%B8%82%E5%87%B1%E9%81%94%E6%A0%BC%E8%98%AD%E5%A4%A7%E9%81%931%E8%99%9F',
            'CustomerEmail' => 'ricky%40example.com.tw',
            'CustomerIdentifier' => '',
            'CustomerName' => 'Ricky',
            'Donation' => '1',
            'InvType' => '07',
            'InvoiceRemark' => '%E9%80%99%E6%98%AF%E6%B8%AC%E8%A9%A6%E7%99%BC%E7%A5%A8',
            'ItemAmount' => '100|300|750',
            'ItemCount' => '1|2|3',
            'ItemName' => '%E5%95%86%E5%93%81%E5%90%8D%E7%A8%B1%E4%B8%80%7C%E5%95%86%E5%93%81%E5%90%8D%E7%A8%B1%E4%BA%8C%7C%E5%95%86%E5%93%81%E5%90%8D%E7%A8%B1%E4%B8%89',
            'ItemPrice' => '100|150|250',
            'ItemRemark' => '%E5%95%86%E5%93%81%E5%90%8D%E7%A8%B1%E4%B8%80%7C%E5%95%86%E5%93%81%E5%90%8D%E7%A8%B1%E4%BA%8C%7C%E5%95%86%E5%93%81%E5%90%8D%E7%A8%B1%E4%B8%89',
            'ItemTaxType' => '1|1|1',
            'ItemWord' => '%E6%89%B9%7C%E9%9A%BB%7C%E5%80%8B',
            'LoveCode' => 'X13579',
            'MerchantID' => '2000132',
            'Print' => '0',
            'RelateNumber' => 'O12345',
            'SalesAmount' => '1150',
            'TaxType' => '1',
            'TimeStamp' => strtotime($apiResponse['InvoiceDate']),
        );
        $config = $this->createTestConfig();
        $invoiceService = $this->getMockBuilder(ECPayInvoice::class)
            ->setMethods(array('makeTime', 'doPost', 'createCheckMac', 'filterInvoiceUnCheckArray'))
            ->getMock();
        $invoiceService
            ->expects($this->once())
            ->method('filterInvoiceUnCheckArray')
            ->with($postData)
            ->willReturn($postData);
        $invoiceService
            ->expects($this->once())
            ->method('createCheckMac')
            ->with($postData, $config['hashKey'], $config['hashIv'])
            ->willReturn($checkMackValue);
        $invoiceService
            ->expects($this->atLeastOnce())
            ->method('makeTime')
            ->willReturn(strtotime($apiResponse['InvoiceDate']));
        $invoiceService
            ->expects($this->once())
            ->method('doPost')
            ->willReturnCallback(function($uri, $data) use($postData, $apiResponse, $checkMackValue){
                $postData['CheckMacValue'] = $checkMackValue;
                ksort($data);
                ksort($postData);
                $this->assertEquals('/Invoice/Issue', $uri);
                $this->assertEquals($postData, $data);
                return $apiResponse;
            })
        ;
        $this->setObjectAttribute($invoiceService, 'config', $config);
        $invoiceItem1 = new InvoiceItem();
        $invoiceItem1
            ->setDescription('商品名稱一')
            ->setQuantity(1)
            ->setUnitPrice(100)
            ->setUnit('批');
        $invoiceItem2 = new InvoiceItem();
        $invoiceItem2
            ->setDescription('商品名稱二')
            ->setQuantity(2)
            ->setUnitPrice(150)
            ->setUnit('隻');
        $invoiceItem3 = new InvoiceItem();
        $invoiceItem3
            ->setDescription('商品名稱三')
            ->setQuantity(3)
            ->setUnitPrice(250)
            ->setUnit('個');
        $invoice = $this->getMockBuilder(Invoice::class)
            ->setMethods(array('save'))
            ->getMock();
        $invoice
            ->expects($this->once())
            ->method('save')
        ;
        $invoice
            ->addInvoiceItem($invoiceItem1)
            ->addInvoiceItem($invoiceItem2)
            ->addInvoiceItem($invoiceItem3)
            ->setId('O12345')
            ->setPaymentPaid(true)
            ->setEmail('ricky@example.com.tw')
            ->setDonateMark(InvoicePeer::DONATE_MARK_DONATE)
            ->setLoveCode('13579')
            ->setApiType(ECPayInvoice::API_TYPE)
            ->setRemark('這是測試發票')
            ->setName('Ricky')
            ->setAddress('台北市凱達格蘭大道1號')
        ;
        $invoice->syncTotalPrice();

        //act
        $result = $invoiceService->sendOne($invoice);

        //assert
        $this->assertTrue($result);
        $this->assertEquals($apiResponse, $invoice->getApiResponse());
        $this->assertEquals($apiResponse['InvoiceNumber'], $invoice->getInvoiceNumber());
        $this->assertEquals($apiResponse['InvoiceDate'], $invoice->getProcessAt('%Y-%m-%d %T'));
    }

    public function test_sendAllowance()
    {
        //arrange
        $apiResponse = array(
            'IA_Allow_No' => '2017062210423489',
            'IA_Date' => '2017-06-22 10:42:44',
            'IA_Invoice_No' => 'TT00011308',
            'IA_Remain_Allowance_Amt' => '0',
            'RtnCode' => '1',
            'RtnMsg' => '成功.',
            'CheckMacValue' => '29244C29B1B6FF5D5F3038F4E521AD1F',
        );

        $postData = array(
            'TimeStamp' => '1498038897',
            'MerchantID' => '2000132',
            'InvoiceNo' => 'EV00004242',
            'AllowanceNotify' => 'E',
            'CustomerName' => 'Ricky',
            'NotifyMail' => 'ricky%40example.com.tw',
            'AllowanceAmount' => '1150',
            'ItemName' => '%E5%95%86%E5%93%81%E5%90%8D%E7%A8%B1%E4%B8%80%7C%E5%95%86%E5%93%81%E5%90%8D%E7%A8%B1%E4%BA%8C%7C%E5%95%86%E5%93%81%E5%90%8D%E7%A8%B1%E4%B8%89',
            'ItemCount' => '1|2|3',
            'ItemWord' => '%E6%89%B9%7C%E9%9A%BB%7C%E5%80%8B',
            'ItemPrice' => '100|150|250',
            'ItemAmount' => '100|300|750',
            'ItemTaxType' => '1|1|1',
            'CheckMacValue' => 'C7FD9CC07E4A5BC58714B8A58599BD46',
        );
        $config = $this->createTestConfig();
        $invoiceService = $this->getMockBuilder(ECPayInvoice::class)
            ->setMethods(array('makeTime', 'doPost'))
            ->getMock();
        $invoiceService
            ->expects($this->once())
            ->method('makeTime')
            ->willReturn(1498038897);
        $invoiceService
            ->expects($this->once())
            ->method('doPost')
            ->willReturnCallback(function($uri, $data) use($postData, $apiResponse){
                ksort($data);
                ksort($postData);
                $this->assertEquals('/Invoice/Allowance', $uri);
                $this->assertEquals($postData, $data);
                return $apiResponse;
            })
        ;
        $this->setObjectAttribute($invoiceService, 'config', $config);
        $invoiceItem1 = new InvoiceItem();
        $invoiceItem1
            ->setDescription('商品名稱一')
            ->setQuantity(1)
            ->setUnitPrice(100)
            ->setUnit('批');
        $invoiceItem2 = new InvoiceItem();
        $invoiceItem2
            ->setDescription('商品名稱二')
            ->setQuantity(2)
            ->setUnitPrice(150)
            ->setUnit('隻');
        $invoiceItem3 = new InvoiceItem();
        $invoiceItem3
            ->setDescription('商品名稱三')
            ->setQuantity(3)
            ->setUnitPrice(250)
            ->setUnit('個');
        $invoice = new Invoice();
        $invoice
            ->addInvoiceItem($invoiceItem1)
            ->addInvoiceItem($invoiceItem2)
            ->addInvoiceItem($invoiceItem3)
            ->setId('O12345')
            ->setPaymentPaid(true)
            ->setEmail('ricky@example.com.tw')
            ->setDonateMark(InvoicePeer::DONATE_MARK_DONATE)
            ->setLoveCode('13579')
            ->setApiType(ECPayInvoice::API_TYPE)
            ->setRemark('這是測試發票')
            ->setName('Ricky')
            ->setAddress('台北市凱達格蘭大道1號')
        ;
        $invoice->setInvoiceNumber('EV00004242');
        $invoice->syncTotalPrice();
        $invoiceAllowance = $this->getMockBuilder(InvoiceAllowance::class)
            ->setMethods(array('save'))
            ->getMock();
        $invoiceAllowance
            ->expects($this->once())
            ->method('save')
        ;
        $invoiceAllowance
            ->setInvoice($invoice)
            ;

        //act
        $result = $invoiceService->sendAllowance($invoiceAllowance);

        //assert
        $this->assertTrue($result);
        $this->assertEquals($apiResponse, $invoiceAllowance->getApiResponse());
        $this->assertEquals($apiResponse['IA_Date'], $invoiceAllowance->getProcessAt('%Y-%m-%d %T'));
    }

    public function test_sendAllowance_fail()
    {
        //arrange
        $apiResponse = array(
            'RtnCode' => '2000018',
            'RtnMsg' => '無該筆發票資料!',
            'CheckMacValue' => 'DA9CE2520652E79AF4B0CBB6F5F2D17D',
        );

        $postData = array(
            'TimeStamp' => '1498038897',
            'MerchantID' => '2000132',
            'InvoiceNo' => 'EV00004242',
            'AllowanceNotify' => 'E',
            'CustomerName' => 'Ricky',
            'NotifyMail' => 'ricky%40example.com.tw',
            'AllowanceAmount' => '1150',
            'ItemName' => '%E5%95%86%E5%93%81%E5%90%8D%E7%A8%B1%E4%B8%80%7C%E5%95%86%E5%93%81%E5%90%8D%E7%A8%B1%E4%BA%8C%7C%E5%95%86%E5%93%81%E5%90%8D%E7%A8%B1%E4%B8%89',
            'ItemCount' => '1|2|3',
            'ItemWord' => '%E6%89%B9%7C%E9%9A%BB%7C%E5%80%8B',
            'ItemPrice' => '100|150|250',
            'ItemAmount' => '100|300|750',
            'ItemTaxType' => '1|1|1',
            'CheckMacValue' => 'C7FD9CC07E4A5BC58714B8A58599BD46',
        );
        $config = $this->createTestConfig();
        $invoiceService = $this->getMockBuilder(ECPayInvoice::class)
            ->setMethods(array('makeTime', 'doPost'))
            ->getMock();
        $invoiceService
            ->expects($this->once())
            ->method('makeTime')
            ->willReturn(1498038897);
        $invoiceService
            ->expects($this->once())
            ->method('doPost')
            ->willReturnCallback(function($uri, $data) use($postData, $apiResponse){
                ksort($data);
                ksort($postData);
                $this->assertEquals('/Invoice/Allowance', $uri);
                $this->assertEquals($postData, $data);
                return $apiResponse;
            })
        ;
        $this->setObjectAttribute($invoiceService, 'config', $config);
        $invoiceItem1 = new InvoiceItem();
        $invoiceItem1
            ->setDescription('商品名稱一')
            ->setQuantity(1)
            ->setUnitPrice(100)
            ->setUnit('批');
        $invoiceItem2 = new InvoiceItem();
        $invoiceItem2
            ->setDescription('商品名稱二')
            ->setQuantity(2)
            ->setUnitPrice(150)
            ->setUnit('隻');
        $invoiceItem3 = new InvoiceItem();
        $invoiceItem3
            ->setDescription('商品名稱三')
            ->setQuantity(3)
            ->setUnitPrice(250)
            ->setUnit('個');
        $invoice = new Invoice();
        $invoice
            ->addInvoiceItem($invoiceItem1)
            ->addInvoiceItem($invoiceItem2)
            ->addInvoiceItem($invoiceItem3)
            ->setId('O12345')
            ->setPaymentPaid(true)
            ->setEmail('ricky@example.com.tw')
            ->setDonateMark(InvoicePeer::DONATE_MARK_DONATE)
            ->setLoveCode('13579')
            ->setApiType(ECPayInvoice::API_TYPE)
            ->setRemark('這是測試發票')
            ->setName('Ricky')
            ->setAddress('台北市凱達格蘭大道1號')
        ;
        $invoice->setInvoiceNumber('EV00004242');
        $invoice->syncTotalPrice();
        $invoiceAllowance = $this->getMockBuilder(InvoiceAllowance::class)
            ->setMethods(array('save'))
            ->getMock();
        $invoiceAllowance
            ->expects($this->once())
            ->method('save')
        ;
        $invoiceAllowance
            ->setInvoice($invoice)
        ;

        //act
        $result = $invoiceService->sendAllowance($invoiceAllowance);

        //assert
        $this->assertFalse($result);
        $this->assertEquals($apiResponse, $invoiceAllowance->getApiResponse());
        $this->assertEquals(strftime('%Y-%m-%d', time()), $invoiceAllowance->getProcessAt('%Y-%m-%d'));
    }

    public function test_filterInvoiceUnCheckArray()
    {
        //arrange
        $originArray = array(
            'CustomerAddr' => ' 臺北市南港區三重路19-2號56樓-2 (D棟)',
            'CustomerEmail' => ' abc@ecpay.com.tw',
            'CustomerName' => ' 綠界科技電子支付股份有限公司',
            'Donation' => ' 2',
            'InvType' => ' 07',
            'ItemAmount' => ' 100',
            'ItemCount' => ' 1',
            'ItemPrice' => ' 100',
            'MerchantID' => ' 2000132',
            'Print' => ' 0',
            'RelateNumber' => ' 1234567890E',
            'SalesAmount' => ' 100',
            'TaxType' => ' 1',
            'TimeStamp' => ' 1489735068',
            'ItemName' => ' 1489735068',
        );

        $resultArray = array(
            'CustomerAddr' => ' 臺北市南港區三重路19-2號56樓-2 (D棟)',
            'CustomerEmail' => ' abc@ecpay.com.tw',
            'CustomerName' => ' 綠界科技電子支付股份有限公司',
            'Donation' => ' 2',
            'InvType' => ' 07',
            'ItemAmount' => ' 100',
            'ItemCount' => ' 1',
            'ItemPrice' => ' 100',
            'MerchantID' => ' 2000132',
            'Print' => ' 0',
            'RelateNumber' => ' 1234567890E',
            'SalesAmount' => ' 100',
            'TaxType' => ' 1',
            'TimeStamp' => ' 1489735068',
        );
        $invoiceService = new ECPayInvoice();

        //act
        $result = $this->callObjectMethod($invoiceService, 'filterInvoiceUnCheckArray', $originArray);

        //assert
        $this->assertEquals($resultArray, $result);
    }

}
