<?php

namespace Widget\InvoiceBundle\Tests\Service;

use Backend\BaseBundle\Tests\Fixture\BaseKernelTestCase;
use Symfony\Component\EventDispatcher\EventDispatcherInterface;
use Widget\InvoiceBundle\Model\Invoice;
use Widget\InvoiceBundle\Model\InvoiceAllowance;
use Widget\InvoiceBundle\Model\InvoiceAllowanceQuery;
use Widget\InvoiceBundle\Model\InvoiceItem;
use Widget\InvoiceBundle\Model\InvoiceQuery;
use Widget\InvoiceBundle\Service\CetustekInvoice;
use Widget\OrderBundle\Model\Order;

class CetustekInvoiceTest extends BaseKernelTestCase
{
    public function test_init_service()
    {
        //arrange
        //act
        $invoiceService = $this->container->get('widget_invoice.invoice.cetustek');

        //assert
        $this->assertInstanceOf(EventDispatcherInterface::class, $this->getObjectAttribute($invoiceService, 'eventDispatcher'));
        $this->assertInstanceOf(EventDispatcherInterface::class, $this->getObjectAttribute($invoiceService, 'eventDispatcher'));
    }

    public function test_injectConfig()
    {
        // arrange
        $config = array(
            CetustekInvoice::API_TYPE => array(
                'gateway_url' => 'https://www.ei.com.tw/InvoiceB2C/InvoiceAPI?wsdl',
            )
        );

        $invoice = $this->getMockBuilder(CetustekInvoice::class)
            ->disableOriginalConstructor()
            ->setMethods(array('initSoapClient'))
            ->getMock();
        $invoice
            ->expects($this->once())
            ->method('initSoapClient')
            ->with($config[CetustekInvoice::API_TYPE]['gateway_url'])
            ;

        // act
        $invoice->injectConfig($config);

        // assert
        $this->assertEquals($config[CetustekInvoice::API_TYPE], $this->getObjectAttribute($invoice, 'config'));
    }

    public function test_injectConfig_null_config()
    {
        // arrange
        $config = array(
            CetustekInvoice::API_TYPE => array()
        );

        $invoice = $this->getMockBuilder(CetustekInvoice::class)
            ->disableOriginalConstructor()
            ->setMethods(array('initSoapClient'))
            ->getMock();
        $invoice
            ->expects($this->never())
            ->method('initSoapClient')
        ;

        // act
        $invoice->injectConfig($config);

        // assert
        $this->assertEquals($config[CetustekInvoice::API_TYPE], $this->getObjectAttribute($invoice, 'config'));
    }

    public function test_soapClient()
    {
        // arrange
        $soapWsdl = 'https://www.ei.com.tw/InvoiceB2C/InvoiceAPI?wsdl';
        $invoice = new CetustekInvoice();

        // act
        $this->callObjectMethod($invoice, 'initSoapClient', $soapWsdl);

        // assert
        $this->assertInstanceOf(\SoapClient::class, $this->getObjectAttribute($invoice, 'soapClient'));
    }

    public function test_xmlToArray()
    {
        //arrange
        $xml = <<<EOT
<?xml version="1.0" encoding="UTF-8"?>
<list attrib="2.8">
    <a>
        <b>data1</b>
        <b>data2</b>
        <b>data3</b>
    </a>
    <b>test2</b>
</list>
EOT;
        $array = array(
            'list' => array(
                '@attributes' => array('attrib' => '2.8'),
                'a' => array(
                    'b' => array(
                        'data1',
                        'data2',
                        'data3'
                    ),
                ),
                'b' => 'test2',
            ),
        );
        $invoiceService = new CetustekInvoice();

        //act
        $result = $this->callObjectMethod($invoiceService, 'xmlToArray', $xml);

        //assert
        $this->assertEquals($array, $result);
    }

    public function test_arrayToXML_with_no_headers()
    {
        //arrange
        $array = array(
            'list' => array(
                'a' => array(
                    'b' => array(
                        'data1',
                        'data2',
                        'data3',
                        array(
                            'd' => array(0, 1, 2, 3),
                        ),
                    ),
                ),
                'b' => 'test2',
            ),
        );
        $xml = '<?xml version="1.0" encoding="utf-8"?><Invoice XSDVersion="2.8"><list><a><b>data1</b><b>data2</b><b>data3</b><b><d>0</d><d>1</d><d>2</d><d>3</d></b></a><b>test2</b></list></Invoice>';
        $invoiceService = new CetustekInvoice();

        //act
        $result = $this->callObjectMethod($invoiceService, 'arrayToXML', $array);

        //assert
        $this->assertEquals($xml, $result);
    }

    public function test_sendInvoice()
    {
        //arrange
        $config = array(
            'rentid' => '12853714',
            'invoice_type' => '03',
        );

        $testXml = 'testXml';
        $invoiceNumber = 'A123456789';
        $apiResponse = json_decode(json_encode(array(
            'return' => $invoiceNumber,
        )));
        $zip = '123';
        $address = 'test address';
        $array = array(
            'OrderId' => 'P12345',
            'OrderDate' => strftime('%Y/%m/%d', time()),
            'BuyerIdentifier' => '12853714',
            'BuyerName' => '數位因子',
            'BuyerAddress' => "$zip $address",
            'BuyerEmailAddress' => 'ricky@dgfactor.com',
            'DonateMark' => '01',
            'InvoiceType' => $config['invoice_type'],
            'CarrierType' => 'CJ12345',
            'CarrierId1' => 'C123456',
            'CarrierId2' => 'C123456',
            'NPOBAN' => '12345',
            'PayWay' => 2,
            'TaxType' => 1,
            'TaxRate' => '0.05',
            'Remark' => "test remark",
            'Details' => array(
                'ProductItem' => array(
                    array(
                        'ProductionCode' => 'A12345',
                        'Description' => 'test product 1',
                        'Quantity' => 10,
                        'UnitPrice' => 100,
                        'Unit' => 'pcs',
                    ),
                    array(
                        'ProductionCode' => 'B12345',
                        'Description' => 'test product 2',
                        'Quantity' => 20,
                        'UnitPrice' => 200,
                        'Unit' => 'pcs',
                    ),
                    array(
                        'ProductionCode' => 'C12345',
                        'Description' => 'test product 3',
                        'Quantity' => 30,
                        'UnitPrice' => 300,
                        'Unit' => 'pcs',
                    ),
                ),
            ),
        );
        $invoiceItem0 = new InvoiceItem();
        $invoiceItem0
            ->setProductId($array['Details']['ProductItem'][0]['ProductionCode'])
            ->setDescription($array['Details']['ProductItem'][0]['Description'])
            ->setQuantity($array['Details']['ProductItem'][0]['Quantity'])
            ->setUnit($array['Details']['ProductItem'][0]['Unit'])
            ->setUnitPrice($array['Details']['ProductItem'][0]['UnitPrice'])
            ;
        $invoiceItem1 = new InvoiceItem();
        $invoiceItem1
            ->setProductId($array['Details']['ProductItem'][1]['ProductionCode'])
            ->setDescription($array['Details']['ProductItem'][1]['Description'])
            ->setQuantity($array['Details']['ProductItem'][1]['Quantity'])
            ->setUnit($array['Details']['ProductItem'][1]['Unit'])
            ->setUnitPrice($array['Details']['ProductItem'][1]['UnitPrice'])
        ;
        $invoiceItem2 = new InvoiceItem();
        $invoiceItem2
            ->setProductId($array['Details']['ProductItem'][2]['ProductionCode'])
            ->setDescription($array['Details']['ProductItem'][2]['Description'])
            ->setQuantity($array['Details']['ProductItem'][2]['Quantity'])
            ->setUnit($array['Details']['ProductItem'][2]['Unit'])
            ->setUnitPrice($array['Details']['ProductItem'][2]['UnitPrice'])
        ;
        $order = new Order();
        $order->setId($array['OrderId']);
        $invoice = new Invoice();
        $invoice
            ->setId($order->getId())
            ->setIdentifier($array['BuyerIdentifier'])
            ->setName($array['BuyerName'])
            ->setZip($zip)
            ->setAddress($address)
            ->setEmail($array['BuyerEmailAddress'])
            ->setDonateMark($array['DonateMark'])
            ->setCarrierType($array['CarrierType'])
            ->setCarrierId($array['CarrierId1'])
            ->setLoveCode($array['NPOBAN'])
            ->setRemark($array['Remark'])
            ->setPayway($array['PayWay'])
            ->addInvoiceItem($invoiceItem0)
            ->addInvoiceItem($invoiceItem1)
            ->addInvoiceItem($invoiceItem2)
            ;

        $soapClient = $this->getMockBuilder(\stdClass::class)
            ->setMethods(array('CreateInvoiceV3'))
            ->getMock();

        $soapClient
            ->expects($this->once())
            ->method('CreateInvoiceV3')
            ->with(array(
                'invoicexml' => $testXml,
                'hastax' => 1,
                'rentid' => $config['rentid']
            ))
            ->willReturn($apiResponse);

        $invoiceService = $this->getMockBuilder(CetustekInvoice::class)
            ->setMethods(array('arrayToXML'))
            ->getMock();

        $invoiceService
            ->expects($this->once())
            ->method('arrayToXML')
            ->with($array)
            ->willReturn($testXml);
        $this->setObjectAttribute($invoiceService, 'config', $config);
        $this->setObjectAttribute($invoiceService, 'soapClient', $soapClient);

        //act
        $result = $this->callObjectMethod($invoiceService, 'sendInvoice', $invoice);

        //assert
        $this->assertTrue($result);
        $this->assertEquals(json_encode($apiResponse), $invoice->getApiResponse());
        $this->assertLessThan(5, time() - $invoice->getProcessAt('U'));
    }

    public function test_sendInvoice_error()
    {
        //arrange
        $config = array(
            'rentid' => '12853714',
            'invoice_type' => '03',
        );

        $testXml = 'testXml';
        $invoiceNumber = 'A123456789';
        $apiResponse = json_decode(json_encode(array(
            'return' => $invoiceNumber,
        )));
        $zip = '123';
        $address = 'test address';
        $array = array(
            'OrderId' => 'P12345',
            'OrderDate' => strftime('%Y/%m/%d', time()),
            'BuyerIdentifier' => '12853714',
            'BuyerName' => '數位因子',
            'BuyerAddress' => "$zip $address",
            'BuyerEmailAddress' => 'ricky@dgfactor.com',
            'DonateMark' => '01',
            'InvoiceType' => $config['invoice_type'],
            'CarrierType' => 'CJ12345',
            'CarrierId1' => 'C123456',
            'CarrierId2' => 'C123456',
            'NPOBAN' => '12345',
            'PayWay' => 2,
            'TaxType' => 1,
            'TaxRate' => '0.05',
            'Remark' => "test remark",
            'Details' => array(
                'ProductItem' => array(
                    array(
                        'ProductionCode' => 'A12345',
                        'Description' => 'test product 1',
                        'Quantity' => 10,
                        'UnitPrice' => 100,
                        'Unit' => 'pcs',
                    ),
                    array(
                        'ProductionCode' => 'B12345',
                        'Description' => 'test product 2',
                        'Quantity' => 20,
                        'UnitPrice' => 200,
                        'Unit' => 'pcs',
                    ),
                    array(
                        'ProductionCode' => 'C12345',
                        'Description' => 'test product 3',
                        'Quantity' => 30,
                        'UnitPrice' => 300,
                        'Unit' => 'pcs',
                    ),
                ),
            ),
        );
        $invoiceItem0 = new InvoiceItem();
        $invoiceItem0
            ->setProductId($array['Details']['ProductItem'][0]['ProductionCode'])
            ->setDescription($array['Details']['ProductItem'][0]['Description'])
            ->setQuantity($array['Details']['ProductItem'][0]['Quantity'])
            ->setUnit($array['Details']['ProductItem'][0]['Unit'])
            ->setUnitPrice($array['Details']['ProductItem'][0]['UnitPrice'])
        ;
        $invoiceItem1 = new InvoiceItem();
        $invoiceItem1
            ->setProductId($array['Details']['ProductItem'][1]['ProductionCode'])
            ->setDescription($array['Details']['ProductItem'][1]['Description'])
            ->setQuantity($array['Details']['ProductItem'][1]['Quantity'])
            ->setUnit($array['Details']['ProductItem'][1]['Unit'])
            ->setUnitPrice($array['Details']['ProductItem'][1]['UnitPrice'])
        ;
        $invoiceItem2 = new InvoiceItem();
        $invoiceItem2
            ->setProductId($array['Details']['ProductItem'][2]['ProductionCode'])
            ->setDescription($array['Details']['ProductItem'][2]['Description'])
            ->setQuantity($array['Details']['ProductItem'][2]['Quantity'])
            ->setUnit($array['Details']['ProductItem'][2]['Unit'])
            ->setUnitPrice($array['Details']['ProductItem'][2]['UnitPrice'])
        ;
        $order = new Order();
        $order->setId($array['OrderId']);
        $invoice = new Invoice();
        $invoice
            ->setId($order->getId())
            ->setIdentifier($array['BuyerIdentifier'])
            ->setName($array['BuyerName'])
            ->setZip($zip)
            ->setAddress($address)
            ->setEmail($array['BuyerEmailAddress'])
            ->setDonateMark($array['DonateMark'])
            ->setCarrierType($array['CarrierType'])
            ->setCarrierId($array['CarrierId1'])
            ->setLoveCode($array['NPOBAN'])
            ->setRemark($array['Remark'])
            ->setPayway($array['PayWay'])
            ->addInvoiceItem($invoiceItem0)
            ->addInvoiceItem($invoiceItem1)
            ->addInvoiceItem($invoiceItem2)
        ;

        $soapClient = $this->getMockBuilder(\stdClass::class)
            ->setMethods(array('CreateInvoiceV3'))
            ->getMock();

        $soapClient
            ->expects($this->once())
            ->method('CreateInvoiceV3')
            ->with(array(
                'invoicexml' => $testXml,
                'hastax' => 1,
                'rentid' => $config['rentid']
            ))
            ->willReturn(null);

        $invoiceService = $this->getMockBuilder(CetustekInvoice::class)
            ->setMethods(array('arrayToXML'))
            ->getMock();

        $invoiceService
            ->expects($this->once())
            ->method('arrayToXML')
            ->with($array)
            ->willReturn($testXml);
        $this->setObjectAttribute($invoiceService, 'config', $config);
        $this->setObjectAttribute($invoiceService, 'soapClient', $soapClient);

        //act
        $result = $this->callObjectMethod($invoiceService, 'sendInvoice', $invoice);

        //assert
        $this->assertFalse($result);
        $this->assertEquals('null', $invoice->getApiResponse());
        $this->assertLessThan(5, time() - $invoice->getProcessAt('U'));
    }

    public function test_sendOne()
    {
        //arrange
        $con = $this->getMockBuilder(\PropelPDO::class)
            ->disableOriginalConstructor()
            ->getMock();
        $invoice = $this->getMockBuilder(Invoice::class)
            ->setMethods(array('save'))
            ->getMock();
        $invoice
            ->expects($this->once())
            ->method('save')
            ->with($con)
            ->willReturn(1);
        $invoice->setPaymentPaid(true);
        $invoiceService = $this->getMockBuilder(CetustekInvoice::class)
            ->setMethods(array('sendInvoice'))
            ->getMock();
        $invoiceService
            ->expects($this->once())
            ->method('sendInvoice')
            ->with($invoice, $con)
            ->willReturn(true);
        //act
        $result = $invoiceService->sendOne($invoice, $con);

        //assert
        $this->assertTrue($result);
    }

    public function test_sendOne_not_paid()
    {
        //arrange
        $con = $this->getMockBuilder(\PropelPDO::class)
            ->disableOriginalConstructor()
            ->getMock();
        $invoice = $this->getMockBuilder(Invoice::class)
            ->setMethods(array('save'))
            ->getMock();
        $invoice
            ->expects($this->never())
            ->method('save')
            ->with($con)
            ->willReturn(1);
        $invoice->setPaymentPaid(false);
        $invoiceService = $this->getMockBuilder(CetustekInvoice::class)
            ->setMethods(array('sendInvoice'))
            ->getMock();
        $invoiceService
            ->expects($this->never())
            ->method('sendInvoice')
            ->with($invoice, $con)
            ->willReturn(true);
        //act
        $result = $invoiceService->sendOne($invoice, $con);

        //assert
        $this->assertFalse($result);
    }

    public function test_sendOne_fail()
    {
        //arrange
        $con = $this->getMockBuilder(\PropelPDO::class)
            ->disableOriginalConstructor()
            ->getMock();
        $invoice = $this->getMockBuilder(Invoice::class)
            ->setMethods(array('save'))
            ->getMock();
        $invoice
            ->expects($this->once())
            ->method('save')
            ->with($con)
            ->willReturn(1);
        $invoice->setPaymentPaid(true);
        $invoiceService = $this->getMockBuilder(CetustekInvoice::class)
            ->setMethods(array('sendInvoice'))
            ->getMock();
        $invoiceService
            ->expects($this->once())
            ->method('sendInvoice')
            ->with($invoice, $con)
            ->willReturn(false);
        //act
        $result = $invoiceService->sendOne($invoice, $con);

        //assert
        $this->assertFalse($result);
    }

    public function test_createAllowance()
    {
        //arrange
        $orderId = 'P0003';
        $invoice = InvoiceQuery::create()->findPk($orderId);
        $reason = '帳務重開3';
        $con = \Propel::getConnection();
        $invoiceService = new CetustekInvoice();
        $con->beginTransaction();
        $invoice
            ->setProcessAt(time())
            ->setPaymentPaid(true)
            ->save();

        //act
        $result = $invoiceService->createAllowance($invoice, $reason, $con);
        $invoiceAllowance = InvoiceAllowanceQuery::create()->findPk($orderId, $con);
        $con->rollBack();

        //assert
        $this->assertInstanceOf(InvoiceAllowance::class, $result);
        $this->assertInstanceOf(InvoiceAllowance::class, $invoiceAllowance);
        $this->assertEquals($orderId, $invoiceAllowance->getId());
        $this->assertEquals($reason, $invoiceAllowance->getReason());
    }

    public function test_sendAllowance()
    {
        //arrange
        $con = $this->getMockBuilder(\PropelPDO::class)
            ->disableOriginalConstructor()
            ->getMock();
        $invoiceAllowance = $this->getMockBuilder(InvoiceAllowance::class)
            ->setMethods(array('save'))
            ->getMock();
        $invoiceAllowance
            ->expects($this->once())
            ->method('save')
            ->with($con)
            ->willReturn(1);
        $invoiceService = $this->getMockBuilder(CetustekInvoice::class)
            ->setMethods(array('sendInvoiceAllowance'))
            ->getMock();
        $invoiceService
            ->expects($this->once())
            ->method('sendInvoiceAllowance')
            ->with($invoiceAllowance, $con)
            ->willReturn(true);

        //act
        $result = $invoiceService->sendAllowance($invoiceAllowance, $con);

        //assert
        $this->assertTrue($result);
    }

    public function test_sendAllowance_not_paid()
    {
        //arrange
        $con = $this->getMockBuilder(\PropelPDO::class)
            ->disableOriginalConstructor()
            ->getMock();
        $invoiceAllowance = $this->getMockBuilder(InvoiceAllowance::class)
            ->setMethods(array('save'))
            ->getMock();
        $invoiceAllowance
            ->expects($this->never())
            ->method('save')
            ->with($con)
            ->willReturn(1);
        $invoiceAllowance->setIsProcess(true);
        $invoiceService = $this->getMockBuilder(CetustekInvoice::class)
            ->setMethods(array('sendInvoiceAllowance'))
            ->getMock();
        $invoiceService
            ->expects($this->never())
            ->method('sendInvoiceAllowance')
            ->with($invoiceAllowance, $con)
            ->willReturn(true);

        //act
        $result = $invoiceService->sendAllowance($invoiceAllowance, $con);

        //assert
        $this->assertFalse($result);
    }

    public function test_sendAllowance_fail()
    {
        //arrange
        $con = $this->getMockBuilder(\PropelPDO::class)
            ->disableOriginalConstructor()
            ->getMock();
        $invoiceAllowance = $this->getMockBuilder(InvoiceAllowance::class)
            ->setMethods(array('save'))
            ->getMock();
        $invoiceAllowance
            ->expects($this->once())
            ->method('save')
            ->with($con)
            ->willReturn(1);
        $invoiceService = $this->getMockBuilder(CetustekInvoice::class)
            ->setMethods(array('sendInvoiceAllowance'))
            ->getMock();
        $invoiceService
            ->expects($this->once())
            ->method('sendInvoiceAllowance')
            ->with($invoiceAllowance, $con)
            ->willReturn(false);

        //act
        $result = $invoiceService->sendAllowance($invoiceAllowance, $con);

        //assert
        $this->assertFalse($result);
    }

    public function test_sendInvoiceAllowance()
    {
        //arrange
        $config = array(
            'rentid' => '12853714',
            'invoice_type' => '03',
        );
        $testXml = 'testXml';
        $invoiceNumber = 'AA12345678';
        $apiResponse = json_decode(json_encode(array(
            'return' => 'A0',
        )));
        $orderId = 'A12345';
        $array = array(
            'AllowanceNumber' => $orderId,
            'InvoiceNumber' => $invoiceNumber,
            'InvoiceYear' => strftime('%Y', time()),
            'AllowanceDate' => strftime('%Y/%m/%d', time()),
            'TaxType' => 1,
            'Reason' => '帳務重開',
            'Details' => array(
                'ProductItem' => array(
                    array(
                        'ProductionCode' => 'A12345',
                        'Description' => '服務費',
                        'Quantity' => 1,
                        'UnitPrice' => 100,
                        'Unit' => '',
                    ),
                ),
            ),
        );
        $invoiceItem0 = new InvoiceItem();
        $invoiceItem0
            ->setProductId($array['Details']['ProductItem'][0]['ProductionCode'])
            ->setDescription($array['Details']['ProductItem'][0]['Description'])
            ->setQuantity($array['Details']['ProductItem'][0]['Quantity'])
            ->setUnitPrice($array['Details']['ProductItem'][0]['UnitPrice'])
        ;
        $invoice = new Invoice();
        $invoice
            ->setId($orderId)
            ->setInvoiceNumber($invoiceNumber)
            ->addInvoiceItem($invoiceItem0)
            ->setProcessAt(time())
            ->save()
        ;
        $invoiceAllowance = new InvoiceAllowance();
        $invoiceAllowance
            ->setId($invoice->getId())
            ->setReason($array['Reason'])
        ;

        $soapClient = $this->getMockBuilder(\stdClass::class)
            ->setMethods(array('CreateAllowance'))
            ->getMock();

        $soapClient
            ->expects($this->once())
            ->method('CreateAllowance')
            ->with(array(
                'allowancexml' => $testXml,
                'checkallowance' => 0,
                'rentid' => $config['rentid']
            ))
            ->willReturn($apiResponse);

        $invoiceService = $this->getMockBuilder(CetustekInvoice::class)
            ->setMethods(array('arrayToXML'))
            ->getMock();

        $invoiceService
            ->expects($this->once())
            ->method('arrayToXML')
            ->with($array)
            ->willReturn($testXml);
        $this->setObjectAttribute($invoiceService, 'config', $config);
        $this->setObjectAttribute($invoiceService, 'soapClient', $soapClient);

        //act
        $result = $this->callObjectMethod($invoiceService, 'sendInvoiceAllowance', $invoiceAllowance);
        $invoiceItem0->delete();
        $invoice->delete();

        //assert
        $this->assertTrue($result);
        $this->assertEquals(json_encode($apiResponse), $invoiceAllowance->getApiResponse());
        $this->assertLessThan(5, time() - $invoiceAllowance->getProcessAt('U'));
    }

    public function test_sendInvoiceAllowance_error()
    {
        //arrange
        $config = array(
            'rentid' => '12853714',
            'invoice_type' => '03',
        );
        $testXml = 'testXml';
        $invoiceNumber = 'AA12345678';
        $apiResponse = json_decode(json_encode(array(
            'return' => 'A0',
        )));
        $orderId = 'A12345';
        $array = array(
            'AllowanceNumber' => $orderId,
            'AllowanceDate' => strftime('%Y/%m/%d', time()),
            'InvoiceNumber' => $invoiceNumber,
            'InvoiceYear' => strftime('%Y', time()),
            'TaxType' => 1,
            'Reason' => '帳務重開',
            'Details' => array(
                'ProductItem' => array(
                    array(
                        'ProductionCode' => 'A12345',
                        'Description' => '服務費',
                        'Quantity' => 1,
                        'UnitPrice' => 100,
                        'Unit' => '',
                    ),
                ),
            ),
        );
        $invoiceItem0 = new InvoiceItem();
        $invoiceItem0
            ->setProductId($array['Details']['ProductItem'][0]['ProductionCode'])
            ->setDescription($array['Details']['ProductItem'][0]['Description'])
            ->setQuantity($array['Details']['ProductItem'][0]['Quantity'])
            ->setUnitPrice($array['Details']['ProductItem'][0]['UnitPrice'])
        ;
        $invoice = new Invoice();
        $invoice
            ->setId($orderId)
            ->setInvoiceNumber($invoiceNumber)
            ->addInvoiceItem($invoiceItem0)
            ->setProcessAt(time())
            ->save()
        ;
        $invoiceAllowance = new InvoiceAllowance();
        $invoiceAllowance
            ->setId($invoice->getId())
            ->setReason($array['Reason'])
        ;

        $soapClient = $this->getMockBuilder(\stdClass::class)
            ->setMethods(array('CreateAllowance'))
            ->getMock();

        $soapClient
            ->expects($this->once())
            ->method('CreateAllowance')
            ->with(array(
                'allowancexml' => $testXml,
                'checkallowance' => 0,
                'rentid' => $config['rentid']
            ))
            ->willReturn(null);

        $invoiceService = $this->getMockBuilder(CetustekInvoice::class)
            ->setMethods(array('arrayToXML'))
            ->getMock();

        $invoiceService
            ->expects($this->once())
            ->method('arrayToXML')
            ->with($array)
            ->willReturn($testXml);
        $this->setObjectAttribute($invoiceService, 'config', $config);
        $this->setObjectAttribute($invoiceService, 'soapClient', $soapClient);

        //act
        $result = $this->callObjectMethod($invoiceService, 'sendInvoiceAllowance', $invoiceAllowance);
        $invoiceItem0->delete();
        $invoice->delete();

        //assert
        $this->assertFalse($result);
        $this->assertEquals('null', $invoiceAllowance->getApiResponse());
        $this->assertLessThan(5, time() - $invoiceAllowance->getProcessAt('U'));
    }

}