<?php
namespace Widget\InvoiceBundle\Service;

use Backend\BaseBundle\Service\ConstraintViolationListConvertor;
use JMS\DiExtraBundle\Annotation as DI;
use Symfony\Component\Finder\Expression\Regex;
use Symfony\Component\Validator\Constraints as Assert;
use Symfony\Component\Validator\Context\ExecutionContextInterface;
use Symfony\Component\Validator\Validator\ValidatorInterface;

/**
 * @DI\Service("invoice_validator")
 */
class InvoiceValidator
{
    /** @var  ValidatorInterface */
    protected $validator;
    /** @var  ConstraintViolationListConvertor */
    protected $convertor;

    /**
     * @param ValidatorInterface $validator
     * @DI\InjectParams()
     */
    public function injectValidator(ValidatorInterface $validator)
    {
        $this->validator = $validator;
    }

    /**
     * @param ConstraintViolationListConvertor $convertor
     * @DI\InjectParams({
     *     "convertor" = @DI\Inject("api.validator.error.convertor")
     * })
     */
    public function injectValidatorErrorConvertor(ConstraintViolationListConvertor $convertor)
    {
        $this->convertor = $convertor;
    }

    public function validate($invoiceInfo)
    {
        $invoiceInfo = array_merge(array(
            'email' => null,
            'mark' => null,
            'loveCode' => null,
            'carrierType' => null,
            'carrierId' => null,
            'companyName' => null,
            'companyId' => null,
            'buyerName' => null,
            'buyerAddress' => null,
            'buyerAddressZip' => null,
        ), $invoiceInfo);
        $violations = $this->validator->validate($invoiceInfo, $this->getContraints());
        return $this->convertor->toArray($violations);
    }

    protected function getContraints()
    {
        return new Assert\Collection(array(
            'fields' => array(
                'email' => array(
                    new Assert\Email(),
                    new Assert\NotBlank(),
                ),
                'mark' => array(
                    new Assert\Choice(array(
                        'choices' => array('e', 'd', 'p'),
                    )),
                    new Assert\NotBlank(),
                    new Assert\Callback(function($value, ExecutionContextInterface $context){
                        if( $value != 'p' && trim($context->getRoot()['companyId']) != ''){
                            $context->addViolation('打統編發票開立型式需設為紙本電子發票');
                        }
                    }),
                ),
                'loveCode' => array(
                    new Assert\Regex(array(
                        'pattern' => '/^\d*$/i',
                    )),
                    new Assert\Callback(function($value, ExecutionContextInterface $context){
                        if($context->getRoot()['mark'] == 'd' && trim($value) == ''){
                            $context->addViolation('不可為空白');
                        }
                    }),
                ),
                'carrierType' => array(
                    new Assert\Choice(array(
                        'choices' => array('0', '1', '2'),
                    )),
                ),
                'carrierId' => array(
                    new Assert\Callback(function($value, ExecutionContextInterface $context){
                        $root = $context->getRoot();
                        if($root['carrierType'] == '0' && trim($value) != ''){
                            $context->addViolation('預設載具不能有載具代碼');
                            return;
                        }
                        if($root['carrierType'] == '1' && trim($value) == ''){
                            $context->addViolation('載具代碼不能為空');
                            return;
                        }
                        if($root['carrierType'] == '2' && trim($value) == ''){
                            $context->addViolation('載具代碼不能為空');
                            return;
                        }
                    }),
                ),
                'companyName' => null,
                'companyId' => array(),
                'buyerName' => array(
                    new Assert\Callback(function($value, ExecutionContextInterface $context){
                        if($context->getRoot()['mark'] == 'p' && trim($value) == ''){
                            $context->addViolation('不可為空白');
                        }
                    }),
                ),
                'buyerAddress' => array(
                    new Assert\Callback(function($value, ExecutionContextInterface $context){
                        if($context->getRoot()['mark'] == 'p' && trim($value) == ''){
                            $context->addViolation('不可為空白');
                        }
                    }),
                ),
                'buyerAddressZip' => array(
                    new Assert\Regex(array(
                        'pattern' => '/^\d*$/i',
                    )),
                    new Assert\Callback(function($value, ExecutionContextInterface $context){
                        if($context->getRoot()['mark'] == 'p' && trim($value) == ''){
                            $context->addViolation('不可為空白');
                        }
                    }),
                ),
            ),
        ));
    }
}