<?php
namespace Widget\InvoiceBundle\Service;

use GuzzleHttp\Client;
use function GuzzleHttp\Psr7\stream_for;
use GuzzleHttp\RequestOptions;
use JMS\DiExtraBundle\Annotation as DI;
use Widget\InvoiceBundle\Model\Invoice;
use Widget\InvoiceBundle\Model\InvoiceAllowance;
use Widget\InvoiceBundle\Model\InvoiceItem;
use Widget\InvoiceBundle\Model\InvoicePeer;

/**
 * @DI\Service("widget_invoice.invoice.ecpay")
 */
class ECPayInvoice extends AbstractInvoice
{
    const API_TYPE = "Ecpay";
    /** @var  Client */
    protected $client;

    /**
     * @DI\InjectParams({
     *     "config" = @DI\Inject("widget_invoice.config")
     * })
     */
    public function injectConfig($config)
    {
        $this->config = (isset($config[static::API_TYPE])?$config[static::API_TYPE]:array());
    }

    protected function makeTime()
    {
        return time();
    }

    public function sendOne(Invoice $invoice, \PropelPDO $con = null)
    {
        $invoiceItems = $invoice->getInvoiceItems($con);
        $dataSet = array(
            'TimeStamp' => $this->makeTime(),
            'MerchantID' => (string)$this->config['merchantId'],
            'RelateNumber' => (string)$invoice->getId(),
            'CustomerID' => $invoice->getId(),
            'CustomerIdentifier' => (string)$invoice->getIdentifier(),
            'CustomerName' => urlencode((string)$invoice->getName()),
            'CustomerAddr' => urlencode("{$invoice->getZip()}{$invoice->getAddress()}"),
            'CustomerEmail' => urlencode($invoice->getEmail()),
            'Print' => $this->convertPrint($invoice),
            'Donation' => $this->convertDonation($invoice),
            'LoveCode' => ($invoice->getLoveCode()?"X{$invoice->getLoveCode()}":''),
            'CarruerType' => $this->convertCarrierType($invoice),
            'CarruerNum' => $this->convertCarrierId($invoice),
            'TaxType' => '1',
            'SalesAmount' => (string)$invoice->getTotalPrice(),
            'InvoiceRemark' => urlencode($invoice->getRemark()),
            'ItemName' => $this->convertItemName($invoiceItems),
            'ItemCount' => $this->convertItemCount($invoiceItems),
            'ItemWord' => $this->convertItemWord($invoiceItems),
            'ItemPrice' => $this->convertItemPrice($invoiceItems),
            'ItemAmount' => $this->convertItemAmount($invoiceItems),
            'ItemRemark' => $this->convertItemName($invoiceItems),
            'ItemTaxType' => $this->convertItemTaxType($invoiceItems),
            'InvType' => '07',
        );
        $checkMac = $this->createCheckMac($this->filterInvoiceUnCheckArray($dataSet), $this->config['hashKey'], $this->config['hashIv']);
        $dataSet['CheckMacValue'] = $checkMac;
        $result = $this->doPost('/Invoice/Issue', $dataSet);
        $invoice
            ->setProcessAt($this->makeTime())
            ->setInvoiceNumber($result['InvoiceNumber']??null)
            ->setApiResponse($result)
            ->save($con);
        return $result['RtnCode'] == '1';
    }

    /**
     * @param InvoiceItem[] $invoiceItems
     */
    protected function convertItemTaxType($invoiceItems)
    {
        $taxTypes = array();
        foreach ($invoiceItems as $invoiceItem){
            $taxTypes[] = 1;
        }
        return implode('|', $taxTypes);
    }

    /**
     * @param InvoiceItem[] $invoiceItems
     */
    protected function convertItemAmount($invoiceItems)
    {
        $amounts = array();
        foreach ($invoiceItems as $invoiceItem){
            $amounts[] = $invoiceItem->getUnitPrice()*$invoiceItem->getQuantity();
        }
        return implode('|', $amounts);
    }

    /**
     * @param InvoiceItem[] $invoiceItems
     */
    protected function convertItemPrice($invoiceItems)
    {
        $prices = array();
        foreach ($invoiceItems as $invoiceItem){
            $prices[] = $invoiceItem->getUnitPrice();
        }
        return implode('|', $prices);
    }

    /**
     * @param InvoiceItem[] $invoiceItems
     */
    protected function convertItemWord($invoiceItems)
    {
        $words = array();
        foreach ($invoiceItems as $invoiceItem){
            $words[] = $invoiceItem->getUnit();
        }
        return urlencode(implode('|', $words));
    }

    /**
     * @param InvoiceItem[] $invoiceItems
     */
    protected function convertItemName($invoiceItems)
    {
        $names = array();
        foreach ($invoiceItems as $invoiceItem){
            $names[] = $invoiceItem->getDescription();
        }
        return urlencode(implode('|', $names));
    }

    /**
     * @param InvoiceItem[] $invoiceItems
     */
    protected function convertItemCount($invoiceItems)
    {
        $counts = array();
        foreach ($invoiceItems as $invoiceItem){
            $counts[] = $invoiceItem->getQuantity();
        }
        return implode('|', $counts);
    }

    /**
     * 3:手機條碼則, 2:自然人憑證
     * @param Invoice $invoice
     */
    protected function convertCarrierType(Invoice $invoice)
    {
        //invoice 手機條碼則:3J0002, 自然人憑證:CQ0001
        if($invoice->getDonateMark() == InvoicePeer::DONATE_MARK_PAPER){
            return '';
        }

        if($invoice->getDonateMark() != InvoicePeer::DONATE_MARK_DONATE){
            return '';
        }

        if($invoice->getIdentifier() != ''){
            return '';
        }

        if($invoice->getCarrierType() == '3J0002'){
            return 3;
        }

        if($invoice->getCarrierType() == 'CQ0001'){
            return 2;
        }

        return 1;
    }

    /**
     * 1: 表示捐贈, 0: 表示不捐贈
     * @param Invoice $invoice
     */
    protected function convertDonation(Invoice $invoice)
    {
        if($invoice->getIdentifier() != ''){
            return '0';
        }

        if($invoice->getDonateMark() != InvoicePeer::DONATE_MARK_DONATE){
            return '0';
        }

        return '1';
    }

    /**
     * 1: 表示列印, 0: 表示不列印
     * @param Invoice $invoice
     * @return int
     */
    protected function convertPrint(Invoice $invoice)
    {
        if($invoice->getIdentifier() != ''){
            return '1';
        }

        if($invoice->getDonateMark() == InvoicePeer::DONATE_MARK_PAPER){
            return '1';
        }

        return '0';
    }

    public function sendAllowance(InvoiceAllowance $invoiceAllowance, \PropelPDO $con = null)
    {
        $invoice = $invoiceAllowance->getInvoice($con);
        $invoiceItems = $invoice->getInvoiceItems($con);
        $dataSet = array(
            'TimeStamp' => (string)$this->makeTime(),
            'MerchantID' => (string)$this->config['merchantId'],
            'InvoiceNo' => (string)$invoice->getInvoiceNumber(),
            'AllowanceNotify' => 'E',
            'CustomerName' => (string)$invoice->getName(),
            'NotifyMail' => urlencode($invoice->getEmail()),
            'AllowanceAmount' => (string)$invoice->getTotalPrice(),
            'ItemName' => $this->convertItemName($invoiceItems),
            'ItemCount' => $this->convertItemCount($invoiceItems),
            'ItemWord' => $this->convertItemWord($invoiceItems),
            'ItemPrice' => $this->convertItemPrice($invoiceItems),
            'ItemAmount' => $this->convertItemAmount($invoiceItems),
            'ItemTaxType' => $this->convertItemTaxType($invoiceItems),
        );
        $checkMac = $this->createCheckMac($this->filterAllowanceUnCheckArray($dataSet), $this->config['hashKey'], $this->config['hashIv']);
        $dataSet['CheckMacValue'] = $checkMac;
        $result = $this->doPost('/Invoice/Allowance', $dataSet);
        $invoiceAllowance
            ->setProcessAt($result['IA_Date']??time())
            ->setApiResponse($result)
            ->setIsProcess(true)
            ->save($con);
        return $result['RtnCode'] == '1';
    }

    protected function createCheckMac($originArray, $hashKey, $hashIv)
    {
        $string = '';
        ksort($originArray);
        foreach ($originArray as $key => $item) {
            $string.="&$key=$item";
        }
        $string = "HashKey=$hashKey$string&HashIV=$hashIv";
        $string = strtolower($this->uglyUrlencode($string));
        return strtoupper(md5($string));
    }

    protected function uglyUrlencode($string)
    {
        $convertMap = array(
            '%2d' => '-',
            '%5f' => '_',
            '%2e' => '.',
            '%21' => '!',
            '%2a' => '*',
            '%28' => '(',
            '%29' => ')'
        );
        $original = urlencode($string);

        $modified = preg_replace_callback('/%[0-9A-F]{2}/', function($matches) {
            return strtolower($matches[0]);
        }, $original);
        return str_replace(array_keys($convertMap), array_values($convertMap), $modified);
    }

    protected function getClient()
    {
        if(!($this->client)){
            $this->client = new Client(array('base_uri' => $this->config['apiBase']));
        }
        return $this->client;
    }

    protected function doPost($uri, $dataSet)
    {
        $client = $this->getClient();
        $response = $client->request('POST', $uri, array(
            RequestOptions::BODY => stream_for($this->makePostBody($dataSet)),
            RequestOptions::HEADERS => array(
                'Content-Type' => 'application/x-www-form-urlencoded',
            ),
        ));
        parse_str($response->getBody(), $result);
        return $result;
    }

    protected function makePostBody($dataSet)
    {
        $result = array();
        foreach ($dataSet as $key => $value){
            $result[]="$key=$value";
        }
        return implode('&', $result);
    }

    protected function filterInvoiceUnCheckArray($dataSet)
    {
        unset($dataSet['InvoiceRemark']);
        unset($dataSet['ItemName']);
        unset($dataSet['ItemWord']);
        unset($dataSet['ItemRemark']);
        return $dataSet;
    }

    protected function filterAllowanceUnCheckArray($dataSet)
    {
        unset($dataSet['ItemName']);
        unset($dataSet['ItemWord']);
        return $dataSet;
    }

    /**
     * @param $allowKeys
     * @param $dataSet
     * @return array
     */
    protected function filterArray($allowKeys, $dataSet)
    {
        $filterdArray = array();
        foreach ($allowKeys as $key) {
            if (isset($dataSet[$key])) {
                $filterdArray[$key] = $dataSet[$key];
            }
        }

        return $filterdArray;
    }

    protected function convertCarrierId(Invoice $invoice)
    {
        $carrierType = $this->convertCarrierType($invoice);

        if($carrierType == ''){
            return '';
        }
        if($carrierType == '1'){
            return '';
        }

        return $invoice->getCarrierId();
    }

}