<?php
namespace Widget\InvoiceBundle\Controller\BackendAPI;

use Backend\BaseBundle\Controller\BackendAPI\BaseBackendAPIController;
use Backend\BaseBundle\Form\Type\APIFormTypeItem;
use Sensio\Bundle\FrameworkExtraBundle\Configuration\Method;
use Sensio\Bundle\FrameworkExtraBundle\Configuration\Security;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\Routing\Annotation\Route;
use Symfony\Component\Validator\Constraints\Email;
use Symfony\Component\Validator\Constraints\NotBlank;
use Widget\InvoiceBundle\Model\Invoice;
use Widget\InvoiceBundle\Model\InvoicePeer;
use Widget\InvoiceBundle\Model\InvoiceQuery;

/**
 * @Route("/invoice")
 * @Security("has_role_or_superadmin('ROLE_INVOICE_READ')")
 */
class InvoiceController extends BaseBackendAPIController
{
    /**
     * @return APIFormTypeItem[]
     */
    protected function getFormConfig()
    {
        return array(
            new APIFormTypeItem('identifier'),
            (new APIFormTypeItem('name'))->setOptions(array(
                'constraints' => array(
                    new NotBlank(array(
                        'message' => 'error.missing_field',
                    )),
                )
            )),
            new APIFormTypeItem('zip'),
            new APIFormTypeItem('address'),
            (new APIFormTypeItem('email'))->setOptions(array(
                'constraints' => array(
                    new NotBlank(array(
                        'message' => 'error.missing_field',
                    )),
                    new Email(array()),
                )
            )),
        );
    }

    /**
     * 搜尋
     * @Route("s")
     * @Method({"GET"})
     * @Security("has_role_or_superadmin('ROLE_INVOICE_READ')")
     */
    public function searchAction(Request $request)
    {
        $query = InvoiceQuery::create();
        $query->filterByPaymentPaid(true);
        return $this->doSearch($request->query->all(), $query, InvoicePeer::class);
    }

    /**
     * 讀取
     * @Route("/{id}")
     * @Method({"GET"})
     * @Security("has_role_or_superadmin('ROLE_INVOICE_READ')")
     */
    public function readAction(Request $request, Invoice $invoice)
    {
        return $this->createJsonSerializeResponse($invoice, array('detail'));
    }

    /**
     * @Route("s/reset_process")
     * @Method({"PUT"})
     * @Security("has_role_or_superadmin('ROLE_INVOICE_WRITE')")
     */
    public function resetProcessAction(Request $request)
    {
        $ids = json_decode($request->getContent(), true);

        if(!is_array($ids)){
            return $this->createFormErrorJsonResponse(Response::HTTP_BAD_REQUEST);
        }

        foreach($ids as $id){
            $invoice = InvoiceQuery::create()->findPk($id);
            if(!$invoice || !empty($invoice->getInvoiceNumber())){
                continue;
            }
            $invoice->setProcessAt(null)->save();
        }

        return $this->createJsonResponse();
    }

    /**
     * 更新
     * @Route("/{id}")
     * @Method({"PUT"})
     * @Security("has_role_or_superadmin('ROLE_INVOICE_WRITE')")
     */
    public function updateAction(Request $request, Invoice $invoice)
    {
        return $this->doProcessForm($invoice, $request->getContent());
    }
}
