<?php

namespace Widget\DiscountBundle\Tests\Service;

use Backend\BaseBundle\Tests\Fixture\BaseTestCase;
use Backend\BaseBundle\Tests\Fixture\BaseWebTestCase;
use PropelPDO;
use Symfony\Component\EventDispatcher\EventDispatcher;
use Widget\Discount\TargetAmountBundle\Discount\TargetAmountDiscount;
use Widget\DiscountBundle\Event\DiscountFactoryEvent;
use Widget\DiscountBundle\Model\DiscountGroup;
use Widget\DiscountBundle\Model\DiscountModule;
use Widget\DiscountBundle\Model\DiscountModuleQuery;
use Widget\DiscountBundle\Service\Discount\DiscountInterface;
use Widget\DiscountBundle\Service\DiscountBuilder;
use Widget\MemberBundle\Model\Group;
use Widget\MemberBundle\Model\Member;
use Widget\MemberBundle\Model\MemberQuery;
use Widget\OrderBundle\Model\Order;
use Widget\OrderBundle\Model\OrderItem;

class DiscountBuilderTest extends BaseWebTestCase
{
    public function init()
    {
        // arrange
        $order = new Order();
        $order->setMemberId('1000');
        $builder = $this->getMockBuilder(DiscountBuilder::class)
            ->disableOriginalConstructor()
            ->setMethods(array('prepareDiscountsRule'))
            ->getMock();
        $member = MemberQuery::create()->findPk(1000);
        $builder
            ->expects($this->once())
            ->method('prepareDiscountsRule')
            ->willReturnCallback(function($memberForTest) use($member){
                $this->assertEquals($member, $memberForTest);
            });

        // act
        $builder->init($order);
    }
    
    public function test_prepareDiscountsRule()
    {
        // arrange
        $order = new Order();
        $orderItem = new OrderItem();
        $orderItem->setProductName('test');
        $orderItem->setUnitPrice(10000);
        $orderItem->setQuantity(1);
        $order->addProducts($orderItem);

        $member = MemberQuery::create()->findPk(1001);
        $group = $member->getGroups()->getFirst();

        $builder = $this->getMockBuilder(DiscountBuilder::class)
            ->setMethods(array('prepareExcludeDiscounts', 'prepareDiscounts'))
            ->disableOriginalConstructor()
            ->getMock();

        $this->setObjectAttribute($builder, 'order', $order);
        // act
        $builder->prepareDiscountsRule($member);

        // assert

    }

    public function test_prepareDiscounts()
    {
        // arrange
        $group = new Group();

        $targetAmount = $this->getMockBuilder(TargetAmountDiscount::class)
            ->disableOriginalConstructor()
            ->getMock();
        $builder = $this->getMockBuilder(DiscountBuilder::class)
            ->setMethods(array('groupToDiscount', 'factory'))
            ->disableOriginalConstructor()
            ->getMock();
        $builder
            ->expects($this->once())
            ->method('groupToDiscount')
            ->willReturnCallback(function(){
                return array(
                    DiscountModuleQuery::create()->findPk(1),
                );
            });
        $builder
            ->expects($this->once())
            ->method('factory')
            ->willReturn($targetAmount);

        // act

        $this->callObjectMethod($builder, 'prepareDiscounts', $group);

        // assert
        $discounts = $this->getObjectAttribute($builder, 'discounts');
        $this->assertInstanceOf(DiscountInterface::class, $discounts[0]);
    }

    public function test_groupToDiscount()
    {

        // arrange
        $member = MemberQuery::create()->findPk(1001);
        // 請參考 fixtures 的順序
        $expectReturnModules = array(
            DiscountModuleQuery::create()->findPk(2),
            DiscountModuleQuery::create()->findPk(1),
        );
        $builder = $this->getMockBuilder(DiscountBuilder::class)
            ->setMethods(null)
            ->disableOriginalConstructor()
            ->getMock();

        // act
        $discountModules = $this->callObjectMethod($builder, 'groupToDiscount', $member->getGroups());

        // assert
        $this->assertEquals($expectReturnModules, $discountModules);
    }

    public function test_groupToDiscount_no_discount()
    {
        // arrange
        $member = MemberQuery::create()->findPk(1009);
        $expectReturnArray = array();
        $builder = $this->getMockBuilder(DiscountBuilder::class)
            ->setMethods(null)
            ->disableOriginalConstructor()
            ->getMock();

        // act
        $discountModules = $this->callObjectMethod($builder, 'groupToDiscount', $member->getGroups());

        // assert
        $this->assertEquals($expectReturnArray, $discountModules);
    }

    public function test_factory_ok()
    {
        // arrange
        $builder = $this->getMockBuilder(DiscountBuilder::class)
            ->setMethods(null)
            ->disableOriginalConstructor()
            ->getMock();

        $eventDispatcher = $this->getMockBuilder(EventDispatcher::class)
            ->disableOriginalConstructor()
            ->setMethods(array('dispatch'))
            ->getMock();
        $targetAmount = $this->getMockBuilder(TargetAmountDiscount::class)
            ->disableOriginalConstructor()
            ->setMethods(null)
            ->getMock();

        $eventDispatcher
            ->expects($this->once())
            ->method('dispatch')
            ->willReturnCallback(function($eventName, DiscountFactoryEvent $event) use($targetAmount){
                $this->assertEquals(DiscountFactoryEvent::EVENT_NAME, $eventName);
                $this->assertInstanceOf(DiscountFactoryEvent::class, $event);
                return $event->setObject($targetAmount);
            });

        $this->setObjectAttribute($builder, 'eventDispatcher', $eventDispatcher);
        $discountModule = DiscountModuleQuery::create()->findPk(1);

        // act
        $discountClass = $builder->factory($discountModule); // 會抓到一個折扣資料
        // assert
        $this->assertInstanceOf(DiscountInterface::class, $discountClass);
    }

    public function test_factory_no_discount_module()
    {
        // arrange
        $builder = $this->getMockBuilder(DiscountBuilder::class)
            ->setMethods(null)
            ->disableOriginalConstructor()
            ->getMock();

        $eventDispatcher = $this->getMockBuilder(EventDispatcher::class)
            ->disableOriginalConstructor()
            ->setMethods(array('dispatch'))
            ->getMock();

        $eventDispatcher
            ->expects($this->once())
            ->method('dispatch')
            ->willReturnCallback(function($eventName, DiscountFactoryEvent $event){
                $this->assertEquals(DiscountFactoryEvent::EVENT_NAME, $eventName);
                $this->assertInstanceOf(DiscountFactoryEvent::class, $event);
                return $event;
            });
        $this->setObjectAttribute($builder, 'eventDispatcher', $eventDispatcher);
        $discountModule = new DiscountModule();
        // act
        $discountModule = $builder->factory($discountModule);

        // assert
        $this->assertNull($discountModule);
    }

    /**
     * 執行優惠折扣計算
     */
    public function test_run()
    {
        // arrange
        $runExcludeDiscountsTrigger = false;
        $runDiscountsTrigger = false;
        $discountBuilder = $this->getMockBuilder(DiscountBuilder::class)
            ->disableOriginalConstructor()
            ->setMethods(array('runExcludeDiscounts', 'runDiscounts'))
            ->getMock();

        $order = new Order();

        $discountBuilder
            ->expects($this->once())
            ->method('runExcludeDiscounts')
            ->willReturnCallback(function($orderForTest) use($order, &$runExcludeDiscountsTrigger){
                $this->assertEquals($order, $orderForTest);
                $runExcludeDiscountsTrigger = true;
            });
        $discountBuilder
            ->expects($this->once())
            ->method('runDiscounts')
            ->willReturnCallback(function($orderForTest) use($order, &$runDiscountsTrigger){
                $this->assertEquals($order, $orderForTest);
                $runDiscountsTrigger = true;
            });

        // act
        $discountBuilder->run($order);

        // assert
        $this->assertTrue($runExcludeDiscountsTrigger);
        $this->assertTrue($runDiscountsTrigger);
    }

    /**
     * 計算互斥優惠 優惠都不成立
     */
    public function test_runExcludeDiscounts()
    {
        // arrange
        $discountBuilder = $this->getMockBuilder(DiscountBuilder::class)
            ->disableOriginalConstructor()
            ->setMethods(null)
            ->getMock();
        $order = new Order();
        $excludeDiscounts = array(
            $this->mockDiscountForExclude($order, false, $this->once()),
            $this->mockDiscountForExclude($order, false, $this->once()),
            $this->mockDiscountForExclude($order, false, $this->once()),
        );

        $this->setObjectAttribute($discountBuilder, 'excludeDiscounts', $excludeDiscounts);
        // act
        $this->callObjectMethod($discountBuilder, 'runExcludeDiscounts', $order);

        // assert
    }

    /**
     * 計算互斥優惠 有優惠成立
     */
    public function test_runExcludeDiscounts_processed()
    {
        // arrange
        $discountBuilder = $this->getMockBuilder(DiscountBuilder::class)
            ->disableOriginalConstructor()
            ->setMethods(null)
            ->getMock();
        $order = new Order();
        $excludeDiscounts = array(
            $this->mockDiscountForExclude($order, false, $this->once()),
            $this->mockDiscountForExclude($order, true, $this->once()),
            $this->mockDiscountForExclude($order, false, $this->never()),
        );

        $this->setObjectAttribute($discountBuilder, 'excludeDiscounts', $excludeDiscounts);
        // act
        $this->callObjectMethod($discountBuilder, 'runExcludeDiscounts', $order);

        // assert
    }

    /**
     * 計算非互斥優惠
     */
    public function test_runDiscounts()
    {
        // arrange
        $discountBuilder = $this->getMockBuilder(DiscountBuilder::class)
            ->disableOriginalConstructor()
            ->setMethods(null)
            ->getMock();
        $order = new Order();
        $discounts = array(
            $this->mockDiscount($order, false),
            $this->mockDiscount($order, true),
            $this->mockDiscount($order, true),
        );

        $this->setObjectAttribute($discountBuilder, 'discounts', $discounts);
        // act
        $this->callObjectMethod($discountBuilder, 'runDiscounts', $order);

        // assert
    }

    /**
     * mock 出互斥專用的 discount interface
     * @param Order $order
     * @param $isProcessed
     * @param $expectRun
     * @return \PHPUnit_Framework_MockObject_MockObject
     */
    protected function mockDiscountForExclude(Order $order, $isProcessed, $expectRun)
    {
        $mockDiscount = $this->getMockBuilder(MockDiscount::class)
            ->setMethods(array('process', 'isProcessed'))
            ->getMock();

        $mockDiscount
            ->expects($expectRun)
            ->method('process')
            ->willReturnCallback(function($orderForTest) use($order){
                $this->assertEquals($order, $orderForTest);
            });
        $mockDiscount
            ->expects(clone $expectRun)
            ->method('isProcessed')
            ->willReturn($isProcessed);
        return $mockDiscount;
    }

    /**
     * mock 出非互斥專用的 discount interface
     * @param Order $order
     * @param $isProcessed
     * @return \PHPUnit_Framework_MockObject_MockObject
     */
    protected function mockDiscount(Order $order, $isProcessed)
    {
        $mockDiscount = $this->getMockBuilder(MockDiscount::class)
            ->setMethods(array('process', 'isProcessed'))
            ->getMock();

        $mockDiscount
            ->expects($this->once())
            ->method('process')
            ->willReturnCallback(function($orderForTest) use($order){
                $this->assertEquals($order, $orderForTest);
            });
        $mockDiscount
            ->expects($this->never())
            ->method('isProcessed')
            ->willReturn($isProcessed);
        return $mockDiscount;
    }

}

class MockDiscount implements DiscountInterface
{
    public function process(Order $order, \PropelPDO $con = null)
    {
    }

    public function isProcessed()
    {
    }

    public function injectSiteId($siteId)
    {
    }
}