<?php

namespace Widget\DiscountBundle\Service;

use JMS\DiExtraBundle\Annotation as DI;
use Symfony\Component\EventDispatcher\EventDispatcherInterface;
use Widget\Discount\TargetAmountBundle\Discount\TargetAmountDiscount;
use Widget\DiscountBundle\Event\DiscountFactoryEvent;
use Widget\DiscountBundle\Model\DiscountModule;
use Widget\DiscountBundle\Model\DiscountModuleQuery;
use Widget\DiscountBundle\Model\DiscountToGroup;
use Widget\DiscountBundle\Model\DiscountToGroupQuery;
use Widget\DiscountBundle\Service\Discount\DiscountInterface;
use Widget\MemberBundle\Model\Group;
use Widget\MemberBundle\Model\GroupQuery;
use Widget\MemberBundle\Model\Member;
use Widget\MemberBundle\Model\MemberQuery;
use Widget\OrderBundle\Model\Order;

/**
 * @DI\Service("widget.discount_bundle.discount_builder")
 */
class DiscountBuilder
{

    /** @var DiscountInterface[] */
    protected $discounts = array();

    /** @var DiscountInterface[]  */
    protected $excludeDiscounts = array();

    /** @var  Order $order */
    protected $order;

    protected $conn;


    /** @var  EventDispatcherInterface */
    protected $eventDispatcher;
    
    protected $parameters;

    /**
     * @DI\InjectParams()
     */
    public function injectEventDispatcher(EventDispatcherInterface $eventDispatcher)
    {
        $this->eventDispatcher = $eventDispatcher;
    }

    public function init(Order $order, \PropelPDO $conn = null, $parameters = array())
    {
        $this->order = $order;
        $this->conn = $conn;
        $this->parameters = $parameters;
        if ($member = MemberQuery::create()->findPk($order->getMemberId(), $this->conn)){
            $this->prepareDiscountsRule($member);
        }        
    }

    public function prepareDiscountsRule(Member $member)
    {

        $tempOrder = array();
        $tempDiscounts = array();
        $tempExcludeDiscounts = array();
        foreach ($member->getGroups() as $group){
            // 試算訂單折扣
            $this->prepareDiscounts($group);
            $this->prepareExcludeDiscounts($group);
            $this->run($this->order);

            // 將訂單金額及折扣暫存住
            $tempDiscounts[$group->getId()] = $this->discounts;
            $tempExcludeDiscounts[$group->getId()] = $this->excludeDiscounts;
            $tempOrder[$group->getId()] = $this->order->getOrderAmountForPay();

            // 清空試算用變數
            $this->discounts = array();
            $this->excludeDiscounts = array();
            $this->order->clearDiscount();
        }

        // 取訂單最低價格
        $groupId = $this->lowerPriceOrder($tempOrder);
        $this->discounts = $tempDiscounts[$groupId];
        $this->excludeDiscounts = $tempExcludeDiscounts[$groupId];

        //$this->prepareDiscounts($member);
        //$this->prepareExcludeDiscounts($member);

    }

    /**
     * 取得訂單應付金額最小的訂單
     * @param array $orderArray
     * @return mixed
     */
    protected function lowerPriceOrder(array $orderArray)
    {
        asort($orderArray);
        reset($orderArray);
        return key($orderArray);
    }
    
    protected function prepareDiscounts(Group $group)
    {
        $discountModules = $this->groupToDiscount($group);
        /** @var DiscountModule $module */
        foreach ($discountModules as $module) {
            $module = $this->factory($module);
            if ( $module != null ) {
                $this->discounts[] = $module;
            }
        }
    }
    
    protected function prepareExcludeDiscounts(Group $group)
    {
        $discountModules = $this->groupToDiscount($group, true);
        /** @var DiscountModule $module */
        foreach ($discountModules as $module) {
            $module = $this->factory($module);
            if ( $module != null ) {
                $this->excludeDiscounts[] = $module;
            }
        }
    }

    /**
     * 從會員群組撈出所有的優惠條件
     * @param $memberGroups
     * @param $exclude
     * @return DiscountModule[]
     */
    protected function groupToDiscount($memberGroups, $exclude = false)
    {
        $discountModules = array();
        $discountModuleQuery = DiscountModuleQuery::create()
            ->useDiscountToGroupQuery()
                ->filterByExclude($exclude)
                ->useDiscountGroupQuery()
                    ->useDiscountGroupToGroupQuery()
                        ->filterByGroup($memberGroups)
                    ->endUse()
                ->endUse()
            ->endUse()
            ->orderBySort(\Criteria::ASC)
            ->orderById(\Criteria::ASC)
            ->distinct()
            ->find($this->conn);

        /** @var DiscountModule $discountModule */
        foreach ($discountModuleQuery as $discountModule) {
            $discountModules[$discountModule->getId()] = $discountModule;
        }

        return array_values($discountModules);
    }
    
    /**
     * @param $module
     * @return mixed
     */
    public function factory(DiscountModule $module)
    {
        $event = new DiscountFactoryEvent($module->toArray(\BasePeer::TYPE_FIELDNAME), $this->parameters);
        $this->eventDispatcher->dispatch(DiscountFactoryEvent::EVENT_NAME, $event);

        if ( $event->getObject() InstanceOf DiscountInterface ) {
            return $event->getObject();
        }
        return null;
    }
    
    /**
     * 加入優惠折扣（非互斥關係）
     * @param DiscountInterface $discount
     * @return $this
     */
    public function add(DiscountInterface $discount)
    {
        $this->discounts[] = $discount;
        return $this;
    }
    
    /**
     * 加入優惠折扣（互斥關係）
     * @param DiscountInterface $discount
     * @return $this
     */
    public function addExclude(DiscountInterface $discount)
    {
        $this->excludeDiscounts[] = $discount;
        return $this;
    }

    /**
     * 執行優惠折扣計算
     * @param Order $order
     * @param \PropelPDO|null $con
     */
    public function run(Order $order, \PropelPDO $con = null)
    {
        $this->runExcludeDiscounts($order, $con);
        $this->runDiscounts($order, $con);
    }

    /**
     * 計算有互斥的優惠折扣
     * @param Order $order
     * @param \PropelPDO|null $con
     */
    protected function runExcludeDiscounts(Order $order, \PropelPDO $con = null)
    {
        foreach ($this->excludeDiscounts as $discount) {
            $discount->process($order, $con);
            if ($discount->isProcessed()) {
                break;
            }
        }
    }

    /**
     * 計算非互斥的優惠折扣
     * @param Order $order
     * @param \PropelPDO|null $con
     */
    protected function runDiscounts(Order $order, \PropelPDO $con = null)
    {
        foreach ($this->discounts as $discount) {
            $discount->process($order, $con);
        }
    }
}